/*************************************
*         Copyright 2015 EDF         *
*************************************/
#pragma once
/**\file
\~french Fichier de dfinition de la classe CAnalyser qui permet d'effectuer des analyses  partir des rsultats d'un calcul de simulation.
\~english Definition file of the CAnalyser class which allows to perform analyses from the results of a simulation calculation.
*/

class CSystem;
class IMonitored;
class CTransition;
class CTrClasses;
class CSeqClasses;

#include "Interfaces.h"
#include <vector>
#include <string>

/**\~french Classe de base d'analyse des rsultats d'une simulation Pycatshoo.
Cette classe propose des fonctions de base permettant l'analyse des rsultats, mais elle peut tre spcialise pour en tendre les possibilits.
\~english Base class for analyzing the results of a Pycatshoo simulation.
This class provides basic functions for analyzing results, but it can be specialized to extend its possibilities.
exemple1 (restitution  d'information sur les rsultats) :
~~~~{.cpp}
//Filter function
bool filtre1(ISequence const&seq){
	if(seq.value(CSystem::glSystem()->getMonitoredElt("cmp1.def","ST"),100)>0
		&& seq.endTime()<80)
		return true;
	return false;
}

void example(){
	//Build the system
	CSystem sys("Test");
	//Load result file
	sys.load("FichierRes.xml");
	//Build analyser
	CAnalyser ana(&sys);
	//Print all grouped sequences
	ana.printFilteredSeq("FichierDeSequences.xml","PySeq.xsl");
	//Add a sequence filter
	ana.setSeqFilter(filtre1);
	//Print filtered sequences
	ana.printFilteredSeq("FichierDeSequencesFiltres.xml","PySeq.xsl");
	//Calculate mean and stadard deviation of a variable at several instants
	std::vector<PyTime>vt;
	for(unsigned i=0;i<10;i++)
		vt.push_back(10*i);
	std::vector<float>means,stdDevs;
	ana.meanValues("comp2.port1",vt,means,stdDevs);
}
~~~~
exemple2 (construction d'un paramtrage pour la prolongation de squences):
~~~~{.cpp}
//Filter Functorstoring the monitored object
class CFilter :public ISequence::IFilterFct{
	IMonitored const&m_Obj;
	//Condition function used to calculate the residence time
	static bool condition(float val){return val>0;}
public:
	//Filter constructor
	CFilter(IMonitored const&obj):m_Obj(obj){}
	//Override the operator to define the filter
	bool operator()(ISequence const&seq)const override{
		return seq.residenceTime(&m_Obj,condition)>50;
	}
};

void exemple(){
	//Build the system
	CSystem sys("Test");
	//Load the knowledge base
	sys.loadBdCLibrary("BdC.dll");
	//load the result file
	sys.load("FichierRes.xml");
	//Build the analyser
	CAnalyser ana(&sys);
	//Create the filter
	CFilter filtre(*ana.monitoredElt("comp2.def","ST"));
	//Set the analyser sequence filter
	ana.setSeqFilter(&filtre);
	//Mark the filtered sequences
	ana.keepFilteredSeq(true);
	//Generate the parameter file for the extension of the filtered sequences
	sys.dumpParam("FichierPar.xml",true);
}
~~~~
*/
class PYC_PUBLIC CAnalyser
{
private:
	PyObject*m_Self;//!<\~french Rfrence  l'objet python correspondant.\~english 
	CSystem&m_System;
	bool m_LocalSystem;
	IFilterFct*m_SeqFilter;//!<\~french Filtre de squences\~english 
	CTrClasses&m_TrClasses;//!<\~french Regroupement des transitions dans des classes\~english 
	CSeqClasses&m_SeqClasses;//!<\~french Regroupement des squences dans des classes\~english 
	static float m_ConditionLimit;//!<\~french Limite utilise pour le calcul des fractiles\~english 
	static bool condition(float value){return value>m_ConditionLimit;}
public:
	/**\~french Construit un analyseur de rsultats pour le systme system.
	Le systme peut tre vide lors de la cration de l'analyseur.
	@param system systme dont les rsultats vont tre analyss
	\english Constructs a results parser for the system system.
	The system can be empty when creating the parser.
	@param system the system whose results will be parsed
	*/
	explicit CAnalyser(CSystem*system=0);
	CAnalyser(PyObject*self,CSystem&system);
	~CAnalyser(void);
	/**\~french Retourne l'objet python associ\~english Return the associated python object*/
	PyObject*self(){return m_Self;}
	/**\~french Retourne le nom complet des diffrents lments monitors.
	@param type type des lments monitors recherchs parmi (VAR : variable, ST : tat, TR : transition, IND : indicateur, NULL : tous)
	\~english Returns the full name of the different monitored elements.
	@param type type of the monitored elements searched among (VAR: variable, ST: state, TR: transition, IND: indicator, NULL: all)
	*/
	std::vector<IMonitored const*>monitoredElts(char const*type=NULL)const;
	/**\~french Retourne l'lment monitor de nom \p name.
	@param name nom de l'lment cherch
	@param type type de l'lment recherch parmi (VAR : variable, ST : tat, TR : transition, IND : indicateur, NULL : indiffrent)
	\~english Returns the monitored element of name \p name.
	@param name name of the element searched
	@param type type of the element searched among (VAR: variable, ST: state, TR: transition, IND: indicator, NULL: indifferent)
	*/
	IMonitored const*monitoredElt(char const*name,char const*type=NULL)const;
	/**\~french Retourne le nom des diffrents indicateurs.
	\~english Return the name of the different indicators.
	*/
	std::vector<IIndicator*>indicators()const;
	/**\~french Retourne l'indicateur de nom name.
	@param name nom de l'lment cherch
	\~english Return the indicator of name \p name.
	@param name name of the searche item
	*/
	IIndicator const*indicator(char const*name)const;
	std::vector<double>instants()const;//!<\~french Retourne la liste des instants de calcul des indicateurs\~english Returns the list of indicator calculation times
	/**\~french Renseigne la fonction de filtrage des squences sous la forme d'un foncteur boolen d'une squence\~english Populates the sequence filter function as a Boolean functor of a sequence*/
	void setSeqFilter(IFilterFct*seqFilter);
	/**\~french Renseigne la fonction de filtrage des squences sous la forme d'une fonction boolenne d'une squence.\~english Populates the sequence filter function as a Boolean function of a sequence.*/
	void setSeqFilter(bool (*seqFilter)(ISequence const&));
	/**\~french Renseigne la fonction de filtrage des squences sous la forme d'un pattern de dfinition d'un filtre de squence (cf. ISeqFilter)\~english Specifies the sequence filtering function in the form of a sequence filter definition pattern (see ISeqFilter)*/
	void setSeqFilter(char const*filterPattern);
	/**\~french Retourne la liste des squences filtres\~english Return the list of the filtered sequences*/
	std::vector<ISequence*>filteredSeq()const;
	/**\~french Charge un fichier rsultat
	@param resFile chemin du fichier de rsultats
	\~english Load a result file
	@param resFile path of the result file
	*/
	void loadResults(char const*resFile);
	/**\~french Retourne dans toutes les valeurs des diffrentes squences  la date \p t pour l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param t date  laquelle les valeurs doivent tre obtenues
	@return vecteur des valeurs rsultat de la requte
	\~english Returns all the values of the different sequences at date \p t for the monitored element with name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param t date at which the values must be obtained
	@return vector of the values resulting from the query
	*/
	std::vector<float>allValues(char const*name,double t)const;
	/**\~french Retourne dans \p values toutes les valeurs de la squence \p seq pour l'lment monitor de nom \pname (le nom doit tre complet).
	@param seq squence  explorer
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts vecteur de stockage des dates des changements de valeurs
	@param values vecteur de stockage des valeurs
	\~english Returns in \p values all the values of the sequence \p seq for the monitored element with name \p name (the name must be complete).
	@param seq sequence to explore
	@param name name of the monitored element whose values are analyzed
	@param ts storage vector of the dates of the value changes
	@param values storage vector of the values
	*/
	void allSeqValues(ISequence const*seq,char const*name,std::vector<double>&ts,std::vector<float>&values)const;
	/**\~french Retourne le nombre de squences pour lesquelles l'lment monitor de nom \p name (le nom doit tre complet) satisfait la condition aux instants \p ts.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates auxquelles les calculs doivent tre faits
	@param condition condition de calcul du temps de rsidence exprime sous forme d'un foncteur boolen de la valeur de l'lment
	@return vecteur rsultat des nombres d'occurrences de la condition aux diffrents instants (autant de valeurs que d'lments dans \p ts)
	\~english Returns the number of sequences for which the monitored element of name \p name (the name must be complete) satisfies the condition at times \p ts.
	@param name name of the monitored element whose values are analyzed
	@param ts dates at which the calculations must be made
	@param condition condition for calculating the residence time expressed as a boolean functor of the value of the element
	@return vector result of the number of occurrences of the condition at different times (as many values as there are elements in \p ts)
	*/
	std::vector<SEQ_ID>occurrences(char const*name,std::vector<double>const&ts,ICondFctFloat const&condition)const;
	/**\~french Retourne le nombre de squences pour lesquelles l'lment monitor de nom \p name (le nom doit tre complet) satisfait la condition aux instants \p ts.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates auxquelles les calculs doivent tre faits
	@param condition condition de calcul du temps de rsidence exprime sous forme d'une fonction boolenne de la valeur de l'lment
	@return vecteur rsultat des nombres d'occurrences de la condition aux diffrents instants (autant de valeurs que d'lments dans \p ts)
	\~english Returns the number of sequences for which the monitored element of name \p name (the name must be complete) satisfies the condition at times \p ts.
	@param name name of the monitored element whose values are analyzed
	@param ts dates at which the calculations must be made
	@param condition condition for calculating the residence time expressed as a boolean function of the value of the element
	@return vector result of the number of occurrences of the condition at different times (as many values as there are elements in \p ts)
	*/
	std::vector<SEQ_ID>occurrences(char const*name,std::vector<double>const&ts,bool (*condition)(float))const{std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));return occurrences(name,ts,*cff);}
	/**\~french Calcule la probabilit que l'lment monitor de nom \p name (le nom doit tre complet) satisfasse la condition avant les instants \p ts.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates limites avant lesquelles la condition doit tre satisfaite
	@param op oprateur de comparaison
	@param value valeur de comparaison
	@return vecteur rsultat des probabilits de ralisation de la condition avant les instants (autant de valeurs que d'lments dans \p ts)
	\~english Calculates the probability that the monitored element with name \p name (the name must be complete) satisfies the condition before the instants \p ts.
	@param name name of the monitored element whose values are analyzed
	@param ts deadlines before which the condition must be satisfied
	@param op comparison operator
	@param value comparison value
	@return vector result of the probabilities of realizing the condition before the instants (as many values as there are elements in \p ts)
	*/
	std::vector<float>realized(char const*name,std::vector<double>const&ts,char const*op,float value)const{std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(op,value));return realized(name,ts,*cff);}
	/**\~french Calcule la probabilit que l'lment monitor de nom \p name (le nom doit tre complet) satisfasse la condition avant les instants \p ts.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param condition condition exprime sous forme d'un foncteur boolen de la valeur de l'lment
	@param ts dates limites avant lesquelles la condition doit tre satisfaite
	@return vecteur rsultat des probabilits de ralisation de la condition avant les instants (autant de valeurs que d'lments dans \p ts)
	\~english Calculates the probability that the monitored element with name \p name (the name must be complete) satisfies the condition before the instants \p ts.
	@param name name of the monitored element whose values are analyzed
	@param condition condition expressed as a boolean functor of the element value
	@param ts deadlines before which the condition must be satisfied
	@return vector result of the probabilities of realizing the condition before the instants (as many values as there are elements in \p ts)
	*/
	std::vector<float>realized(char const*name,std::vector<double>const&ts,ICondFctFloat const&condition)const;
	/**\~french Calcule la probabilit que l'lment monitor de nom \p name (le nom doit tre complet) satisfasse la condition avant les instants \p ts.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param condition condition exprime sous forme d'une fonction boolenne de la valeur de l'lment
	@param ts dates limites avant lesquelles la condition doit tre satisfaite
	@return vecteur rsultat des probabilits de ralisation de la condition avant les instants (autant de valeurs que d'lments dans \p ts)
	\~english Calculates the probability that the monitored element with name \p name (the name must be complete) satisfies the condition before the instants \p ts.
	@param name name of the monitored element whose values are analyzed
	@param condition condition expressed as a boolean function of the element value
	@param ts deadlines before which the condition must be satisfied
	@return vector result of the probabilities of realizing the condition before the instants (as many values as there are elements in \p ts)
	*/
	std::vector<float>realized(char const*name,std::vector<double>const&ts,bool (*condition)(float))const{std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));return realized(name,ts,*cff);}
	/**\~french Retourne le temps de sjour moyen entre \p it et \p ft pendant lequel l'lment monitor de nom \p name (le nom doit tre complet) satisfait la condition.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param it temps initial du calcul
	@param ft temps final du calcul
	@param condition condition de calcul du temps de rsidence exprime sous forme d'un foncteur boolen de la valeur de l'lment
	\~english Returns the average residence time between \p it and \p ft during which the monitored element of name \p name (the name must be complete) satisfies the condition.
	@param name name of the monitored element whose values are analyzed
	@param it initial time of the calculation
	@param ft final time of the calculation
	@param condition condition for calculating the residence time expressed as a boolean functor of the element value
	*/
	float residenceTime(char const*name,double it,double ft,ICondFctFloat const&condition)const;
	/**\~french Retourne le temps de sjour moyen entre \p it et \p ft pendant lequel l'lment monitor de nom \p name (le nom doit tre complet) satisfait la condition.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param it temps initial du calcul
	@param ft temps final du calcul
	@param condition condition de calcul du temps de rsidence exprime sous forme d'une fonction boolenne de la valeur de l'lment
	\~english Returns the average residence time between \p it and \p ft during which the monitored element of name \p name (the name must be complete) satisfies the condition.
	@param name name of the monitored element whose values are analyzed
	@param it initial time of the calculation
	@param ft final time of the calculation
	@param condition condition for calculating the residence time expressed as a boolean function of the element value
	*/
	float residenceTime(char const*name,double it,double ft,bool (*condition)(float))const{std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));return residenceTime(name,it,ft,*cff);}
	/**\~french Retourne les temps de sjour moyen de l'lment monitor de nom \p name (le nom doit tre complet) entre les bornes dfinies par \p lims.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param it date de dbut de calcul de la distribution
	@param ft date de fin de calcul de la distribution
	@param lims limites des intervalles de calcul de la distribution dans l'ordre croissant (si toutes les limites sont nulles, le tableau est rempli automatiquement)
	@return vecteur de stockage de la distribution (une valeur de plus que dans \p lims)
	\~english Returns the average residence times of the monitored element with name \p name (the name must be complete) between the limits defined by \p lims.
	@param name name of the monitored element whose values are analyzed
	@param it start date of the distribution calculation
	@param ft end date of the distribution calculation
	@param lims limits of the distribution calculation intervals in ascending order (if all the limits are zero, the table is filled automatically)
	@return storage vector of the distribution (one value more than in \p lims)
	*/
	std::vector<double>residenceTimes(char const*name,double it,double ft,std::vector<float>const&lims)const;
	/**\~french Calcule les valeurs extrmes de l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param[out] min valeur minimale prise par l'lment
	@param[out] max valeur maximale prise par l'lment
	\~english Calculates the extreme values of the monitored element of name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param[out] min minimum value taken by the element
	@param[out] max maximum value taken by the element
	*/
	void extremeValues(char const*name,float&min,float&max)const;
	/**\~french Calcule les valeurs extrmes de l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@return vecteur[valeur minimale,valeur maximale]
	\~english Calculates the extreme values of the monitored element of name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@return vector[minimum value,maximum value]
	*/
	std::vector<float>extremeValues(char const*name)const{std::vector<float>v(2,(float)HUGE_VAL);extremeValues(name,v[0]=(float)-HUGE_VAL,v[1]);return v;}
	/**\~french Retourne la valeur de l'lment monitor de nom \p name (le nom doit tre complet) telle que la probabilit d'tre au dessus  l'instant \p t est gal  \p pct.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param t instant de calcul
	@param pct dfinition du fractile
	\~english Returns the value of the monitored element with name \p name (the name must be complete) such that the probability of being above at time \p t is equal to \p pct.
	@param name name of the monitored element whose values are analyzed
	@param t calculation time
	@param pct definition of the fractile
	*/
	float quantileGt(char const*name,float t,float pct)const;
	/**\~french Retourne les valeurs de l'lment monitor de nom \p name (le nom doit tre complet) telles que la probabilit d'tre au dessus aux instants \p ts est gal  \p pct.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts instants de calcul
	@param pct dfinition du fractile
	\~english Returns the values of the monitored element with name \p name (the name must be complete) such that the probability of being above at times \p ts is equal to \p pct.
	@param name name of the monitored element whose values are analyzed
	@param ts calculation times
	@param pct definition of the fractile
	*/
	std::vector<float> quantilesGt(char const*name, std::vector<double>const & ts, float pct)const;
	/**\~french Retourne la valeur de l'lment monitor de nom \p name (le nom doit tre complet) telle que la probabilit d'tre au dessous  l'instant \p t est gal  \p pct.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param t instant de calcul
	@param pct dfinition du fractile
	\~english Returns the value of the monitored element with name \p name (the name must be complete) such that the probability of being below at time \p t is equal to \p pct.
	@param name name of the monitored element whose values are analyzed
	@param t calculation time
	@param pct definition of the fractile
	*/
	float quantileLe(char const*name,float t,float pct)const;
	/**\~french Retourne les valeurs de l'lment monitor de nom \p name (le nom doit tre complet) telles que la probabilit d'tre au dessous aux instants \p ts est gal  \p pct.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts instants de calcul
	@param pct dfinition du fractile
	\~english Returns the values of the monitored element with name \p name (the name must be complete) such that the probability of being below at times \p ts is equal to \p pct.
	@param name name of the monitored element whose values are analyzed
	@param ts calculation times
	@param pct definition of the fractile
	*/
	std::vector<float> quantilesLe(char const*name, std::vector<double>const & ts, float pct)const;
	/**\~french Retourne la valeur de l'lment monitor de nom \p name (le nom doit tre complet) telle que le pourcentage moyen du temps pass au dessus est gal  \p pct.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param pct dfinition du fractile : pourcentage du temps de simulation
	@param precis cart maximum tolr entre le rsultat et la valeur exact (le fractile est estim par dichotomie)
	\~english Returns the value of the monitored element with name \p name (the name must be complete) such that the average percentage of time spent above is equal to \p pct.
	@param name name of the monitored element whose values are analyzed
	@param pct definition of the fractile: percentage of the simulation time
	@param precis maximum tolerated difference between the result and the exact value (the fractile is estimated by dichotomy)
	*/
	float quantile(char const*name,float pct,float precis)const;
	/**\~french Retourne les  fractiles min et max pour les instants du vecteur \p ts pour l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts vecteur temps pour le calcul des fractiles
	@param[out] quantilesMin vecteur des quantiles minimaux
	@param pctMin niveaux des quantiles minimaux demands
	@param[out] quantilesMax vecteur des quantiles maximaux
	@param pctMax niveaux des quantiles maximaux demands
	\~english Returns the min and max fractiles for the instants of the vector \p ts for the monitored element with name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param ts time vector for calculating fractiles
	@param[out] quantilesMin vector of minimal quantiles
	@param pctMin levels of minimal quantiles requested
	@param[out] quantilesMax vector of maximal quantiles
	@param pctMax levels of maximal quantiles requested
	*/
	void quantiles(char const*name, std::vector<double>const & ts, std::vector<float> & quantilesMin, float pctMin, std::vector<float> & quantilesMax, float pctMax)const;
	/**\~french Retourne la moyenne de la valeur de l'lment monitor de nom \p name (le nom doit tre complet) entre les temps \p it et \p ft.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param it instant initial de calcul
	@param ft instant final de calcul
	@return valeur moyenne calcule
	\~english Returns the average of the value of the monitored element with name \p name (the name must be complete) between times \p it and \p ft.
	@param name name of the monitored element whose values are analyzed
	@param it initial instant of calculation
	@param ft final instant of calculation
	@return average value calculated
	*/
	float meanValue(char const*name,double it,double ft)const;//Moyenne globale entre les temps it et ft
	/**\~french Retourne dans \p mean et \p stdDev la moyenne et l'cart type  l'instant \p t pour l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param t date  laquelle les calculs doivent tre faits
	@param[out] mean moyenne rsultat de l'analyse  la date \p t
	@param[out] stdDev cart type rsultat de l'analyse  la date \p t
	\~english Returns in \p mean and \p stdDev the mean and standard deviation at time \p t for the monitored element with name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param t date on which the calculations must be made
	@param[out] mean mean result of the analysis at date \p t
	@param[out] stdDev standard deviation result of the analysis at date \p t
	*/
	void meanValue(char const*name,double t,float&mean,float&stdDev)const;//Moyenne et cart type  l'instant t
	/**\~french Retourne dans \p means et \p stdDevs les moyennes et les carts types aux instants \p ts pour l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates auxquelles les calculs doivent tre faits
	@param[out] means vecteur de stockage des moyennes (autant de valeurs que de points dans \p ts)
	@param[out] stdDevs vecteur de stockage des carts type (autant de valeurs que de points dans \p ts)
	\~english Returns in \p means and \p stdDevs the means and standard deviations at times \p ts for the monitored element with name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param ts dates at which the calculations must be made
	@param[out] means storage vector of means (as many values as points in \p ts)
	@param[out] stdDevs storage vector of standard deviations (as many values as points in \p ts)
	*/
	void meanValues(char const*name,std::vector<double>const&ts,std::vector<float>&means,std::vector<float>&stdDevs)const;
	/**\~french Retourne les moyennes aux instants \p ts pour l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates auxquelles les calculs doivent tre faits
	@return vecteur de stockage des moyennes (autant de valeurs que de points dans \p ts)
	\~english Returns the averages at times \p ts for the monitored element with name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param ts dates at which the calculations must be made
	@return vector for storing the averages (as many values as there are points in \p ts)
	*/
	std::vector<float>meanValues(char const*name,std::vector<double>const&ts)const;
	/**\~french Retourne les intervalles de confiance aux instants \p ts pour l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates auxquelles les calculs doivent tre faits
	@param pct pourcentage de confiance de l'intervalle
	@return vecteur de stockage des intervalles de confiance (autant de valeurs que de points dans \p ts)
	\~english Returns the confidence intervals at times \p ts for the monitored element with name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param ts dates at which the calculations must be made
	@param pct confidence percentage of the interval
	@return vector for storing the confidence intervals (as many values as there are points in \p ts)
	*/
	std::vector<float>confIntValues(char const*name,std::vector<double>const&ts,float pct)const;
	/**\~french Retourne les carts types aux instants \p ts pour l'lment monitor de nom \p name (le nom doit tre complet).
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates auxquelles les calculs doivent tre faits
	@return vecteur de stockage des carts type (autant de valeurs que de points dans \p ts)
	\~english Returns the standard deviations at times \p ts for the monitored element with name \p name (the name must be complete).
	@param name name of the monitored element whose values are analyzed
	@param ts dates at which the calculations must be made
	@return vector for storing the standard deviations (as many values as there are points in \p ts)
	*/
	std::vector<float>stdDevValues(char const*name,std::vector<double>const&ts)const;
	/**\~french Retourne la distribution de la valeur de l'lment monitor de nom \p name (le nom doit tre complet)  l'instant \p t.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param t date de calcul de la distribution
	@param lims limites des intervalles de calcul de la distribution dans l'ordre croissant (si toutes les limites sont nulles, le tableau est rempli automatiquement)
	@return vecteur de stockage de la distribution (une valeur de plus que dans \p lims)
	\~english Returns the distribution of the value of the monitored element with name \p name (the name must be complete) at time \p t.
	@param name name of the monitored element whose values are analyzed
	@param t date of calculation of the distribution
	@param lims limits of the intervals of calculation of the distribution in ascending order (if all the limits are zero, the table is filled automatically)
	@return vector of storage of the distribution (one value more than in \p lims)
	*/
	std::vector<float>distribution(char const*name,double t,std::vector<float>const&lims)const;
	/**\~french Retourne dans la distribution de la valeur de l'lment monitor de nom \p name (le nom doit tre complet) moyenne sur les instants \p ts.
	@param name nom de l'lment monitor dont on analyse les valeurs
	@param ts dates de calcul de la distribution
	@param lims limites des intervalles de calcul de la distribution dans l'ordre croissant (si toutes les limites sont nulles, le tableau est rempli automatiquement)
	@return vecteur de stockage de la distribution (une valeur de plus que dans \p lims)
	\~english Returns in the distribution the value of the monitored element of name \p name (the name must be complete) averaged over the instants \p ts.
	@param name name of the monitored element whose values are analyzed
	@param ts calculation dates of the distribution
	@param lims limits of the calculation intervals of the distribution in ascending order (if all the limits are zero, the table is filled automatically)
	@return storage vector of the distribution (one value more than in \p lims)
	*/
	std::vector<float> meanDistribution(char const * name, std::vector<double>const & ts, std::vector<float>const & lims)const;
	/**\~french Enregistre un fichier XML des squences filtres
	Par dfaut, les squences contenant les mmes successions de transitions sont regroupes.
	Il est possible de regrouper diffremment les squences en dfinissant des classes de transitions setTransitionClass().
	Dans ce cas, le regroupement tient compte du fait que 2 transitions appartenant  une mme classe sont considres comme identiques.
	Il est aussi possible de dfinir explicitement des groupes de squences en utilisant des filtres setSequenceClass()
	@param pct pourcentage des squences filtres restitu
	@param file nom du fichier XML de squences  construire
	@param xsltFile nom du fichier de feuille de style utilis pour construire le fichier HTML correspondant (si NULL, le fichier HTML n'est pas construit)
	\~english Saves an XML file of the filtered sequences
	By default, sequences containing the same successions of transitions are grouped together.
	It is possible to group sequences differently by defining transition classes setTransitionClass().
	In this case, the grouping takes into account the fact that 2 transitions belonging to the same class are considered identical.
	It is also possible to explicitly define groups of sequences by using filters setSequenceClass()
	@param pct percentage of filtered sequences returned
	@param file name of the XML file of sequences to build
	@param xsltFile name of the style sheet file used to build the corresponding HTML file (if NULL, the HTML file is not built)
	*/
	bool printFilteredSeq(float pct,char const*file,char const*xsltFile=NULL)const;
	/**\~french Marque les squences filtres comme  conserver ou non pour une poursuite de la simulation\~english Marks the filtered sequences as to be kept or not for further simulation.*/
	bool keepFilteredSeq(bool restart);
	/**\~french Excute l'analyse paramtre par le fichier
	@param filename nom du fichier de paramtrage de l'analyse
	\~english Runs the analysis parameterized by the file
	@param filename name of the analysis parameter file
	*/
	bool analyse(char const*filename);
	/**\~french Place la transition dans une classe (ou supprime la transition des classes si className est vide)\~english Places the transition in a class (or removes the transition from classes if className is empty)*/
	void setTransitionClass(ITransition const*trans,int ind,char const*className);
	/**\~french Supprime une classe de transition\~english Remove a transition class*/
	void removeTransitionClass(char const*className){setTransitionClass(NULL,0,className);}
	/**\~french Dfinit une classe de squences par un filtre\~english Defines a sequence class by a filter*/
	void setSequenceClass(char const*className,IFilterFct*filter);
	/**\~french Supprime une classe de squence (toutes si className est NULL)\~english Removes a sequence class (all if className is NULL)*/
	void removeSequenceClass(char const*className){setSequenceClass(className,NULL);}
};

