/*************************************
*         Copyright 2015 EDF         *
*************************************/
#ifndef COMPONENT_H
#define COMPONENT_H
/**\file
\~french Fichier de dfinition de la classe CComponent qui doit servir de base  toutes les classes de composants utilisables dans PyCATSHOO.
\~english Definition file for the CComponent class which should serve as the basis for all component classes usable in PyCATSHOO.
*/

#include "Interfaces.h"

class CState;
class CMessageBox;
class CTransition;
class CSystem;
class CComponentP;
class CVariable;

/**\~french Classe de gestion des composants.
	Les composants correspondent aux types de la BdC.
	Il est possible d'inclure des composants dans un autre composant (diagramme hirarchique).

	Un composant est un fils d'un systme ou d'un autre composant ; il peut contenir :
	- un ou plusieurs diagrammes d'tats (automate (IAutomaton), tats (IState) et transitions (ITransition) entre tats),
	- des variables (IVariable),
	- des rfrences  ces variables (IReference),
	- des botes de messages (IMessageBox).

	Un composant peut tre muni de mthodes qui peuvent tre dclares comme sensibles aux variations des tats et des variables.
	Les mthodes sensibles (callback) doivent tre de retour void et sans paramtre.

	Un composant peut tre muni de mthodes appeles systmatiquement par le systme aux instants suivants :
	- une unique fois avant le lancement de la simulation
	-  chaque dbut de squence
	- avant l'appel des mthodes sensibles
	- aprs l'appel des mthodes sensibles

exemple:
~~~~{.cpp}
//Dfinition de la classe CComp
class CComp :public CComponent
{
public:
	//Constructeur pour un composant directement plac dans le systme
	Ccomp(char const*name,CSystem&system):CComponent(name,system){
		//Construction des variables, des rfrences, des botes de messages, des graphes d'tats
		//Construction des gestionnaires d'ODE
		//Construction des systmes d'quations linaires
		//Dclaration de la mthode d'initialisation
		addStartMethod("start",&CComp::start) ;
	}
	//Dfinition d'une mthode
	void methode(){
		//Actions de la mthode
	}
	//Dfinition d'une mthode d'initialisation
	void start(){
		//Initialisations
	}
}
~~~~
\~english Component management class.
Components correspond to the types in the Knowledge Base.
It is possible to include components within other components (hierarchical diagram).

A component is a child of a system or another component; it can contain:
- one or more state diagrams (automaton (`IAutomaton`), states (`IState`), and transitions (`ITransition`) between states),
- variables (`IVariable`),
- references to these variables (`IReference`),
- message boxes (`IMessageBox`).

A component can be equipped with methods that can be declared as sensitive to changes in states and variables.
Sensitive methods (callbacks) must have a `void` return type and no parameters.

A component can also be equipped with methods called systematically by the system at the following moments:
- once, before the simulation begins,
- at the start of each sequence,
- before calling the sensitive methods,
- after calling the sensitive methods.

example:
~~~~{.cpp}
// Definition of the CComp class
class CComp : public CComponent
{
public:
	// Constructor for a component directly placed in the system
	CComp(char const* name, CSystem& system) : CComponent(name, system) {
		// Construction of variables, references, message boxes, and state graphs
		// Construction of ODE managers
		// Construction of linear equation systems
		// Declaration of the initialization method
		addStartMethod("start", &CComp::start);
	}

	// Definition of a method
	void methode() {
		// Actions of the method
	}

	// Definition of an initialization method
	void start() {
		// Initializations
	}
}
~~~~
*/
class PYC_PUBLIC CComponent:public CNamedComp
{
	friend class CSystem;
	friend class CSystemP;
	friend class CComponentP;
	friend struct CPycObj;
	CComponentP*m_CompP;
	bool m_bRunning;//Si vrai, le composant a t cr pendant la simulation
	bool m_bDeleted;//Si vrai, le composant est dj dtruit
	bool m_bKB;//Si vrai le composant a t construit par la BdC (construction d'une hirarchie automatique)
	void updatePlanning();
	void updatePlanningInt();
	static CSystem* glSystem();
	CState*buildState(char const*name,char const*automate,unsigned index);
	CVariable*buildVariable(char const*,char const*type);
	void activate();//!<\~french Active tous les tats du composant mais pas de ses sous composants.\~english Activates all the states of the component but not its subcomponents.
	void activateAll();//!<\~french Active les tats du composant et de ses sous composants.\~english Activates the states of the component and its subcomponents.
	void resetAll();//!<\~french Rinitialise toutes les variables et dsactive les tats du composant et de ses sous composants.\~english Resets all the variables and deactivates the states of the component and its subcomponents.
	void deleteData();
protected:
	CComponent(PyObject*pyObj,char const*name,CSysNamedComp*parent=NULL);
	CComponent(char const* name, CSystem& system);//!<\~french Pour la compatibilit ascendante\~english For backward compatibility.
	PyObject*m_Self;//!<\~french Rfrence  l'objet python correspondant (NULL, si c'est un objet purement C++).\~english Reference to the corresponding Python object (NULL if it is a purely C++ object).
	void storeParameter(char const*name,char const*value);//!<\~french Uniquement pour autobuild\~english Only for autobuild.
	void updateName()override;
	void addSysMethod(char const*name,void(CComponent::*method)(),PyObject*pVoid,bool bCheck);
	CMethod* method(char const* name, void(CSysNamedComp::* method)())override;
	CMethod* method(char const* name, PyObject* pyObj)override;

	virtual void deleted(){}//!<\~french Appele en fin de deleteCmp pour effectuer un mnage spcifique (singletons Python), mais inutile dans le destructeur !!\~english Called at the end of `deleteCmp` to perform specific cleanup (Python singletons), but unnecessary in the destructor!
	void addSLEEquation(char const* systemName, char const* equation, PyObject* pyObj) { SLEManager(systemName)->addEquation(equation, *this, pyObj); }
	void addMILPInequality(char const* systemName, char const* equation, PyObject* pyObj) { MILPManager(systemName)->addInequality(equation, *this, pyObj); }
	void addPDMPEquationMethod(char const* systemName, char const* ode, PyObject* pyObj, int order = 0) { PDMPManager(systemName)->addEquationMethod(this, ode, NULL, pyObj, order); }
	void addPDMPJacobianMethod(char const* systemName, char const* ode, PyObject* pyObj, int order = 0) { PDMPManager(systemName)->addJacobianMethod(this, ode, NULL, pyObj, order); }
	void addPDMPCondition(char const* systemName, char const* condition, PyObject* pyObj) { PDMPManager(systemName)->addCondition(this, condition, NULL, pyObj); }
	void addPDMPBeginMethod(char const* systemName, char const* fct, PyObject* pyObj) { PDMPManager(systemName)->addBeginMethod(this, fct, NULL, pyObj); }
	void addPDMPEndMethod(char const* systemName, char const* fct, PyObject* pyObj) { PDMPManager(systemName)->addEndMethod(this, fct, NULL, pyObj); }
	void addPDMPPreStepMethod(char const* systemName, char const* fct, PyObject* pyObj) { PDMPManager(systemName)->addPreStepMethod(this, fct, NULL, pyObj); }
	void addPDMPPostStepMethod(char const* systemName, char const* fct, PyObject* pyObj) { PDMPManager(systemName)->addPostStepMethod(this, fct, NULL, pyObj); }
public:
	virtual~CComponent(void);
	CComponent(char const*name,CSysNamedComp&parent);//!<\~french Constructeur d'un composant de nom name comme lment du parent\~english Constructor for a component named name as an element of the parent.
	explicit CComponent(char const*name);//!<\~french Constructeur d'un composant de nom name comme lment du systme courant\~english Constructor for a component named name as an element of the current system.
	CComponentP&compP(){return *m_CompP;}//!<\~french Retourne le stockage priv (uniquement en usage interne)\~english Returns the private storage (for internal use only).
	CComponentP const&compP()const{return *m_CompP;}//!<\~french Retourne le stockage priv (uniquement en usage interne)\~english Returns the private storage (for internal use only).
	void deleteCmp();//!<\~french Dtruit le composant sauf si m_Self est non nul (alors, ne dtruit que l'intrieur)\~english Destroys the component unless m_Self is not null (in which case, only the internal part is destroyed).
	void setKBDefault();//!<\~french Signifie que l'tat courant du composant est celui construit par la BdC\~english Indicates that the current state of the component is the one built by the BdC.
	bool isKBComponent()const { return m_bKB; }//!<\~french Retourne vrai si le composant a t construit automatiquement par la BdC\~english Returns true if the component was automatically constructed by the BdC.
	PyObject*self()const{return m_Self;}
	void rename(char const*newName);//!<\~french Modifie le nom du composant\~english Modifies the name of the component.
	std::vector<CComponent*>components(char const*name)const;//!<\~french Retourne les sous composants de nom name (nom compos dans le composant : pas d'expression rgulire)\~english Returns the subcomponents named `name` (name composed within the component: no regular expressions).
	std::vector<CComponent*>getComponents(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getComponents","components");return components(name);}//!<\~french Obsolte\~english Deprecated.
	std::vector<IMessageBox*>messageBoxes()const;//!<\~french Retourne les botes de messages du composant (pas de ses enfants)\~english Returns the message boxes of the component (not its children).
	std::vector<IMessageBox*>getMessageBoxes(){ILogManager::glLogManager().msgObsolete("CComponent::getMessageBoxes","messageBoxes");return messageBoxes();}//!<\~french Obsolte\~english Deprecated.
	std::vector<IVariable*> variables()const;//!<\~french Retourne les variables du composant (pas de ses enfants)\~english Returns the variables of the component (not its children).
	std::vector<IVariable*> variables(char const *name, TVarType::EVarType * type)const;//!<\~french Retourne les variables du composant (pas de ses enfants) de nom matchant avec name et de type type\~english Returns the variables of the component (not its children) that match the name `name` and type `type`.
	std::vector<IVariable*> variables(char const *name)const{return variables(name,NULL);}
	std::vector<IVariable*>getVariables(){ILogManager::glLogManager().msgObsolete("CComponent::getVariables","variables");return variables();}//!<\~french Obsolte\~english Deprecated.
	std::vector<IReference*>references()const;//!<\~french Retourne les rfrences du composant (pas de ses enfants)\~english Returns the references of the component (not its children).
	std::vector<IReference*>getReferences(){ILogManager::glLogManager().msgObsolete("CComponent::getReferences","references");return references();}//!<\~french Obsolte\~english Deprecated.
	std::vector<IAutomaton*>automata()const;//!<\~french Retourne les automates du composant (pas de ses enfants)\~english Returns the automatons of the component (not its children).
	std::vector<IAutomaton*>getAutomata(){ILogManager::glLogManager().msgObsolete("CComponent::getAutomata","automata");return automata();}//!<\~french Obsolte\~english Deprecated.
	std::vector<IState*>states()const;//!<\~french Retourne les tats du composant (pas de ses enfants)\~english Returns the states of the component (not its children).
	std::vector<IState*>getStates(){ILogManager::glLogManager().msgObsolete("CComponent::getStates","states");return states();}//!<\~french Obsolte\~english Deprecated.
	std::vector<ITransition*>transitions()const;//!<\~french Retourne les transitions du composant (pas de ses enfants)\~english Returns the transitions of the component (not its children).
	std::vector<ITransition*>getTransitions()const{ILogManager::glLogManager().msgObsolete("CComponent::getTransitions","transitions");return transitions();}//!<\~french Obsolte\~english Deprecated.
	IPDMPManager*PDMPManager(char const*name)const;//!<\~french Retourne le gestionnaire d'ODE de nom name\~english Returns the ODE manager named `name`.
	IPDMPManager*getPDMPManager(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getPDMPManager","PDMPManager");return PDMPManager(name);}//!<\~french Obsolte\~english Deprecated.
	ISLEManager*SLEManager(char const*name)const;//!<\~french Retourne le gestionnaire de systme d'quations linaires de nom name\~english Returns the linear equation system manager named `name`.
	ISLEManager*getSLEManager(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getSLEManager","SLEManager");return SLEManager(name);}//!<\~french Obsolte\~english Deprecated.
	IMILPManager* MILPManager(char const* name)const;//!<\~french Retourne le gestionnaire d'optimisation lineaire sous contraintes de nom name\~english Returns the constrained linear optimization manager named `name`.
	IMILPManager* getMILPManager(char const* name)const { ILogManager::glLogManager().msgObsolete("CComponent::getMILPManager", "MILPManager"); return MILPManager(name); }//!<\~french Obsolte\~english Deprecated.
	IAutomaton*automaton(char const*name)const;//!<\~french Retourne l'automate de nom name.\~english Returns the automaton named `name`.
	IAutomaton*getAutomaton(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getAutomaton","automaton");return automaton(name);}//!<\~french Obsolte\~english Deprecated.
	IPDMPManager*addPDMPManager(char const*name);//!<\~french Ajoute un gestionnaire d'ODE de nom name s'il n'existe pas dj\~english Adds an ODE manager named `name` if it does not already exist.
	ISLEManager*addSLEManager(char const*name);//!<\~french Ajoute un gestionnaire de systme d'quations linaires de nom name s'il n'existe pas dj\~english Adds a linear equation system manager named `name` if it does not already exist.
	IMILPManager* addMILPManager(char const* name);//!<\~french Ajoute un gestionnaire d'optimisation linaire sous contraintes de nom name s'il n'existe pas dj\~english Adds a constrained linear optimization manager named `name` if it does not already exist.
	/**\~french Ajoute une mthode de contrle du paramtrage du composant appele en dbut de simulation\~english Adds a component configuration control method called at the start of the simulation.*/
	template<typename Type>void addCheckMethod(char const*name,void(Type::*method)()){checkMethodAddrSize(sizeof(method));addSysMethod(name,static_cast<void(CComponent::*)()>(method),NULL,true);}
	/**\~french Ajoute une mthode de dmarrage du composant (appele en dbut de chaque squence\~english Adds a component start method (called at the beginning of each sequence).*/
	template<typename Type>void addStartMethod(char const*name,void(Type::*method)()){checkMethodAddrSize(sizeof(method));addSysMethod(name,static_cast<void(CComponent::*)()>(method),NULL,false);}
	/**\~french Ajoute une tape de nom name.
	La mthode retourne l'tape ajoute ou retrouve dans la liste des tapes.
	\~english Adds a step named `name`.  
	The method returns the step that was added or found in the list of steps.*/
	IStep*addStep(char const*name);
	/**\~french Ajoute une mthode de nom methodName  l'tape de nom stepName.\~english Adds a method named `methodName` to the step named `stepName`.*/
	void addStepMethod(char const*stepName,char const*methodName);
	/**\~french Construit un automate de nom name
	@param name nom de l'automate  ajouter
	@return l'automate construit
	\~english Constructs an automaton named `name`.  
	@param name The name of the automaton to be added.  
	@return The constructed automaton.*/
	IAutomaton*addAutomaton(char const*name);
	/**\~french Retourne le nom de la classe du composant\~english Returns the class name of the component.*/
	virtual char const*className()const;
	/**\~french Retourne le sous composant de nom name ventuellement compos (si check = vrai, lance une exception si le composant est introuvable).\~english Returns the subcomponent named `name`, potentially composite (if `check` is true, raises an exception if the component is not found).*/
	CComponent*component(char const*name,bool check=true)const;
	/**\~french Retourne le sous composant de nom name ventuellement compos (si check = vrai, lance une exception si le composant est introuvable) (obsolte).\~english Returns the subcomponent named `name`, potentially composite (if `check` is true, raises an exception if the component is not found) (deprecated).*/
	DEPRECATED CComponent*getComponent(char const*name,bool check=true)const{ILogManager::glLogManager().msgObsolete("CComponent::getComponent","component");return component(name,check);}//!<\~french Obsolte\~english Deprecated.
	/**\~french Ajoute une quation  rsoudre.
	@param systemName nom du systme d'quations linaires  modifier
	@param name nom de l'quation  ajouter
	@param equation mthode de dfinition de l'quation
	\~english Adds an equation to solve.
	@param systemName The name of the linear equation system to modify.
	@param name The name of the equation to add.
	@param equation The method for defining the equation.
	*/
	template<typename Type>void addSLEEquation(char const*systemName,char const*name,void (Type::*equation)(IEquation&)){
		checkMethodAddrSize(sizeof(equation));
		SLEManager(systemName)->addEquation(name,*this,static_cast<void(CSysNamedComp::*)(IEquation&)>(equation));
	}
	/**\~french Ajoute une variable au systme linaire de nom name.
	@param systemName nom du systme d'quations linaires  modifier
	@param v variable  ajouter dans le systme
	\~english Adds a variable to the linear system named `name`.
	@param systemName The name of the linear equation system to modify.
	@param v The variable to add to the system.
	*/
	void addSLEVariable(char const*systemName,IVariable&v){SLEManager(systemName)->addVariable(v);}
	/**\~french Ajoute une inquation  rsoudre pour l'optimisation linaire.
	@param systemName nom de l'optimiseur sous contraintes  modifier
	@param name nom de l'inquation  ajouter
	@param inequality mthode de dfinition de l'inquation
	\~english Adds an inequality to solve for linear optimization.
	@param systemName The name of the constrained optimizer to modify.
	@param name The name of the inequality to add.
	@param inequality The method for defining the inequality.
	*/
	template<typename Type>void addMILPInequality(char const*systemName,char const*name,void (Type::*inequality)(IInequality&)){
		checkMethodAddrSize(sizeof(inequality));
		MILPManager(systemName)->addInequality(name,*this,static_cast<void(CComponent::*)(IInequality&)>(inequality));
	}
	/**\~french Ajoute une variable  l'optimiseur linaire de nom name.
	@param systemName nom de l'optimiseur linaire  modifier
	@param v variable  ajouter au systme
	@param co coefficient de la fonction objectif associ  la variable
	\~english Adds a variable to the linear optimizer named `name`.
	@param systemName The name of the linear optimizer to modify.
	@param v The variable to add to the system.
	@param co The coefficient of the objective function associated with the variable.
	*/
	void addMILPVariable(char const*systemName,IVariable&v,double co){MILPManager(systemName)->addVariable(v,co);}
	/**\~french Ajoute une variable  l'optimiseur linaire de nom name.
	@param systemName nom de l'optimiseur linaire  modifier
	@param v variable  ajouter au systme
	\~english Adds a variable to the linear optimizer named `name`.
	@param systemName The name of the linear optimizer to modify.
	@param v The variable to add to the system.
	*/
	void addMILPVariable(char const*systemName,IVariable&v){MILPManager(systemName)->addVariable(v);}
	/**\~french Ajoute une mthode de calcul des quations diffrentielles et/ou des variables calcules.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode de calcul
	@param odeFct mthode de calcul
	@param order numro d'ordre d'appel de la mthode
	\~english Adds a method to compute the differential equations and/or calculated variables.
	@param systemName The name of the ODE system to modify.
	@param name The name of the computation method.
	@param odeFct The method for computation.
	@param order The order number for calling the method.
	*/
	template<typename Type>void addPDMPEquationMethod(char const*systemName,char const*name,void(Type::*odeFct)(),int order = 0){
		checkMethodAddrSize(sizeof(odeFct));
		PDMPManager(systemName)->addEquationMethod(name,*this,static_cast<void(CComponent::*)()>(odeFct),order);
	}
	/**\~french Ajoute une mthode de calcul des quations diffrentielles et/ou des variables calcules.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode de calcul
	@param odeFct mthode de calcul
	@param order numro d'ordre d'appel de la mthode
	\~english Adds a method to compute the differential equations and/or calculated variables.
	@param systemName The name of the ODE system to modify.
	@param name The name of the computation method.
	@param odeFct The method for computation.
	@param order The order number for calling the method.
	*/
	template<typename Type>void addPDMPJacobianMethod(char const*systemName,char const*name,void(Type::*odeFct)(),int order = 0){
		checkMethodAddrSize(sizeof(odeFct));
		PDMPManager(systemName)->addJacobianMethod(name,*this,static_cast<void(CComponent::*)()>(odeFct),order);
	}
	/**\~french Ajoute une variable comme lment de l'tat de l'ODE de nom name.
	@param systemName nom du systme d'ODE  modifier
	@param v variable gre par le systme d'ODE.
	\~english Adds a variable as an element of the ODE state named `name`.
	@param systemName The name of the ODE system to modify.
	@param v The variable managed by the ODE system.
	*/
	void addPDMPODEVariable(char const*systemName,IVariable&v);
	/**\~french Ajoute une variable comme lment de l'tat du systme d'quations diffrentielles de nom name.
	@param systemName nom du systme d'quations diffrentielles  modifier
	@param v variable gre par le systme d'quations diffrentielles.
	\~english Adds a variable as an element of the differential equations system state named `name`.
	@param systemName The name of the differential equations system to modify.
	@param v The variable managed by the differential equations system.
	*/
	void addPDMPDifferentialVariable(char const*systemName,IVariable&v){addPDMPODEVariable(systemName,v);}
	/**\~french Ajoute une variable comme lment de l'tat du systme d'quations algbriques de nom name.
	@param systemName nom du systme d'quations algbriques  modifier
	@param v variable gre par le systme d'quations algbriques.
	\~english Adds a variable as an element of the algebraic equations system state named `name`.
	@param systemName The name of the algebraic equations system to modify.
	@param v The variable managed by the algebraic equations system.
	*/
	void addPDMPAlgebraicVariable(char const*systemName,IVariable&v);
	/**\~french Ajoute une variable comme lment calcul au cours de l'ODE.
	@param systemName nom du systme d'ODE  modifier
	@param v variable calcule  ajouter
	\~english Adds a variable as a computed element during the ODE.
	@param systemName The name of the ODE system to modify.
	@param v The computed variable to add.
	*/
	void addPDMPExplicitVariable(char const*systemName,IVariable&v);
	/**\~french Ajoute une condition  l'ODE.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode  utiliser comme condition
	@param condFct mthode condition (doit retourner un double)
	\~english Adds a condition to the ODE.
	@param systemName The name of the ODE system to modify.
	@param name The name of the method to use as the condition.
	@param condFct The condition method (must return a double).
	*/
	template<typename Type>void addPDMPCondition(char const*systemName,char const*name,double(Type::*condFct)()){
		checkMethodAddrSize(sizeof(condFct));
		PDMPManager(systemName)->addCondition(name,*this,static_cast<double(CComponent::*)()>(condFct));
	}
	/**\~french Ajoute une condition  l'ODE.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode  utiliser comme condition
	@param condFct mthode condition (doit retourner un double strictement positif pour continuer l'intgration)
	\~english Adds a condition to the ODE.
	@param systemName The name of the ODE system to modify.
	@param name The name of the method to use as the condition.
	@param condFct The condition method (must return a strictly positive double to continue the integration).
	*/
	template<typename Type>void addPDMPBoundaryCheckerMethod(char const*systemName,char const*name,double(Type::*condFct)()){
		checkMethodAddrSize(sizeof(condFct));
		PDMPManager(systemName)->addCondition(name,*this,static_cast<double(CComponent::*)()>(condFct));
	}
	/**\~french Ajoute un condition  l'ODE sous la forme de la condition de la transition trans (l'intgration est stoppe si la condition est vrifie)\~english Adds a condition to the ODE in the form of the transition condition `trans` (the integration is stopped if the condition is met).*/
	void addPDMPWatchedTransition(char const*systemName,ITransition&trans){PDMPManager(systemName)->addWatchedTransition(trans);}
	/**\~french Ajoute un condition  l'ODE sous la forme de la condition de la transition de nom name (l'intgration est stoppe si la condition est vrifie)\~english Adds a condition to the ODE in the form of the transition condition named `name` (the integration is stopped if the condition is met).*/
	void addPDMPWatchedTransition(char const*systemName,char const*trans){PDMPManager(systemName)->addWatchedTransition(*transition(trans));}
	/**\~french Dfinit une fonction appele avant l'intgration de l'ODE.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode  appeler en dbut d'intgration
	@param beginFct mthode  appeler
	\~english Defines a function to be called before the ODE integration.
	@param systemName The name of the ODE system to modify.
	@param name The name of the method to call at the beginning of the integration.
	@param beginFct The method to call.
	*/
	template<typename Type>void addPDMPBeginMethod(char const*systemName,char const*name,void(Type::*beginFct)()){
		checkMethodAddrSize(sizeof(beginFct));
		PDMPManager(systemName)->addBeginMethod(name,*this,static_cast<void(CComponent::*)()>(beginFct));
	}
	/**\~french Dfinit une fonction appele aprs l'intgration de l'ODE.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode  appeler en fin d'intgration
	@param endFct mthode  appeler
	\~english Defines a function to be called after the ODE integration.
	@param systemName The name of the ODE system to modify.
	@param name The name of the method to call at the end of the integration.
	@param endFct The method to call.
	*/
	template<typename Type>void addPDMPEndMethod(char const*systemName,char const*name,void(Type::*endFct)(char const*)){
		checkMethodAddrSize(sizeof(endFct));
		PDMPManager(systemName)->addEndMethod(name,*this,static_cast<void(CComponent::*)(char const*)>(endFct));
	}
	/**\~french Dfinit une fonction appele avant chaque pas d'intgration de l'ODE.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode  appeler en fin d'intgration
	@param preFct mthode  appeler
	\~english Defines a function to be called before each integration step of the ODE.
	@param systemName The name of the ODE system to modify.
	@param name The name of the method to call before each integration step.
	@param preFct The method to call.
	*/
	template<typename Type>void addPDMPPreStepMethod(char const*systemName,char const*name,void(Type::*preFct)()){
		checkMethodAddrSize(sizeof(preFct));
		PDMPManager(systemName)->addPreStepMethod(name,*this,static_cast<void(CComponent::*)()>(preFct));
	}
	/**\~french Dfinit une fonction appele aprs chaque pas d'intgration de l'ODE.
	@param systemName nom du systme d'ODE  modifier
	@param name nom de la mthode  appeler en fin d'intgration
	@param postFct mthode  appeler
	\~english Defines a function to be called after each integration step of the ODE.
	@param systemName The name of the ODE system to modify.
	@param name The name of the method to call after each integration step.
	@param postFct The method to call.
	*/
	template<typename Type>void addPDMPPostStepMethod(char const*systemName,char const*name,void(Type::*postFct)()){
		checkMethodAddrSize(sizeof(postFct));
		PDMPManager(systemName)->addPostStepMethod(name,*this,static_cast<void(CComponent::*)()>(postFct));
	}
	/**\~french Ajoute une bote de message.
	@param name nom de la bote de messages  ajouter
	@return la bote de messages construite
	\~english Adds a message box.
	@param name The name of the message box to add.
	@return The constructed message box.
	*/
	IMessageBox*addMessageBox(char const*name);
	/**\~french Ajoute une variable  exporter par la bote de messages.
	@param name nom de la bote de messages
	@param v variable  exporter
	@param alias nom sous lequel la variable doit tre vue lors de la connexion  cette bote de messages
	\~english Adds a variable to be exported by the message box.
	@param name The name of the message box.
	@param v The variable to export.
	@param alias The name under which the variable should be seen when connecting to this message box.
	*/
	void addMessageBoxExport(char const*name,IVarBase*v,char const*alias=NULL);
	void addMessageBoxExport(char const*name,IVarBase&v,char const*alias=NULL){addMessageBoxExport(name,&v,alias);}
	/**\~french Ajoute une rfrence  importer par la bote de message et prcise ne nom de la variable  y connecter.
	@param name nom de la bote de messages
	@param r rfrence  importer
	@param varName nom ou alias de la variable  connecter  la rfrence
	\~english Adds a reference to be imported by the message box and specifies the name of the variable to connect to it.
	@param name The name of the message box.
	@param r The reference to import.
	@param varName The name or alias of the variable to connect to the reference.
	*/
	void addMessageBoxImport(char const*name,IReference*r,char const*varName);
	void addMessageBoxImport(char const*name,IReference&r,char const*varName){addMessageBoxImport(name,&r,varName);}
	/**\~french Ajoute une rfrence  la bote de message et prcise ne nom de la variable  y connecter de manire optionnelle ainsi que la valeur  attribuer en l'absence de connexion.
	@param name nom de la bote de messages
	@param r rfrence  importer
	@param type type de la valeur parmi TValueType
	@param defaultValue valeur par dfaut attribue en l'abscence de connexion
	@param varName nom ou alias de la variable  connecter  la rfrence
	\~english Adds a reference to the message box and optionally specifies the name of the variable to connect to it, along with the value to assign if no connection exists.
	@param name The name of the message box.
	@param r The reference to import.
	@param type The type of the value from `TValueType`.
	@param defaultValue The default value to assign in the absence of a connection.
	@param varName The name or alias of the variable to connect to the reference.
	*/
	void addMessageBoxOptionalImport(char const*name,IReference*r,int type, double defaultValue,char const*varName);
	void addMessageBoxOptionalImport(char const*name,IReference&r,int type, double defaultValue,char const*varName){addMessageBoxOptionalImport(name,&r,type,defaultValue,varName);}
	/**\~french Retourne l'lment de nom name et de type type. Si type est NULL, recherche une variable\~english Returns the element named `name` and of type `type`. If `type` is NULL, it searches for a variable.*/
	IVariable*element(char const*name,char const*type)const;
	/**\~french Ajoute une variable retarde, dfinit son type et sa valeur initiale.
	@param name nom de la nouvelle variable
	@param type type de la nouvelle variable parmi TValueType
	@param val valeur initiale de la nouvelle variable (sous forme de double)
	@return la variable construite
	\~english Adds a delayed variable, defines its type and initial value.
	@param name The name of the new variable.
	@param type The type of the new variable from `TValueType`.
	@param val The initial value of the new variable (as a double).
	@return The constructed variable.
	*/
	IVariable*addDelayedVariable(char const*name,TVarType::EVarType type,double val);
	/**\~french Ajoute une variable, dfinit son type et sa valeur initiale.
	@param name nom de la nouvelle variable
	@param type type de la nouvelle variable parmi TValueType
	@param val valeur initiale de la nouvelle variable (sous forme de double)
	@param bReinitialized si vrai, la variable est rinitialise aprs le tirage des transitions
	@return la variable construite
	\~english Adds a variable, defines its type and initial value.
	@param name The name of the new variable.
	@param type The type of the new variable from `TValueType`.
	@param val The initial value of the new variable (as a double).
	@param bReinitialized If true, the variable is reinitialized after the transition sampling.
	@return The constructed variable.
	*/
	IVariable*addVariable(char const*name,TVarType::EVarType type,double val,bool bReinitialized=false);
	//Pour limiter les ambiguits avec addVariable initialise avec un char*
	IVariable*addVariable(char const*name,TVarType::EVarType type,float val,bool bReinitialized=false){return addVariable(name,type,(double)val,bReinitialized);}
	IVariable*addVariable(char const*name,TVarType::EVarType type,int val,bool bReinitialized=false){return addVariable(name,type,(double)val,bReinitialized);}
	IVariable*addVariable(char const*name,TVarType::EVarType type,bool val,bool bReinitialized=false){return addVariable(name,type,(double)val,bReinitialized);}
	/**\~french Ajoute une variable, dfinit son type et sa valeur initiale sous forme d'une chane de caractres.
	@param name nom de la nouvelle variable
	@param type type de la nouvelle variable
	@param val valeur initiale de la nouvelle variable (sous forme de chane de caractres)
	@param bReinitialized si vrai, la variable est rinitialise aprs le tirage des transitions
	@return la variable construite
	\~english Adds a variable, defines its type and initial value as a string.
	@param name The name of the new variable.
	@param type The type of the new variable.
	@param val The initial value of the new variable (as a string).
	@param bReinitialized If true, the variable is reinitialized after the transition sampling.
	@return The constructed variable.
	*/
	IVariable*addVariable(char const*name,TVarType::EVarType type,char const*val,bool bReinitialized=false);
	/**\~french Retourne la variable de nom name (le nom peut tre compos pour accder aux variables des sous composants).\~english Returns the variable named `name` (the name can be composed to access variables of subcomponents).*/
	IVariable*variable(char const*name)const;
	/**\~french Retourne la variable de nom name (le nom peut tre compos pour accder aux variables des sous composants) (obsolte).\~english Returns the variable named `name` (the name can be composed to access variables of subcomponents) (deprecated).*/
	DEPRECATED IVariable*getVariable(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getVariable","variable");return variable(name);}//!<\~french Obsolte\~english Deprecated.
	/**\~french Ajoute une rfrence  des variable.
	@param name nom de la nouvelle rfrence
	@return la rfrence construite
	\~english Adds a reference to variables.  
	@param name The name of the new reference.  
	@return The constructed reference.
	*/
	IReference*addReference(char const*name);
	/**\~french Retourne la rfrence de nom name (le nom peut tre compos pour accder aux rfrences des sous composants).\~english Returns the reference named `name` (the name can be composed to access references of subcomponents).*/
	IReference*reference(char const*name)const;
	/**\~french Retourne la rfrence de nom name (le nom peut tre compos pour accder aux rfrences des sous composants) (obsolte).\~english Returns the reference named `name` (the name can be composed to access references of subcomponents) (deprecated).*/
	DEPRECATED IReference*getReference(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getReference","reference");return reference(name);}//!<\~french Obsolte\~english Deprecated.
	/**\~french Retourne la transition de nom name (le nom peut tre compos pour accder aux transitions des sous composants).\~english Returns the transition named `name` (the name can be composed to access transitions of subcomponents).*/
	ITransition*transition(char const*name)const;
	/**\~french Retourne la transition de nom name (le nom peut tre compos pour accder aux transitions des sous composants) (obsolte).\~english Returns the transition named `name` (the name can be composed to access transitions of subcomponents) (deprecated).*/
	DEPRECATED ITransition*getTransition(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getTransition","transition");return transition(name);}//!<\~french Obsolte\~english Deprecated.
	/**\~french Retourne le bote de message de nom name (le nom peut tre compos pour accder aux botes des sous composants).\~english Returns the message box named `name` (the name can be composed to access message boxes of subcomponents).*/
	IMessageBox*messageBox(char const*name)const;
	/**\~french Retourne le bote de message de nom name (le nom peut tre compos pour accder aux botes des sous composants) (obsolte).\~english Returns the message box named `name` (the name can be composed to access message boxes of subcomponents) (deprecated).*/
	DEPRECATED IMessageBox*getMessageBox(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getMessageBox","messageBox");return messageBox(name);}//!<\~french Obsolte\~english Deprecated.
	/**\~french Ajoute une transition de sortie de l'tat start.
	@param name nom de la nouvelle transition
	@param start nom de l'tat de dpart de la transition
	@return la transition construite
	\~english Adds an outgoing transition from the `start` state.  
	@param name The name of the new transition.  
	@param start The name of the starting state for the transition.  
	@return The constructed transition.
	*/
	ITransition*addTransition(char const*name,char const*start);
	/**\~french Ajoute une transition de sortie de l'tat start.
	@param name nom de la nouvelle transition
	@param start tat de dpart de la transition
	@return la transition construite
	\~english Adds an outgoing transition from the `start` state.  
	@param name The name of the new transition.  
	@param start The starting state of the transition.  
	@return The constructed transition.
	*/
	ITransition*addTransition(char const*name,CState*start);
	/**\~french Ajoute un tat dans un automate.
	@param automatonName nom de l'automate
	@param name nom du nouvel tat
	@param index indice de l'tat (si ngatif, l'indice est gal au nombre d'tats dj dans l'automate)
	@return l'tat construit
	\~english Adds a state to an automaton.
	@param automatonName The name of the automaton.
	@param name The name of the new state.
	@param index The index of the state (if negative, the index is set to the number of states already in the automaton).
	@return The constructed state.
	*/
	IState*addState(char const*automatonName,char const*name,int index);
	/**\~french Ajoute un tat.
	@param name nom du nouvel tat
	@return l'tat construit
	\~english Adds a state.
	@param name The name of the new state.
	@return The constructed state.
	*/
	IState*addState(char const*name);
	/**\~french Retourne l'tat de nom name (le nom peut tre compos pour accder aux tats des sous composants).\~english Returns the state named `name` (the name can be composed to access states of subcomponents).*/
	IState*state(char const*name)const;
	/**\~french Retourne l'tat de nom name (le nom peut tre compos pour accder aux tats des sous composants) (obsolte).\~english Returns the state named `name` (the name can be composed to access states of subcomponents) (deprecated).*/
	DEPRECATED IState*getState(char const*name)const{ILogManager::glLogManager().msgObsolete("CComponent::getState","state");return state(name);}//!<\~french Obsolte\~english Deprecated.
	/**\~french Dfinit l'tat de nom name comme tat initialement actif.\~english Sets the state named `name` as initially active.*/
	void setInitState(char const*name){setInitState(state(name));}
	/**\~french Dfinit l'tat comme tat initialement actif.\~english Sets the state as initially active.*/
	void setInitState(IState*st);
	void setInitState(IState&st){setInitState(&st);}
	/**\~french Retourne vrai si l'tat de nom name est initialement actif.\~english Returns true if the state named `name` is initially active.*/
	bool isInitState(char const*name)const;
	/**\~french Renseigne un paramtre quelconque qui doit tre interprt  partir de la chane de caractres value.
	La mthode doit tre surcharge dans les sous classes.
	\~english Sets any parameter that should be interpreted from the string `value`.
	The method should be overridden in subclasses.
	*/
	virtual void setParameter(const char*name,const char*value);
	/**\~french Retourne les noms des paramtres pris en compte.
	La mthode doit tre surcharge dans les sous classes.
	\~english Returns the names of the parameters taken into account.
	The method should be overridden in subclasses.
	*/
	virtual std::vector<std::string>parameterNames()const;
	/**\~french Retourne la valeur du parametre de nom pname (NULL) si pas de paramtre de ce nom.
	La mthode doit tre surcharge dans les sous classes.
	\~english Returns the value of the parameter named `pname` (NULL if no parameter with this name exists).
	The method should be overridden in subclasses.
	*/
	virtual char const* parameter(const char* pname)const;
	/**\~french Verifie le composant avant simulation\~english Verifies the component before simulation.*/
	bool verify()override;
};

#endif
