/*************************************
*         Copyright 2015 EDF         *
*************************************/
#pragma once
/**\file
\~french Fichier de dfinition de l'interface d'accs au gestionnaire de messages.
\~english Message manager access interface definition file.
*/

#include "Global.h"
#include <stdexcept>
#include <vector>

/**\~french Structure de dfinition de l'numr des messages gr par le gestionnaire de messages\~english Structure for defining the enumeration of messages managed by the message manager.*/
struct TErrorLevel{
	/**\~french Type numr des messages gr par le gestionnaire de messages\~english Enumerated type of messages managed by the message manager.*/
	typedef enum EErrorLevel{
		info=0,//!<\~french Messages d'information\~english Informational messages
		warning=1,//!<\~french Messages d'avertissement de niveau faible\~english Low-level warning messages
		warning_last=99,//!<\~french Messages d'avertissement de niveau lev\~english High-level warning messages
		error//!<\~french Messages d'erreur\~english Error messages
	}EErrorLevel;
};

/**\~french Structure de dfinition de l'numr d'encodage des message\~english Structure for defining the enumeration of message encoding.*/
struct TEncoding{
	/**\~french Type numr d'encodage des message\~english Enumerated type of message encoding.*/
	typedef enum EEncoding{
		unknown=-1,//!<\~french Encodage indtermin\~english Indeterminate encoding
		latin1=0,//!<\~french Encodage standard\~english Standard encoding
		console,//!<\~french Encodage console Windows\~english Windows console encoding
		UTF8//!<UTF8
	}EEncoding;
};

/**\~french Interface de base du gestionnaire de messages.
Cette classe est implmente sous forme d'un singleton par la classe CLogManager.
Le gestionnaire de message sert  enregistrer et mettre en forme les diffrents messages d'erreur ou d'information.
Il peut ventuellement construire un fichier de log.
De nombreux composants de PyCATSHOO permettent d'afficher des informations de trace.
Ces informations sont envoyes dans le flux obtenu par trTrace.
Ce flux peut tre modifi par setTraceFile ou setTraceStream.
\~english Basic interface of the message manager.
This class is implemented as a singleton by the CLogManager class.
The message manager is used to record and format various error or informational messages.
It can optionally construct a log file.
Many components of PyCATSHOO allow for the display of trace information.
These messages are sent to the stream obtained by trTrace.
This stream can be modified using setTraceFile or setTraceStream.
*/
class PYC_PUBLIC ILogManager
{
protected:
	static ILogManager*st_LogManager;
	static std::ostream*st_TrStream;//Stream d'criture de la trace
	static std::ostream*st_LogStream;//Stream d'criture des messages
	static std::ostream*st_NullStream;//Stream d'absorbtion
	ILogManager(ILogManager const&);//Interdit
	ILogManager const&operator=(ILogManager const&);//Interdit
	static void (*st_Logger)(void*mi,TErrorLevel::EErrorLevel,char const*);//Fonction de log si on constitue un FMU
	static void*st_ModelInstance;//Adresse de la structure dfinissant le FMU
public:
	typedef enum{
		tm_none=0,//Aucune option
		tm_print=1,//Demande d'impression console
		tm_info=2,//Demande d'impression message
		tm_trace=4,//Demande de restitution dans le flux de trace
		tm_store=8,//Demande de stockage
		tm_all=15,
		tm_mask
	}ETmOption;
	ILogManager(){}
	virtual ~ILogManager(){}
	/*\~french Classe de gestion des contextes d'mission des messages.
	La construction d'un objet de cette classe incrmente le contexte des messages
	La destruction de l'objet dcrmente le contexte.
	\~english Class for managing message emission contexts.
	Creating an object of this class increments the message context.
	Destroying the object decrements the context.
	*/
	class PYC_PUBLIC CCtx{
		CCtx(CCtx const&);//Interdit
		CCtx const&operator=(CCtx const&);//Interdit
	public:
		/**\~french Construit un nouveau contexte de message de texte ctx\~english Constructs a new text message context ctx.*/
		explicit CCtx(char const*ctx){ILogManager::glLogManager().pushContext(ctx);}
		~CCtx(){ILogManager::glLogManager().popContext();}
		/**\~french Change le contexte de message courant et le remplace par ctx\~english Changes the current message context and replaces it with ctx.*/
		void changeContext(char const*ctx){ILogManager::glLogManager().popContext();ILogManager::glLogManager().pushContext(ctx);}
	};
	/**\~french Dclare le systme comme FMU et fournit sont instance (mi) et la fonction de log\~english Declares the system as an FMU and provides its instance (mi) and the log function.*/
	static void setFMULogger(void*mi,void(*logger)(void*,TErrorLevel::EErrorLevel,char const*));
	/**\~french Retourne vrai si le systme est un FMU\~english Returns true if the system is an FMU.*/
	static bool isFMU(){return st_Logger!=NULL;}
	/**\~french Retourne l'unique gestionnaire de message global\~english Returns the single global message manager.*/
	static ILogManager&glLogManager();
	/**\~french Encode la chane str suppose en Latin1\~english Encodes the string str as Latin1.*/
	static std::string encode(char const* str, TEncoding::EEncoding enc);
	/**\~french Modifie le flux d'criture de la trace (si NULL on reprend cout)\~english Modifies the trace output stream (if NULL, it reverts to cout).*/
	static void setTraceFile(char const*file);
	/**\~french Modifie le flux d'criture de la trace\~english Modifies the trace output stream.*/
	static void setTraceStream(std::ostream* trStream);
	/**\~french Retourne le flux d'criture de la trace\~english Returns the trace output stream.*/
	static std::ostream& trStream();
	/**\~french Modifie le flux d'criture des messages (Si NULL, les messages sont enregistrs)\~english Modifies the message output stream (if NULL, the messages are logged).*/
	static void setLogFile(char const*file);
	/**\~french Modifie le flux d'criture des messages (Si NULL, les messages sont enregistrs)\~english Modifies the message output stream (if NULL, the messages are logged).*/
	static void setLogStream(std::ostream* logStream);
	/**\~french Retourne le flux d'criture des message (Si NULL, les messages sont enregistrs)\~english Returns the message output stream (if NULL, the messages are logged).*/
	static std::ostream&logStream(){return*st_LogStream;}
	/**\~french Modifie la gestion des messages (enregistrement ou restitution sur std::cout (par dfaut)\~english Modifies the message handling (logging or output to std::cout by default).*/
	static void recordMessages(bool bRecord);
	/**\~french Modifie le niveau de restitution locale des messages (0 : infos, 1 - 99 : warnings, 100 : erreurs).\~english Modifies the local message output level (0: info, 1 - 99: warnings, 100: errors).*/
	virtual void setDisplayLevel(int level)=0;
	/**\~french Retourne le niveau de restitution locale des messages.\~english Returns the local message output level.*/
	virtual int displayLevel()const=0;
	/**\~french Enregistre une erreur systme de code ErrorCode. L'erreur est transforme en texte.\~english Logs a system error with the code ErrorCode. The error is converted to text.*/
	virtual const char *sysError(unsigned long ErrorCode=0)=0;
	/**\~french Enregistre un message d'erreur\~english Logs an error message.*/
	virtual TErrorLevel::EErrorLevel msgError(char const*msg)=0;
	/**\~french Lance une exception de message msg\~english Throws a message exception msg.*/
	virtual void throwError(char const*msg=0,bool bThrow=true)=0;
	/**\~french Lance une exception avec le message de la dernire erreur s'il y en a un\~english Throws an exception with the message of the last error, if there is one.*/
	virtual void throwLastError()=0;
	/**\~french enregistre un message d'avertissement de niveau level\~english Logs a warning message of level level.*/
	virtual TErrorLevel::EErrorLevel msgWarning(char const*msg,TErrorLevel::EErrorLevel level=TErrorLevel::warning)=0;
	/**\~french Enregistre un message d'information\~english Logs an informational message.*/
	virtual TErrorLevel::EErrorLevel msgInfo(char const*msg)=0;
	/**\~french Enregistre un message d'obsolescence\~english Logs a deprecation message.*/
	virtual void msgObsolete(char const*oldName,char const*newName)=0;
	/**\~french Vide l'enregistrement de tous les messages et contextes\~english Clears the log of all messages and contexts.*/
	virtual void clear()=0;
	/**\~french Vide l'enregistrement de tous les messages\~english Clears the log of all messages.*/
	virtual void clearMessages()=0;
	/**\~french Incrmente le contexte d'mission des messages avec le contexte ctx.\~english Increments the message emission context with the context ctx.*/
	virtual int pushContext(char const*ctx)=0;
	/**\~french Dcrmente le contexte d'mission des messages\~english Decrements the message emission context.*/
	virtual int popContext(int toLevel=-1)=0;
	/**\~french Retourne le nombre d'rreurs enregistres\~english Returns the number of errors logged.*/
	virtual int nbErrors()const=0;
	/**\~french Modifie la valeur de l'encodage souhait\~english Modifies the value of the desired encoding.*/
	virtual void setExcEncoding(TEncoding::EEncoding enc)=0;
	/**\~french Retourne l'encodage courant\~english Returns the current encoding.*/
	virtual TEncoding::EEncoding excEncoding()const=0;
	/**\~french Retourne une chane de caractres contenant l'ensemble des messages enregistrs\~english Returns a string containing all the logged messages.*/
	virtual char const* messages(TEncoding::EEncoding enc)const=0;
	char const*UTF8Messages()const{return messages(TEncoding::UTF8);}
	/**\~french Imprime les messages enregistrs sur la sortie standard\~english Prints the logged messages to the standard output.*/
	virtual void printMessages()const=0;
	/**\~french Imprime le texte dans le flux de trace sauf si la trace est inhibe\~english Prints the text to the trace stream unless the trace is inhibited.*/
	virtual void trace(char const*text)const=0;
	/**\~french Retourne le temps cpu stock s'il existe, 0 sinon\~english Returns the stored CPU time if it exists, 0 otherwise.*/
	virtual unsigned long long cpuTime()const=0;
	/**\~french Retourne le temps cpu actuel en millisecondes depuis initTime.
	@param initTime instant initial de mesure du temps cpu
	@param options gre l'impression du temps cpu depuis initTime (=0 pas d'impression, =1 impression console, =2 impression message d'info, =4 stockage)
	@return l'instant cpu en millisecondes
	\~english Returns the current CPU time in milliseconds since initTime.  
	@param initTime initial instant for measuring CPU time  
	@param options controls the printing of CPU time since initTime (=0 no printing, =1 console printing, =2 informational message printing, =4 storage)  
	@return the CPU time in milliseconds
	*/
	virtual unsigned long long cpuTime(unsigned long long initTime,int options)=0;
	/**\~french Retourne le temps d'attente stock s'il existe, 0 sinon\~english Returns the stored wait time if it exists, 0 otherwise.*/
	virtual unsigned long long waitTime()const=0;
	/**\~french Retourne le temps d'attente en millisecondes depuis initTime.
	@param initTime date initiale de mesure du temps d'attente
	@param options gre l'impression du temps d'attente depuis initTime (=0 pas d'impression, =1 impression console, =2 impression message d'info, =4 impression dans le flux de trace, =8 stockage)
	@return la date actuelle en millisecondes
	\~english Returns the wait time in milliseconds since initTime.  
	@param initTime initial date for measuring the wait time  
	@param options controls the printing of the wait time since initTime (=0 no printing, =1 console printing, =2 informational message printing, =4 printing to trace stream, =8 storage)  
	@return the current date in milliseconds
	*/
	virtual unsigned long long waitTime(unsigned long long initTime,int options)=0;
	/**\~french Retourne vrai si l'exception est une exception lance par ce gestionnaire de messages\~english Returns true if the exception is an exception thrown by this message manager.*/
	virtual bool isLogManagerException(std::exception const&exc)const=0;
	/**\~french Retourne la version de la bibliothque Pycatshoo\~english Returns the version of the Pycatshoo library.*/
	virtual char const*version()const=0;
	/**\~french Contrle la correspondance des versions. Retourne faux et emet un message d'erreur s'il n'y a pas correspondance\~english Controls the version match. Returns false and logs an error message if there is no match.*/
	virtual bool checkVersionFor(char const*name, char const*version)=0;
	/**\~french Retourne le copyright de la bibliothque Pycatshoo\~english Returns the copyright of the Pycatshoo library.*/
	virtual char const*copyright()const=0;
	/**\~french Inhibe ou deshinibe l'affichage de la trace.\~english Enables or disables the trace output.*/
	virtual void inhibateTrace(bool inhib=true)=0;
	/**\~french Retourne l'tat d'inhibition de la trace.\~english Returns the trace inhibition status.*/
	virtual bool traceInhibited()const=0;
	/**\~french Inhibe ou deshinibe l'affichage des messages d'obsolescence.\~english Enables or disables the display of deprecation messages.*/
	virtual void inhibateObsolete(bool inhib=true)=0;
	/**\~french Retourne l'tat d'inhibition des messages d'obsolescence.\~english Returns the deprecation message inhibition status.*/
	virtual bool obsoleteInhibited()const=0;
};

PYC_PUBLIC std::string formatMsg(char const*format,...);
PYC_PUBLIC std::string formatMsgV(char const*format,va_list args);
PYC_PUBLIC std::vector<std::string> splitStr(char const* text, char sep);
PYC_PUBLIC std::string replaceAll(std::string str, std::string const& from, std::string const& to);
PYC_PUBLIC ILogManager& glLogManager();

