/*************************************
*         Copyright 2015 EDF         *
*************************************/
#pragma once
/**\file
\~french Fichier de dfinition de l'interface de manipulation des gnrateurs de nombres pseudo alatoires.
\~english Definition file for the interface for manipulating pseudo-random number generators.
*/

#include "Global.h"
#include <iosfwd>

/**\~french Interface d'accs  un gnrateurs de nombres pseudo alatoires.
Il est possible de demander la construction de plusieurs gnrateurs alatoires grce  newRNG, mais c'est dconseill.
Il est prfrable d'utiliseer un unique gnrateur en prcisant ventuellement son type gce  l'appel de CSystem::setRNG.

De mme, il est prfrable d'viter d'appeler explicitement le gnrateur alatoire afin de permettre l'utilisation de l'exploration systmatique de l'arbre des squences.

Cette classe permet d'obtenir des nombres pseudo alatoires sous plusieurs formes :
- rpartis uniformment entre 0 et 1
- rpartis suivant des lois classiques (normale, Weibull, ...).
Certains gnrateurs alatoires (yarn5) permettent d'effectuer des sauts d'un nombre dtermins de pas de gnration (jump).
\~english Interface for accessing pseudo-random number generators.
It is possible to request the creation of multiple random generators using newRNG, but this is not recommended.
It is preferable to use a single generator, optionally specifying its type through the call to CSystem::setRNG.

Similarly, it is better to avoid explicitly calling the random generator to enable the use of systematic sequence tree exploration.

This class allows obtaining pseudo-random numbers in several forms:
- uniformly distributed between 0 and 1
- distributed according to classical laws (normal, Weibull, ...).
Some random generators (such as yarn5) allow for jumps of a determined number of generation steps (jump).
*/
class PYC_PUBLIC IRNG
{
	IRNG(IRNG const&);//Interdit
	IRNG const&operator=(IRNG const&);//Interdit
protected:
	unsigned int max_test;//!<\~french Nombre max de test pour les distributions tronques\~english Maximum number of tests for truncated distributions.
public:
	/**\~french Construit une interface de gnration de nombres pseudo alatoires\~english Constructs an interface for generating pseudo-random numbers.*/
	IRNG():max_test(1000){}
	virtual ~IRNG(){}
	/**\~french Cre un gnrateur de type name.
	La destruction du gnrateur doit tre effectue par l'utilisateur.
	@param name nom du gnrateur parmi (KISS, yarn5 et mt19937)
	\~english Creates a generator of type name.  
	The destruction of the generator must be performed by the user.  
	@param name name of the generator, chosen from (KISS, yarn5, and mt19937).
	*/
	static IRNG*newRng(const char*name);
	/**\~french Retourne le nom du gnrateur.\~english Returns the name of the generator.*/
	virtual char const*name()const=0;
	/**\~french Modifie la demande de trace du gnrateur\~english Modifies the trace request for the generator.*/
	virtual void setTrace(int trace){}
	/**\~french Retourne la demande de trace du gnrateur\~english Returns the trace request for the generator.*/
	virtual int trace()const{return 0;}
	/**\~french Modifie la graine du gnrateur\~english Modifies the seed of the generator.*/
	virtual void srand(unsigned int the_seed)=0;
	/**\~french Retourne un nombre alatoire dans l'intervalle [0,1]\~english Returns a random number in the interval [0, 1].*/
	virtual double drand()=0;
	/**\~french Retourne un nombre alatoire dans l'intervalle [0,1[\~english Returns a random number in the interval [0, 1[.*/
	virtual double drand_1()=0;
	/**\~french Retourne un nombre alatoire dans l'intervalle ]0,1]\~english Returns a random number in the interval ]0, 1].*/
	virtual double drand_0()=0;
	/**\~french Retourne un nombre alatoire dans l'intervalle ]0,1[\~english Returns a random number in the interval ]0, 1[.*/
	virtual double drand_0_1()=0;
	/**\~french Retourne un nombre entier alatoire dans l'intervalle [0,N]\~english Returns a random integer in the interval [0, N].*/
	virtual size_t irand(size_t N);
	/**\~french Retourne un nombre alatoire respectant une loi exponentielle de taux lambda\~english Returns a random number following an exponential distribution with rate lambda.*/
	virtual double exponential(double lambda);
	/**\~french Retourne un nombre alatoire respectant une loi normale de moyenne mu et d'cart type sigma\~english Returns a random number following a normal distribution with mean mu and standard deviation sigma.*/
	virtual double normal(double mu,double sigma);
	/**\~french Retourne un nombre alatoire respectant une loi normale de moyenne mu et d'cart type sigma tronque  l'intervalle [min,max]\~english Returns a random number following a normal distribution with mean mu and standard deviation sigma, truncated to the interval [min, max].*/
	virtual double normal(double mu,double sigma,double min,double max);
	/**\~french Retourne un nombre alatoire respectant une loi lognormale de moyenne mu et d'cart type sigma\~english Returns a random number following a lognormal distribution with mean mu and standard deviation sigma.*/
	virtual double lognormal(double mu,double sigma);
	/**\~french Retourne un nombre alatoire respectant une loi loguniforme sur l'intervalle [min,max]\~english Returns a random number following a loguniform distribution over the interval [min, max].*/
	virtual double loguniform(double min, double max);
	/**\~french Retourne un nombre alatoire respectant une loi uniforme sur l'intervalle [min,max]\~english Returns a random number following a uniform distribution over the interval [min, max].*/
	virtual double uniform(double min, double max);
	/**\~french Retourne un nombre alatoire respectant une loi triangulaire sur l'intervalle [min,max] avec un maximum du triangle en interm\~english Returns a random number following a triangular distribution over the interval [min, max] with the triangle's peak at interm.*/
	virtual double triangle(double min,double interm,double max);
	/**\~french Retourne un nombre alatoire respectant un loi gamma de paramtres alpha et beta\~english Returns a random number following a gamma distribution with parameters alpha and beta.*/
	virtual double gamma(double alpha,double beta);
	/**\~french Retourne un nombre alatoire respectant un loi beta de paramtres p, q, a et b\~english Returns a random number following a beta distribution with parameters p, q, a, and b.*/
	virtual double beta(double p,double q,double a,double b);
	/**\~french Retourne un nombre alatoire respectant un loi de Gumbel de paramtres mu et sigma\~english Returns a random number following a Gumbel distribution with parameters mu and sigma.*/
	virtual double gumbel(double mu,double sigma);
	/**\~french Retourne un nombre alatoire respectant un loi de Frechet de paramtres alpha et beta\~english Returns a random number following a Frchet distribution with parameters alpha and beta.*/
	virtual double frechet(double alpha,double beta);
	/**\~french Retourne un nombre alatoire respectant un loi de Pareto de paramtres gamma, theta\~english Returns a random number following a Pareto distribution with parameters gamma and theta.*/
	virtual double pareto(double gamma,double theta);
	/**\~french Retourne un nombre alatoire respectant un loi de Weibull de paramtres theta et beta\~english Returns a random number following a Weibull distribution with parameters theta and beta.*/
	virtual double weibull(double theta,double beta);
	/**\~french Retourne vrai si le gnrateur permet d'effectuer rapidement des sauts de N gnrations de nombres\~english Returns true if the generator allows for fast jumps of N generations of numbers.*/
	virtual unsigned int can_jump()const=0;
	/**\~french Effectue un saut de size gnrations de nombres\~english Performs a jump of size generations of numbers.*/
	virtual void jump(unsigned long long size)=0;
	/**\~french Fixe la taille des blocs de gnration de nombres (ne fait rien par dfaut)\~english Sets the size of the number generation blocks (does nothing by default).*/
	virtual void set_bloc_size(unsigned int size){};//Ne fait rien par dfaut
	/**\~french Effectue un saut de gnrations de nombres de manire  atteindre le dbut du bloc suivant\~english Performs a jump of number generations to reach the beginning of the next block.*/
	virtual void jump(){};//Ne fait rien par dfaut
	/**\~french Retourne la taille mmoire de l'tat du gnrateur\~english Returns the memory size of the generator's state.*/
	virtual int statusSize()const=0;
	/**\~french Rcupre l'tat du gnrateur (st doit avoir la taille suffisante pour stocker l'tat : pas de contrle)\~english Retrieves the state of the generator (st must have sufficient size to store the state: no check is performed).*/
	virtual void status(char*st)const=0;
	/**\~french Rtablit l'tat du gnrateur\~english Restores the state of the generator.*/
	virtual void setStatus(char const*st)=0;
	/**\~french Ecrit l'tat binaire du gnrateur dans le flux\~english Writes the binary state of the generator to the stream.*/
	virtual void writeStatus(char const*st,std::ostream&os)const=0;
	/**\~french Ecrit l'tat du gnrateur dans le flux\~english Writes the state of the generator to the stream.*/
	virtual void writeStatus(std::ostream&os)const=0;
	/**\~french Lit l'tat du gnrateur dans le flux\~english Reads the state of the generator from the stream.*/
	virtual void readStatus(std::istream&is)=0;
};
