/*************************************
*         Copyright 2015 EDF         *
*************************************/
#ifndef INTERFACES_H
#define INTERFACES_H
#include "Version.h"
/**\def VER_FILEVERSION_STR pour assurer la synchronisation de la version de documentation */
#define VER_FILEVERSION_STR "1.4.1.0"

/**\file
* \~french Fichier de dfinition des principales interfaces de Pycatshoo.
* 
* Ce fichier ne contient pas les dfinitions de CSystem, CComponent, ILogManager et IRNG qui sont dfinis respectivement dans System.h, Component.h, ILogManager.h et IRNG.h
* \~english Definition file for the main Pycatshoo interfaces.
*
* This file does not contain the definitions of CSystem, CComponent, ILogManager and IRNG which are defined in System.h, Component.h, ILogManager.h and IRNG.h respectively
*/
/**\mainpage
\~french
La prsente documentation correspond  la version 1.4.1.0

Les mthodes classiques dvaluation probabiliste de sret (EPS), souvent utilises dans le domaine nuclaire,
mme lorsquelles dpassent les approches statiques, restent, pour des raisons historiques, confines dans le cadre des
vnements discrets et atteignent rapidement leurs limites lorsquune valuation plus fine de la sret de fonctionnement est ncessaire.

Ces mthodes ngligent une grande partie des informations sur nos systmes.
En particulier, la modlisation des phnomnes physiques, nest pas explicitement prsente dans les EPS et est souvent remplace par des hypothses dites conservatrices.
Pourtant, plusieurs tentatives ont t faites pour remdier  cela. Mais leurs implmentations
sappuient parfois sur un outil ddi  la modlisation  vnements discrets dans lequel sont intruses des fonctionnalits de modlisation continue.
A linverse, le point de dpart peut tre un outil ddi  la
modlisation physique 0D patch avec des fonctionnalits de modlisation  vnements discrets. Dans les deux cas,
les rsultats ne sont pas satisfaisants car leur utilisation pour l'valuation de systmes mme simples
ncessite souvent une modlisation alambique qui peut tre si difficile  maintenir.
PyCATSHOO apporte une solution originale en ce qu'il propose une mthode et un outil o les deux
paradigmes, dterministe continu d'une part et stochastique discret d'autre part,
sont nativement intgrs. Il fournit galement de nombreux outils pratiques qui permettent  l'utilisateur de se sentir  l'aise avec
l'augmentation de la complexit de modlisation due  cette intgration.

PyCATSHOO permet de modliser et d'valuer les performances de systmes dynamiques hybrides complexes.
Ici, performance signifie la plupart du temps fiabilit et le qualificatif "hybride" dsigne la coexistence et l'interaction,
dans un mme modle, de comportements discrets et stochastiques d'une part et de phnomnes physiques dterministes et continus d'autre part.
PyCATSHOO est une bibliothque dynamique dcline en plusieurs versions. Ces diffrentes versions permettent
l'utilisation de PyCATSHOO en langage C++ ou en langage Python, sur les plateformes Windows 64, Linux 64 ou Mac OS.
La bibliothque propose trois types d'outils :
- Des outils qui aident  la modlisation gnrique c'est--dire  la construction de bibliothques de classes qui reprsentent les composants d'une catgorie de systme donne. Ces bibliothques sont galement appeles bases de connaissances.
- Des outils ddis  la modlisation d'un systme spcifique bass sur l'instanciation des classes importes d'une base de connaissances. Ces outils permettent galement de personnaliser et de lancer des simulations parallles.
- Des outils de post-traitement pour aider  l'analyse des rsultats.
PyCATSHOO est bas sur la notion d'abstractions de composants qui sont des classes, drives de CComponent, qui appartiennent  une base de connaissances.
Le modle PyCATSHOO d'un systme doit tre inclus dans une classe drive du CSystem o :
- Les instanciations des classes de la base de connaissances sont ralises en fonction de la composition du systme
- Les composants du systme sont lis entre eux en fonction de son architecture
- La quantification du systme est ralise via une simulation de Monte Carlo
- Les quations des phnomnes physiques sont rsolues

Les restitutions des rsultats de quantification et leur post-traitement peuvent tre faites de deux manires en fonction des objectifs de l'tude.

La premire manire ncessite la dfinition d'indicateurs de performance avant le lancement de la simulation. C'est l'approche recommande car elle ncessite moins de mmoire bien qu'elle ralentisse lgrement les simulations.

La deuxime manire consiste  surveiller certains tats et variables du systme pendant la simulation.
Cela signifie que toutes les valeurs prises par ces lments au cours des squences simules sont mmorises. A la fin de la simulation, on peut utiliser une instance de la classe CAnalyser. Cette classe
fournit des routines de calcul statistique. Elle permet de demander des indicateurs tels que les valeurs moyennes, l'cart type, les temps de sjour, etc.
\~english
This documentation corresponds to version 1.4.1.0

Classic methods of probabilistic safety assessments (PSA), often used in nuclear domain, even
when they go beyond the static approaches, remain, for historical reasons, confined in discrete
events framework and quickly reach their limits when finer dependability assessment is needed.

These methods neglect a large part of information about our systems. In particular, physical phenomena modeling,
are not explicitly present in PSA and are often replaced by so called conservative assumptions.
Yet several attempts have been made to remedy this. But their implementations are sometimes based on a tool dedicated
to the discrete events modeling into which some continuous modeling functionalities are intruded. Conversely, the starting point may be a tool dedicated to the
0D physical modeling patched with some discrete events modeling functionalities. In both cases
the results are not satisfactory because their use for the assessment of even simple systems often
requires some convoluted modeling which may be so difficult to maintain
PyCATSHOO brings an original solution in that it offers a method and a tool where the two
paradigms, continuous deterministic on the one hand and discrete stochastic on the other hand,
are natively integrated. It also provides a lot of convenient tools which make user feel at ease with
the increase of modeling complexity due to this integration.

PyCATSHOO can be used to model and assess the performances of complex hybrid dynamic
systems. Here, performance, most of the time, means dependability and "hybrid" qualifier denotes the coexistence and interaction, in the same model, of discrete and stochastic behavior in the one
hand with deterministic and continuous physical phenomena in the other hand.
PyCATSHOO is a dynamic library released in several versions. These different versions allow the
use of PyCATSHOO in C ++ language or in Python language, on Windows 64, Linux 64 or Mac OS platforms.
The library offers three kinds of tools:
- Tools that help in generic modeling i.e. building libraries of classes that stand for the components of a given system category. These libraries are also called knowledge bases.
- Tools dedicated to the modeling of a specific system based on the instantiation of the classes imported from a knowledge base. These tools also allow to customize and start parallel simulations.
- Post processing tools to help with the analyzing of the results.

PyCATSHOO is based on the notion of component abstractions which are classes, derived from
the CComponent, that belong to a knowledge base. The PyCATSHOO model of a system must be
included in a class derived from CSystem where:
- The instantiations of the knowledge base classes are realized according to the system composition
- The components of the system are linked to each other according to its architecture
- The system quantification is carried out via Monte Carlo simulation
- The equations of the physical phenomena are solved

The restitutions of quantification results and their post processing may be done in two ways according to the goals of the study.

The first way requires the definition of performance indicators before the simulation run. This is the recommended approach as it requires less memory although it slightly slows the simulations.

The second way consists in monitoring some system states and variables during the simulation.
This means that all the values taken by these elements during the simulated sequences are memorized. At the end of the simulation, one can use an instance of the CAnalyser class. This class
provides statistical computational routines. It helps requesting indicators such as mean values,
standard deviation, sojourn times, etc.
*/

#include "ILogManager.h"
#include "NamedComp.h"

#include <limits.h>
#include <vector>
#include <complex>
#include <typeinfo>
#include <map>
#include <memory>

typedef std::complex<double> pycomplex;

PYC_PUBLIC bool operator<(pycomplex const& c1, pycomplex const& c2);

class CSystem;
class CSystemP;
class IVariable;
class ITransition;
class IState;
class IEquation;
class IIndicator;
class IInequality;
class IAutomaton;
class IFunctionalMockupUnit;
class CMessageBox;
class CComponent;
class CVariable;
class CTransition;
class CTreeNode;
class IMDPManager;

/**\~french Interface de base d'un moniteur.
	Cette classe n'est drive que par la classe CSysteme.
	Elle permet aux lments monitors de connatre le moniteur et de lui signaler les changements.
	\~english Basic interface of a monitor.
	This class is only derived by the CSysteme class.
	It allows monitored elements to know the monitor and notify it of changes.
*/
class PYC_PUBLIC IMonitor
{
protected:
	IMonitor(void){}
public:
	/**\~french Demande au moniteur d'enregistrer la valeur de l'lment.\~english Ask the monitor to record the element's value.*/
	virtual void doMonitor(IMonitored const&elt)=0;
	/**\~french Demande d'ajouter ou d'enlever l'lment du monitoring.\~english Request to add or remove the element from monitoring.*/
	virtual void monitor(IMonitored&elt,bool bMonit)=0;
};

/**\~french Interface de base d'un lment pouvant tre suivi par un moniteur.
	Cette classe est spcialise par ITransition, IIndicator et IVariable.
	L'interface dfinit localement la gestion du moniteur et de la trace.
	\~english Basic interface for an element that can be monitored by a monitor.
	This class is specialized by ITransition, IIndicator, and IVariable.
	The interface locally defines the management of the monitor and trace.
*/
class PYC_PUBLIC IMonitored{
	friend class CSystemP;
protected:
	IMonitor*m_Monitor;//!<\~french Moniteur de suivi de l'lment.\~english Monitor tracking the element.
	int m_Trace;//!<\~french Niveau de trace de l'lment (0 = pas de trace, 1 = trace en fin de cycle, 2 = trace  chaque affectation, 4 = trace pendant PDMP).\~english Trace level of the element (0 = no trace, 1 = trace at the end of the cycle, 2 = trace on each assignment, 4 = trace during PDMP).
	bool m_AlwaysMonitored;//!<\~french Si vrai, l'lment est monitor en permanence, sinon, l'lment n'est monitor que lors de ses changement de valeur.\~english If true, the element is continuously monitored; otherwise, the element is only monitored when its value changes.
	int m_Id;//!<\~french Id d'identification de l'lment monitor (dfini de manire unique (si ngatif : fourni et non modifiable)).\~english Identification ID of the monitored element (uniquely defined, if negative: provided and not modifiable).
	IMonitored():m_Monitor(0),m_Trace(0),m_AlwaysMonitored(false),m_Id(0){}
	virtual~IMonitored(){}
public:
	/**\~french Retourne l'Id de l'lment monitor.\~english Returns the ID of the monitored element.*/
	unsigned int id()const{return (unsigned)(m_Id>0?m_Id:-m_Id);}
	/**\~french Dsigne le moniteur de suivi de l'lment (si NULL, l'lment n'est pas suivi), renseigne l'id prdfini et modifie le niveau de monitoring.\~english Designates the monitor tracking the element (if NULL, the element is not tracked), sets the predefined ID, and modifies the monitoring level.*/
	virtual void setMonitor(IMonitor*monitor,unsigned id,bool always){if(monitor)monitor->monitor(*this,true);else if(m_Monitor)m_Monitor->monitor(*this,false);m_AlwaysMonitored=(monitor && always);m_Id=monitor?-(int)id:0;}
	/**\~french Retourne le moniteur de suivi de l'lment.\~english Returns the monitor tracking the element.*/
	virtual IMonitor*monitor()const{return m_Monitor;}
	/**\~french Modifie le niveau de trace de l'lment (si 0 pas de trace de l'lment).\~english Modifies the trace level of the element (if 0, no trace of the element).*/
	virtual void setTrace(int trace){m_Trace=trace;}
	/**\~french Retourne le niveau de trace de l'lment.\~english Returns the trace level of the element.*/
	virtual int trace()const{return m_Trace;}
	/**\~french Retourne VRAI si l'lment est monitor.\~english Returns TRUE if the element is monitored.*/
	virtual bool isMonitored()const{return m_Monitor != NULL;}
	/**\~french Modifie l'option de monitoring permanent.\~english Modifies the permanent monitoring option.*/
	void setAlwaysMonitored(bool bAlways){m_AlwaysMonitored=bAlways;}
	/**\~french Retourne vrai si l'lment est monitor en permanence.\~english Returns true if the element is permanently monitored.*/
	int isAlwaysMonitored()const{return m_AlwaysMonitored;}
	/**\~french Retourne le nom simple de l'lment.\~english Returns the simple name of the element.*/
	virtual char const*basename()const{return 0;}
	/**\~french Retourne le nom complet de l'lment (incluant la hirarchie des composants propritaires de l'lment).\~english Returns the full name of the element (including the hierarchy of the element's owning components).*/
	virtual char const*name()const=0;
	/**\~french Retourne le type d'lment.\~english Returns the type of the element.*/
	virtual char const*type()const=0;
	/**\~french Retourne la valeur initiale float de l'lment.\~english Returns the initial float value of the element.*/
	virtual float fInitValue()const=0;
	/**\~french Retourne l'objet parent de l'lment (dans Pycatshoo, il s'agit d'un CComponent).\~english Returns the parent object of the element (in Pycatshoo, this is a CComponent).*/
	virtual CNamedComp*parent()const=0;
	int compare(IMonitored const&other)const;//!<\~french Comparaison en fonction du type et du nom.\~english Comparison based on type and name.
	bool operator<(IMonitored const&other)const;//!<\~french Oprateur de comparaison en fonction du type et du nom.\~english Comparison operator based on type and name.
};

/**\~french Structure de dfinition de l'numr de choix des oprateurs de calcul des indicateurs, des cibles et des filtres de squences.
	\~english Definition structure of the enumerated choice for operators used in calculating indicators, targets, and sequence filters.*/
struct PYC_PUBLIC TOperator{
	/**\~french Enumr de dfinition des oprateurs de calcul des indicateurs, des cibles et des filtres de squences.
	\~english Enumerated definition of operators for calculating indicators, targets, and sequence filters.*/
	typedef enum EOp{
		unknown,//!<\~french Inconnu\~english Unknown
		ne,//!<\~french Diffrent\~english Different
		eq,//!<\~french Egal\~english Equal
		lt,//!<\~french Strictement infrieur\~english Strictly less than
		gt,//!<\~french Strictement suprieur\~english Strictly greater than
		le,//!<\~french Infrieur ou gal\~english Less than or equal to
		ge//!<\~french Suprieur ou gal\~english Greater than or equal to
	}EOp;
	static char const*opToStr(EOp op);//!<\~french Convertit le type de l'oprateur en chane de caractres.\~english Converts the operator type to a string.
	static EOp strToOp(char const*str);//!<\~french Lit le type de l'oprateur  partir d'une chane au format du fichier XML de paramtres.\~english Reads the operator type from a string in the XML parameter file format.
	static bool compare(float val1,EOp op,float val2);
	static bool compare(double val1,EOp op,double val2);
};

/**\~french Interface des foncteurs condition avec argument rel.
	L'interface est utilise dans les analyses occurrences(), realized(), residenceTime().
	Les mthodes statiques newCondFctFloat permettent de construire des implmentations de ces interfaces pour diffrents cas simples.
	\~english Interface for conditional functors with a real argument.
	The interface is used in the occurrences(), realized(), and residenceTime() analyses.
	The static methods newCondFctFloat allow the construction of implementations of these interfaces for various simple cases.
*/
struct PYC_PUBLIC ICondFctFloat{
	virtual ~ICondFctFloat(){}
	/**\~french Oprateur de test : retourne vrai si la condition est ralise.\~english Test operator: returns true if the condition is met.*/
	virtual bool operator()(float val)const{return true;}
	virtual float limit()const{return 0;}
	virtual TOperator::EOp op()const{return TOperator::unknown;}
	static ICondFctFloat*newCondFctFloat(bool (*condition)(float));
	static ICondFctFloat*newCondFctFloat(char const*op, float value);
	static ICondFctFloat*newCondFctFloat(TOperator::EOp op, float value);
};

/**\~french Interface d'accs  une branche de mmorisation de squence.
La branche contient la date de mmorisation, l'indice de branche a la date et les valeurs mmorises pour chaque lment monitor  cette occasion.
\~english Interface for accessing a sequence memory branch.
The branch contains the memory date, the branch index at the date, and the values stored for each monitored element on that occasion.
*/
class PYC_PUBLIC IBranch{
protected:
	IBranch(){}
	virtual~IBranch(){}
public:
	virtual PyTime time()const=0;//!<\~french Retourne l'instant du pas de temps courant.\~english Returns the current time step instant.
	virtual unsigned index()const=0;//!<\~french Retourne l'indice de branche pour le pas de temps courant.\~english Returns the branch index for the current time step.
	virtual float value(IMonitored const*obj,bool*exists=NULL)const=0;//!<\~french Retourne la valeur de l'lment obj  l'instant courant s'il existe.\~english Returns the value of the element obj at the current instant, if it exists.
	virtual std::vector<IMonitored const*>monitoredElts()const=0;//!<\~french Retourne la liste des lments monitors  l'instant courant.\~english Returns the list of monitored elements at the current instant.
};

/**\~french Interface d'accs  l'itrateur de squence.
L'itrateur est obtenu en interrogeant la squence  parcourir:
\~english Interface for accessing the sequence iterator.
The iterator is obtained by querying the sequence to be traversed:
\~
~~~~{.cpp}
ISequence*seq;
for(ISeqIter it=seq->iterator();!it.atEnd();it.next())
    IBranch*br = it.branch();
~~~~*/
class PYC_PUBLIC ISeqIter{
public:
	virtual~ISeqIter(){}
	virtual void restart()=0;//!<\~french Replace l'itrateur en dbut de squence.\~english Resets the iterator to the beginning of the sequence.
	virtual bool atEnd()const=0;//!<\~french Retourne vrai si l'itrateur a atteint la fin de la squence.\~english Returns true if the iterator has reached the end of the sequence.
	virtual bool atBegin()const=0;//!<\~french Retourne vrai si l'itrateur est en dbut de la squence.\~english Returns true if the iterator is at the beginning of the sequence.
	virtual void next()=0;//!<\~french Avance au pas d'enregistrement suivant (le temps peut ne pas changer).\~english Moves to the next recorded step (the time may not change).
	virtual void previous()=0;//!<\~french Recule au pas d'enregistrement prcdent (le temps peut ne pas changer).\~english Moves back to the previous recorded step (the time may not change).
	virtual IBranch const*branch()const=0;//!<\~french Retourne la branche courante.\~english Returns the current branch.
	virtual PyTime time()const{return branch()->time();}//!<\~french Retourne l'instant du pas de temps courant.\~english Returns the current time step instant.
	virtual unsigned index()const{return branch()->index();}//!<\~french Retourne l'indice de branche pour le pas de temps courant.\~english Returns the branch index for the current time step.
	virtual float value(IMonitored const*obj,bool*exists=NULL)const=0;//!<\~french Retourne la valeur de l'lment obj  l'instant courant s'il existe.\~english Returns the value of the element obj at the current instant, if it exists.
	virtual std::vector<IMonitored const*>monitoredElts()const{return !atEnd()?branch()->monitoredElts():std::vector<IMonitored const*>();}//!<\~french Retourne la liste des lments monitors  l'instant courant.\~english Returns the list of monitored elements at the current instant.
};

/**\~french Interface d'accs  une squence.
La squence permet d'accder aux valeurs des lments monitors  n'importe quel instant.
Pour suivre le droulement d'une squence il faut utiliser un itrateur (ISeqIter) obtenu avec la mthode iterator().
Une squence donne accs aux lments suivants :
- numro de squence num()
- vraisemblance likelyhood()
- indicateur de conservation : permet de dsigner les squences  poursuivre (toKeep(), setToKeep())
- date de fin de la squence endDate()
- cause d'arrt de la squence endCause()
- nombre de points d'enregistrement branchCount()
- ime point d'enregistrement branch(i)
\~english Interface for accessing a sequence.
The sequence allows access to the values of monitored elements at any instant.
To follow the progression of a sequence, an iterator (ISeqIter) obtained with the method iterator() must be used.
A sequence provides access to the following elements:
 - sequence number num()
 - likelihood likelyhood()
 - preservation indicator: used to designate sequences to be continued (toKeep(), setToKeep())
 - end date of the sequence endDate()
 - cause of sequence termination endCause()
 - number of recorded points branchCount()
 - ith recorded point branch(i)
*/
class PYC_PUBLIC ISequence{
	CSystem const&m_System;
protected:
	ISequence(CSystem const&system):m_System(system){}
	virtual~ISequence(){}
public:
	/**\~french Retourne le systme propritaire.\~english Returns the owning system.*/
	CSystem const&system()const{return m_System;}
	/**\~french Retourne un itrateur sur la squence.\~english Returns an iterator over the sequence.*/
	virtual ISeqIter*iterator()const=0;
	/**\~french Retourne le numro de la portion de squence.\~english Returns the sequence segment number.*/
	virtual SEQ_ID num()const=0;
	/**\~french Retourne la probabilit de la squence.\~english Returns the probability of the sequence.*/
	virtual double probability()const=0;
	/**\~french Retourne la vraisemblance de la squence.\~english Returns the likelihood of the sequence.*/
	virtual double likelyhood()const=0;
	/**\~french Modifie l'indice de conservation de la squence.\~english Modifies the preservation index of the sequence.*/
	virtual void setToKeep(bool toKeep)=0;
	/**\~french Retourne vrai si la squence doit tre conserve.\~english Returns true if the sequence should be preserved.*/
	virtual bool toKeep()const=0;
	/**\~french Ajoute une prolongation  la squence.\~english Adds an extension to the sequence.*/
	virtual void addExtend(ITransition*tr,PyTime date)=0;
	/**\~french Retourne la valeur de l'objet obj  l'instant t.\~english Returns the value of the object obj at instant t.*/
	virtual float value(IMonitored const*obj,PyTime t)const=0;
	/**\~french Calcule les valeurs extrmes de l'lment monitor (min et max doivent tre initialiss).\~english Calculates the extreme values of the monitored element (min and max must be initialized).*/
	virtual void extremeValues(IMonitored const*obj,float&min,float&max)const=0;
	std::vector<float>extremeValues(IMonitored const*obj)const{std::vector<float>v(2,(float)HUGE_VAL);extremeValues(obj,v[0]=(float)-HUGE_VAL,v[1]);return v;}
	/**\~french Calcule les temps de sjour entre les bornes dfinies par lims.\~english Calculates the residence times between the bounds defined by lims.*/
	virtual void residenceTimes(IMonitored const*obj,PyTime it,PyTime ft,std::vector<float>const&lims,std::vector<PyTime>&resTs)const=0;
	std::vector<PyTime>residenceTimes(IMonitored const*obj,PyTime it,PyTime ft,std::vector<float>const&lims)const{std::vector<PyTime>v;residenceTimes(obj,it,ft,lims,v);return v;}
	/**\~french Retourne le temps de sjour satisfaisant la condition.\~english Returns the residence time that satisfies the condition.*/
	virtual float residenceTime(IMonitored const*obj,PyTime it,PyTime ft,ICondFctFloat const&condition)const=0;
	/**\~french Retourne le temps de sjour satisfaisant la condition "valeur(obj) op value".\~english Returns the residence time that satisfies the condition "value(obj) op value."*/
	virtual float residenceTime(IMonitored const*obj,PyTime it,PyTime ft,char const*op,float value)const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(op,value));
		return residenceTime(obj,it,ft,*cff);
	}
	/**\~french Retourne le temps de sjour satisfaisant la condition.\~english Returns the residence time that satisfies the condition.*/
	virtual float residenceTime(IMonitored const*obj,PyTime it,PyTime ft,bool (*condition)(float))const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));
		return residenceTime(obj,it,ft,*cff);
	}
	/**\~french Retourne la date de premire ralisation de la condition.\~english Returns the date of the first occurrence of the condition.*/
	virtual PyTime realized(IMonitored const*obj,ICondFctFloat const&condition)const=0;
	/**\~french Retourne la date de premire ralisation de la condition "valeur(obj) op value".\~english Returns the date of the first occurrence of the condition "value(obj) op value."*/
	virtual PyTime realized(IMonitored const*obj,char const*op,float value)const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(op,value));
		return realized(obj,*cff);
	}
	/**\~french Retourne la date de premire ralisation de la condition.\~english Returns the date of the first occurrence of the condition.*/
	PyTime realized(IMonitored const*obj,bool (*condition)(float))const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));
		return realized(obj,*cff);
	}
	/**\~french Retourne vrai dans real[i] si la condition est satisfaite avant ts[i].\~english Returns true in real[i] if the condition is met before ts[i].*/
	virtual void realized(IMonitored const*obj,ICondFctFloat const&condition,std::vector<PyTime>const&ts,std::vector<bool>&real)const=0;
	/**\~french Retourne vrai dans real[i] si la condition "valeur(obj) op value" est satisfaite avant ts[i].\~english Returns true in real[i] if the condition "value(obj) op value" is met before ts[i].*/
	void realized(IMonitored const*obj,char const*op,float value,std::vector<PyTime>const&ts,std::vector<bool>&real)const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(op,value));
		realized(obj,*cff,ts,real);
	}
	/**\~french Retourne vrai dans real[i] si la condition est satisfaite avant ts[i].\~english Returns true in real[i] if the condition is met before ts[i].*/
	void realized(IMonitored const*obj,bool (*condition)(float),std::vector<PyTime>const&ts,std::vector<bool>&real)const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));
		realized(obj,*cff,ts,real);
	}
	/**\~french Retourne vrai dans return[i] si la condition est satisfaite avant ts[i].\~english Returns true in return[i] if the condition is met before ts[i].*/
	std::vector<bool>realized(IMonitored const*obj,ICondFctFloat const&condition,std::vector<PyTime>const&ts)const{std::vector<bool>v;realized(obj,condition,ts,v);return v;}
	/**\~french Retourne vrai dans return[i] si la condition "valeur(obj) op value" est satisfaite avant ts[i].\~english Returns true in return[i] if the condition "value(obj) op value" is met before ts[i].*/
	std::vector<bool>realized(IMonitored const*obj,char const*op,float value,std::vector<PyTime>const&ts)const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(op,value));
		return realized(obj,*cff,ts);
	}
	/**\~french Retourne vrai dans return[i] si la condition est satisfaite avant ts[i].\~english Returns true in return[i] if the condition is met before ts[i].*/
	std::vector<bool>realized(IMonitored const*obj,bool (*condition)(float),std::vector<PyTime>const&ts)const{
		std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));
		return realized(obj,*cff,ts);
	}
	/**\~french Retourne vrai si la condition est ralise avant le temps t.\~english Returns true if the condition is met before time t.*/
	bool realized(IMonitored const*obj,ICondFctFloat const&condition,PyTime t){std::vector<bool>v;realized(obj,condition,std::vector<PyTime>(1,t),v);return v[0];}
	/**\~french Retourne vrai si la condition "valeur(obj) op value" est ralise avant le temps t.\~english Returns true if the condition "value(obj) op value" is met before time t.*/
	bool realized(IMonitored const*obj,char const*op,float value,PyTime t){std::vector<bool>v;std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(op,value));realized(obj,*cff,std::vector<PyTime>(1,t),v);return v[0];}
	/**\~french Retourne vrai si la condition est ralise avant le temps t.\~english Returns true if the condition is met before time t.*/
	bool realized(IMonitored const*obj,bool (*condition)(float),PyTime t){std::vector<bool>v;realized(obj,condition,std::vector<PyTime>(1,t),v);return v[0];}
	/**\~french Retourne le nombre de branches de la squence.\~english Returns the number of branches in the sequence.*/
	virtual unsigned branchCount()const=0;
	/**\~french Retourne la ime branche de la squence.\~english Returns the ith branch of the sequence.*/
	virtual IBranch const*branch(unsigned i)const=0;
	/**\~french Retourne la date d'arrt de la squence.\~english Returns the end date of the sequence.*/
	virtual PyTime endTime()const=0;
	/**\~french Retourne la cause d'arrt de la squence.\~english Returns the cause of the sequence termination.*/
	virtual char const*endCause()const=0;
	/**\~french Retourne vrai si la cause est cause.\~english Returns true if the cause is cause.*/
	virtual bool isEndCause(char const*cause)const=0;
	/**\~french Retourne l'intgrale de la valeur de l'objet obj entre les temps it et ft.\~english Returns the integral of the value of the object obj between times it and ft.*/
	virtual float integralValue(IMonitored const*obj,PyTime it,PyTime ft)const=0;
	/**\~french Retourne dans values les valeurs de l'objet obj aux instants ts.\~english Returns in values the values of the object obj at the instants ts.*/
	virtual void values(IMonitored const*obj,std::vector<PyTime>const&ts,std::vector<float>&values)const=0;
	/**\~french Retourne les valeurs de l'objet obj aux instants ts.\~english Returns the values of the object obj at the instants ts.*/
	std::vector<float>values(IMonitored const*obj,std::vector<PyTime>const&ts)const{std::vector<float>v;values(obj,ts,v);return v;}
};

/**\~french Interface des expressions textuelles.
Une premire expression est obtenue en parsant un texte (elle n'est pas directement valuable).
Cette premire expression doit tre instancie dans le contexte global du systme ou dans le contexte d'un composant.
La valeur de l'expression instancie peut tre value par dValue()

Les expressions textuelles manipulent des valeurs relles pour lesquelles on considre qu'une valeur non nulle correspond  VRAI et une valeur nulle  FAUX.

Le langage utilise les oprateurs de calcul suivants :
- +, -, *, / : oprateurs arithmtiques
- =, >, <, >=, <=, != : oprateurs de comparaison (le rsultat vaut 1 si la comparaison est vraie et 0 sinon)
- ! : oprateur de ngation boolenne
- - : oprateur de ngation arithmtique
- |, &, : oprateurs boolens ou et et
- ^ : oprateur boolen ou exclusif (attention ! c'est un oprateur binaire et A^B^C n'est pas gal  XOR(A,B,C)).
Le langage propose les oprateurs complexes suivants :
- C?(V1,VN) : oprateur de choix parmi N ; la valeur dpend de la valeur de C. si C appartient  [0,N], alors la valeur retourne est VC, sinon, la valeur retourne est VN
- atleast C(V1,VN) : retourne 1 si au moins C valeurs parmi les VI sont non nulles et 0 sinon
- atmost C(V1,VN) : retourne 1 si au plus C valeurs parmi les VI sont non nulles et 0 sinon

Le langage permet d'accder  un lment Pycatshoo sous les formes suivantes :
- Accs dans l'objet courant (pour les conditions des transitions) :\n 
  `NomVariable`\n 
  Il s'agit ici d'une variable Pycatshoo
- Accs absolu :
  `#NomObjet.NomVariable`
  Il s'agit ici d'une variable Pycatshoo
Pour accder aux tats, automates et rfrences, il faut faire prcder l'accs par l'un des typages suivants :
- VAR : accs  une variable
- AUT : accs  un automate
- ST : accs  un tat
- REF : accs  une rfrence
Un accs typ absolu peut alors prendre la forme suivante :
`AUT:#NomObjet.NomAutomate`

\~english Interface for textual expressions.
An initial expression is obtained by parsing text (it is not directly evaluable).
This initial expression must be instantiated in the global system context or in the context of a component.
The value of the instantiated expression can be evaluated using dValue().

Textual expressions handle real values where a non-zero value is considered TRUE and a zero value is considered FALSE.

The language uses the following operators:
- +, -, *, / : arithmetic operators
- =, >, <, >=, <=, != : comparison operators (the result is 1 if the comparison is true and 0 otherwise)
- ! : boolean negation operator
- - : arithmetic negation operator
- |, & : boolean OR and AND operators
- ^ : exclusive OR operator (note! This is a binary operator, and A^B^C is not equal to XOR(A,B,C)).
The language offers the following complex operators:
- C?(V1,VN) : selection operator among N; the value depends on C. If C is in [0,N], then the returned value is VC; otherwise, VN is returned
- atleast C(V1,VN) : returns 1 if at least C values among the VIs are non-zero, otherwise 0
- atmost C(V1,VN) : returns 1 if at most C values among the VIs are non-zero, otherwise 0

The language allows access to a Pycatshoo element in the following ways:
- Access within the current object (for transition conditions):
  `VariableName`
  This refers to a Pycatshoo variable
- Absolute access:
  `#ObjectName.VariableName`
  This refers to a Pycatshoo variable

To access states, automata, and references, prepend one of the following types:
- VAR : access a variable
- AUT : access an automaton
- ST : access a state
- REF : access a reference

A typed absolute access may take the following form:
`AUT:#ObjectName.AutomatonName`
*/
class IExpression{
public:
	virtual ~IExpression(){}
	virtual bool isUndefined()const=0;
	virtual bool isZero()const=0;
	virtual bool isValue()const=0;
	virtual IVariable*variable()const=0;
	virtual CComponent*component()const=0;
	virtual IExpression*instanciate(CSystem&system)const=0;
	virtual IExpression*instanciate(CComponent&component)const=0;
	virtual double value()const=0;
	static IExpression*parse(char const*text);
};

/**\~french Interface des filtres de squence
Un filtre permet de dsigner les squences  conserver soit lors de la simulation, soit lors de l'analyse.
Une squence est conserve lorsque l'oprateur bool()(seq) retourne vrai
Un filtre peut tre dfini en utilisant les mthode statiques de cration, soit :
- par un pattern textuel imitant les expressions rgulires : newPatternFilter()
- par une condition exprime sur une variable  un instant donn : newConditionFilter()

exemple :
\~english Sequence filter interface
A filter is used to designate sequences to be retained either during simulation or analysis.
A sequence is retained when the boolean operator bool()(seq) returns true.
A filter can be defined using static creation methods, either:
- through a textual pattern resembling regular expressions: newPatternFilter()
- by a condition applied to a variable at a given instant: newConditionFilter()

example:
\~
~~~~{.cpp}
bool filter(ISequence const&seq){
    return !seq.endTime()<80;
}

CAnalyser ana(CSystem:glSystem());
ana.setSeqFilter(filter);
ana.printFilteredSeq("File.xml","Styles.xsd");
~~~~
\~french
Il peut aussi tre construit en implmentant l'interface et en surchargeant l'oprateur () (uniquement en C++).
\~english
It can also be constructed by implementing the interface and overloading the () operator (C++ only).
*/
class PYC_PUBLIC IFilterFct{
	friend class CSystem;
	friend class CSystemP;
	friend class CAnalyser;
public:
	virtual ~IFilterFct(){}
	virtual bool operator()(ISequence const&seq)const=0;//!<\~french Oprateur de filtrage : retourne vrai si la squence doit tre conserve
	virtual char const*pattern()const{return NULL;}//!<\~french Retourne le pattern de dfinition du filtre ou NULL si le filtre est dfini diffremment
	virtual std::vector<std::vector<std::string> >patterns()const{return std::vector<std::vector<std::string> >();}
	virtual PyTime time()const{return 0;}//!<\~french Retourne l'instant de test de valeur si le filtre est une condition exprime sur une variable
	virtual IMonitored const*monitored()const{return NULL;}
	static IFilterFct*newPythonScriptFilter(char const*script);//!<\~french Construit un filtre  partir du corps d'une fonction Python ayant pour parametres self (system) et sequence
	static IFilterFct*newPythonFilter(PyObject*pyObj);//!<\~french Construit un filtre  partir d'une fonction python ayant pour paramtre sequence
	static IFilterFct*newPatternFilter(char const*pattern);//!<\~french Construit un filtre  partir d'un pattern textuel
	static IFilterFct*newConditionFilter(char const*object,PyTime time,char const*op,float val);//!<\~french Construit un filtre testant si la valeur de l'objet  l'instant time vrifie "object op val"
};

/**\~french Structure de dfinition de l'numr de dfinition restitutions d'un indicateur.\~english Definition structure of the enumeration for indicator restitutions.*/
struct PYC_PUBLIC TIndicatorType{
	/**\~french Enumr de dfinition restitutions d'un indicateur.
		Les lments du type peuvent tre additionns pour cumuler plusieurs restitutions.
		\~english Enumeration definition for indicator restitutions.
		Elements of this type can be added to combine multiple restitutions.
	*/
	typedef enum EIndicatorType{
		no_value=0,				//!<\~french Aucune mmorisation\~english No memorization.
		mean_values=1,			//!<\~french Calcul de la moyenne\~english Average calculation.
		std_dev=2,				//!<\~french Calcul de l'cart type\~english Standard deviation calculation.
		all_values=4,			//!<\~french Mmorisation de toutes les valeurs\~english Memorization of all values.
		quantile_le = 8,		//!<\~french Calcul du quantile infrieur (pourcentage  dfinir par IIndicator::setPctQuantileLeValue())\~english Calculation of the lower quantile (percentage to be defined by IIndicator::setPctQuantileLeValue()).
		quantile_gt = 16,		//!<\~french Calcul du quantile suprieur (pourcentage  dfinir par IIndicator::setPctQuantileGtValue())\~english Calculation of the upper quantile (percentage to be defined by IIndicator::setPctQuantileGtValue()).
		distribution=32,		//!<\~french Calcul de la distribution (limites des intervalles de calcul  dfinir par IIndicator::setDistribLimits())\~english Calculation of the distribution (calculation interval limits to be defined by IIndicator::setDistribLimits()).
		max_values = 64,		//!<\~french Mmorisation de la valeur maximale pour chaque chantillon parmi les squences\~english Memorization of the maximum value for each sample among the sequences.
		min_values = 128,		//!<\~french Mmorisation de la valeur minimale pour chaque chantillon parmi les squences\~english Memorization of the minimum value for each sample among the sequences.
		monitor_values=512,		//!<\~french Mmorisation des valeurs dans les squences comme si l'indicateur tait une variable monitore du systme\~english Memorization of values in the sequences as if the indicator were a monitored variable of the system.
		seq_max_values = 1024,	//!<\~french Mmorisation de la valeur maximale obtenue pour chaque squence parmi les valeurs chantillonnes\~english Memorization of the maximum value obtained for each sequence among the sampled values.
		seq_min_values = 2048,	//!<\~french Mmorisation de la valeur minimale obtenue pour chaque squence parmi les valeurs chantillonnes\~english Memorization of the minimum value obtained for each sequence among the sampled values.
		seq_last_values = 4096	//!<\~french Mmorisation de la dernire valeur de chaque squence\~english Memorization of the last value of each sequence.
	}EIndicatorType;
	static char const*st_Names[14];
	static EIndicatorType fromStr(char const*value);//!<\~french Lit le type d'indicateur  partir d'une chane au format du fichier XML de paramtres.\~english Reads the indicator type from a string in the XML parameter file format.
};
/**\~french Structure de dfinition de l'numr de dfinition de la valeur calcule par un indicateur.\~english Definition structure of the enumeration for the value calculated by an indicator.*/
struct PYC_PUBLIC TComputationType{
	/**\~french Enumr de dfinition de la valeur calcule par l'indicateur  partir de la variable dsigne.\~english Enumeration definition of the value calculated by the indicator from the designated variable.*/
	typedef enum EComputationType{
		simple=0,				//!<\~french La valeur de la variable est utilise sans modification\~english The value of the variable is used without modification.
		res_time,				//!<\~french La valeur utilise est le temps pendant lequel la variable a pris une valeur positive\~english The value used is the time during which the variable held a positive value.
		residence_time=res_time,//!<\~french La valeur utilise est le temps pendant lequel la variable a pris une valeur positive\~english The value used is the time during which the variable held a positive value.
		nb_visits,				//!<\~french La valeur utilise est le nombre de fois o la variable a atteint une valeur positive\~english The value used is the number of times the variable reached a positive value.
		realized				//!<\~french La valeur utilise est 0 si la variable n'a jamais t positive et 1 sinon\~english The value used is 0 if the variable was never positive and 1 otherwise.
	}EComputationType;
	static char const*st_Names[6];
	static EComputationType fromStr(char const*value);//!<\~french Lit le type de calcul d'indicateur  partir d'une chane au format du fichier XML de paramtres\~english Reads the indicator calculation type from a string in the XML parameter file format.
};

/**\~french Interface d'acces aux indicateurs.
Un indicateur permet d'enregistrer des donnes statistiques pendant la simulation.
La valeur suivie peut tre dfinie  partir :
- d'une variable dsigne par setVariable()
- d'une mthode d'un des objets du systme ou du systme lui mme (uniquement  partir de CSystem::addIndicator)
La valeur simule peut tre calcule en effectuant une comparaison avec une limite `setCondFct(op, limite)` : "valeur suivie op limite"
La valeur utilise pour les calculs statistiques peut tre calcule  partir de la valeur suivie en fonction du type `TComputationType::EComputationType` setComputation().
Les restitutions statistiques sont prcises  l'aide de `setRestitution()`.

exemple :
\~english Indicator access interface.
An indicator is used to record statistical data during simulation.
The tracked value can be defined from:
- a variable designated by setVariable()
- a method of one of the system objects or the system itself (only through CSystem::addIndicator)

The simulated value can be calculated by performing a comparison with a limit `setCondFct(op, limit)`: "tracked value op limit".
The value used for statistical calculations can be derived from the tracked value based on the type `TComputationType::EComputationType` setComputation().
Statistical restitutions are specified using `setRestitution()`.

example:
\~
~~~~{.cpp}
IIndicator*ind=sys.addIndicator("IndicatorName","VariableName","VAR",NULL,0);
ind->setRestitution(TIndicatorType::std_dev|TIndicatorType::distribution);
ind->setDistribLimits(std::vector<float>({-1.,0.,2.,3.});
~~~~
*/
class PYC_PUBLIC IIndicator:public IMonitored, public CSysNamedComp{
	IIndicator(IIndicator const&);
	IIndicator&operator=(IIndicator const&);
protected:
	IIndicator(char const*name,CSystem&sys);
public:
	char const*basename()const override{return CSysNamedComp::basename();}
	char const*name()const override{return CSysNamedComp::name();}
	CNamedComp*parent()const override{return NULL;}
	virtual void setRestitutions(int type)=0;//!<\~french Modifie les restitutions de l'indicateur : type est un cumul d'lment de TIndicatorType::EIndicatorType\~english Modifies the restitutions of the indicator: type is a combination of elements from TIndicatorType::EIndicatorType.
	virtual int restitutions()const=0;//!<\~french Retourne les restitutions de l'indicateur\~english Returns the restitutions of the indicator.
	virtual void setComputation(TComputationType::EComputationType comp)=0;//!<\~french Modifie le calcul global effectu  partir de la valeur simule\~english Modifies the global calculation performed based on the simulated value.
	virtual TComputationType::EComputationType computation()const=0;//!<\~french Retourne le type de calcul global effectu  partir de la valeur simule\~english Returns the type of global calculation performed based on the simulated value.
	virtual void setPctQuantileLeValue(float fracMin)=0;//!<\~french Modifie le pourcentage de calcul du fractile min\~english Modifies the percentage for calculating the minimum fractile.
	virtual void setPctQuantileGtValue(float fracMax)=0;//!<\~french Modifie le pourcentage de calcul du fractile max\~english Modifies the percentage for calculating the maximum fractile.
	virtual float pctQuantileLeValue()const=0;//!<\~french Retourne le pourcentage de calcul du fractile min\~english Returns the percentage for calculating the minimum fractile.
	virtual float pctQuantileGtValue()const=0;//!<\~french Retourne le pourcentage de calcul du fractile max\~english Returns the percentage for calculating the maximum fractile.
	virtual void setDistribLimits(std::vector<float>lims)=0;//!<\~french Modifie les limites de calcul de la distribution\~english Modifies the calculation limits of the distribution.
	virtual void setVariable(IVariable*var)=0;//!<\~french Dfinit la variable suivie par l'indicateur\~english Defines the variable tracked by the indicator.
	virtual IVariable const*variable()const=0;//!<\~french Retourne la variable  partir de laquelle l'indicateur est calcul si elle existe\~english Returns the variable from which the indicator is calculated, if it exists.
	virtual void setMethod(float (CSystem::*fct)())=0;//!<\~french Dfinit la mthode systme utilise pour calculer l'indicateur\~english Defines the system method used to calculate the indicator.
	virtual void setMethod(double (CSystem::*fct)())=0;//!<\~french Dfinit la mthode systme utilise pour calculer l'indicateur\~english Defines the system method used to calculate the indicator.
	virtual void setMethod(PyObject*pyObj)=0;//!<\~french Dfinit la mthode Python utilise pour calculer l'indicateur\~english Defines the Python method used to calculate the indicator.
	virtual char const*pythonScript()const=0;//!<\~french Retourne le corps de la fonction Python\~english Returns the body of the Python function.
	virtual void setPythonScript(char const*script)=0;//!<\~french Dfinit le corps de la fonction Python utilise pour calculer l'indicateur\~english Defines the body of the Python function used to calculate the indicator.
	virtual char const*expression()const=0;//!<\~french Retourne l'expression textuelle de calcul de la cible\~english Returns the textual expression for calculating the target.
	virtual void setExpression(char const*expr)=0;//!<\~french Dfinit l'expression textuelle de calcul de la cible (si pas d'oprateur, cible atteinte si valeur non nulle)\~english Defines the textual expression for calculating the target (if no operator, target is achieved if the value is non-zero).
	virtual void setCondFct(char const*op,float value)=0;//!<\~french Dfinit une comparaison de la valeur courante avec un rel\~english Defines a comparison of the current value with a real number.
	virtual TOperator::EOp op()const=0;//!<\~french Retourne l'oprateur de la comparaison\~english Returns the comparison operator.
	virtual char const*opStr()const=0;//!<\~french Retourne l'oprateur de la comparaison sous forme textuelle\~english Returns the comparison operator as a text string.
	virtual float limit()const=0;//!<\~french Retourne la limite de la comparaison\~english Returns the comparison limit.
	virtual std::vector<float>distribLimits()const=0;//!<\~french Retourne les limites de calcul de la distribution\~english Returns the calculation limits of the distribution.
	virtual float fValue()const=0;//!<\~french Retourne la valeur courante calcule\~english Returns the current calculated value.
	virtual std::vector<float>const&values(int i)const=0;//!<\~french Retourne toutes les ime valeurs\~english Returns all ith values.
	virtual std::vector<float>seqValues(SEQ_ID i)const = 0;//!<\~french Retourne toutes les valeurs de la ime squence\~english Returns all values of the ith sequence.
	virtual float mean(int i)const=0;//!<\~french Retourne la ime moyenne\~english Returns the ith average.
	virtual float stdDev(int i)const=0;//!<\~french Retourne le ime cart type\~english Returns the ith standard deviation.
	virtual float quantileLe(int i)const=0;//!<\~french Retourne le ime fractile min\~english Returns the ith minimum fractile.
	virtual float quantileGt(int i)const=0;//!<\~french Retourne le ime fractile max\~english Returns the ith maximum fractile.
	virtual float confInt(float pct,int i)const=0;//!<\~french Retourne le ime intervalle de confiance  pct%\~english Returns the ith confidence interval at pct%.
	virtual float min(int i)const = 0;//!<\~french Retourne le ime minimum\~english Returns the ith minimum.
	virtual float max(int i)const = 0;//!<\~french Retourne le ime maximum\~english Returns the ith maximum.
	virtual float seqMin(SEQ_ID i)const = 0;//!<\~french Retourne le minimum de la ime squence\~english Returns the minimum of the ith sequence.
	virtual float seqMax(SEQ_ID i)const = 0;//!<\~french Retourne le maximum de la ime squence\~english Returns the maximum of the ith sequence.
	virtual float seqLast(SEQ_ID i)const = 0;//!<\~french Retourne la dernire valeur de la ime squence\~english Returns the last value of the ith sequence.
	virtual std::vector<float>distribution(int i)const=0;//!<\~french Retourne les probabilits de la ime distribution\~english Returns the probabilities of the ith distribution.
	virtual std::vector<float>meanDistribution()const=0;//!<\~french Retourne les probabilits de la distribution moyenne sur tous les instants\~english Returns the probabilities of the distribution averaged over all instants.
	virtual std::vector<float>means()const=0;//!<\~french retourne les moyennes aux diffrents instants\~english Returns the averages at different instants.
	virtual std::vector<float>stdDevs()const=0;//!<\~french retourne les carts types aux diffrents instants\~english Returns the standard deviations at different instants.
	virtual std::vector<float>quantilesLe()const=0;//!<\~french retourne les fractiles min aux diffrents instants\~english Returns the minimum fractiles at different instants.
	virtual std::vector<float>quantilesGt()const=0;//!<\~french retourne les fractiles max aux diffrents instants\~english Returns the maximum fractiles at different instants.
	virtual std::vector<float>confInts(float pct)const=0;//!<\~french retourne les intervalles de confiance  pct% aux diffrents instants\~english Returns the confidence intervals at pct% at different instants.
	virtual std::vector<float>mins()const = 0;//!<\~french retourne les minimums aux diffrents instants\~english Returns the minimums at different instants.
	virtual std::vector<float>maxs()const = 0;//!<\~french retourne les maximums aux diffrents instants\~english Returns the maximums at different instants.
	virtual std::vector<float>seqMins()const = 0;//!<\~french retourne les minimums des diffrentes squences\~english Returns the minimums of the different sequences.
	virtual std::vector<float>seqMaxs()const = 0;//!<\~french retourne les maximums des diffrentes squences\~english Returns the maximums of the different sequences.
	virtual std::vector<float>seqLasts()const = 0;//!<\~french retourne les dernires valeurs des diffrentes squences\~english Returns the last values of the different sequences.
	virtual std::vector<float>result(int type)const=0;//!<\~french retourne le resultat correspondant  la restitution de type type (une seule restitution)\~english Returns the result corresponding to the restitution of type type (a single restitution).
	virtual void setToKeep(bool toKeep)=0;//!<\~french Modifie l'indice de conservation de l'indicateur\~english Modifies the preservation index of the indicator.
	virtual bool toKeep()const=0;//!<\~french Retourne vrai si l'indicateur doit tre conserv\~english Returns true if the indicator should be preserved.
};

/**\~french Interface d'accs aux cibles.
Une cible constitue une condition d'arrt de simulation d'une squence.
La condition peut tre exprime  partir d'une valeur suivie sous forme :
- d'une variable dsigne par setVariable()
- d'une mthode d'un des objets du systme ou du systme lui mme (uniquement  partir de CSystem::addTarget)
- d'une expression textuelle
La condition d'atteinte de la cible peut tre :
- le fait que la valeur suivie est diffrente de 0 (par dfaut)
- le fait que la valeur suivie vrifie la condition de comparaison `setCondFct(op, limite)` : "valeur suivie op limite"
Une cible peut tre inhibe setInhibited()

exemple :
\~english Target access interface.
A target constitutes a stopping condition for simulating a sequence.
The condition can be expressed from a tracked value in the following ways:
- by a variable designated by setVariable()
- by a method of one of the system objects or the system itself (only through CSystem::addTarget)
- by a textual expression

The target achievement condition can be:
- the fact that the tracked value is different from 0 (default)
- the fact that the tracked value satisfies the comparison condition `setCondFct(op, limit)`: "tracked value op limit"

A target can be inhibited with setInhibited()

example:
\~
~~~~{.cpp}
class System: public CSystem{
public:
	bool target();//Target definition method
};
ITarget*tgt1=sys.addTarget("TargetName1","VarName","VAR",">",0);
ITarget*tgt2=sys.addTarget("TargetName2",&System::targ);
~~~~
*/
class ITarget: public CSysNamedComp{
	ITarget(ITarget const&);
	ITarget&operator=(ITarget const&);
protected:
	ITarget(char const*name,CSystem&sys);
public:
	virtual void setInhibited(bool value)=0;//!<\~french Inhibe la cible\~english Inhibits the target.
	virtual bool inhibited()const=0;//!<\~french Retourne vrai si la cible est inhibe\~english Returns true if the target is inhibited.
	virtual void setVariable(IVariable*var)=0;//!<\~french Dfinit l'indicateur  partir de la valeur d'une variable\~english Defines the target based on the value of a variable.
	virtual IVariable const*variable()const=0;//!<\~french Retourne la variable  partir de laquelle l'indicateur est calcul si elle existe\~english Returns the variable from which the target is calculated, if it exists.
	virtual void setMethod(bool (CSystem::*fct)())=0;//!<\~french Dfinit la mthode systme utilise pour calculer la cible\~english Defines the system method used to calculate the target.
	virtual void setMethod(PyObject*pyObj)=0;//!<\~french Dfinit la mthode Python utilise pour calculer la cible\~english Defines the Python method used to calculate the target.
	virtual char const*pythonScript()const=0;//!<\~french Retourne le corps de la fonction Python\~english Returns the body of the Python function.
	virtual void setPythonScript(char const*script)=0;//!<\~french Dfinit le corps de la fonction Python utilise pour calculer la cible\~english Defines the body of the Python function used to calculate the target.
	virtual char const*expression()const=0;//!<\~french Retourne l'expression textuelle de calcul de la cible\~english Returns the textual expression for calculating the target.
	virtual void setExpression(char const*expr)=0;//!<\~french Dfinit l'expression textuelle de calcul de la cible (si pas d'oprateur, cible atteinte si valeur non nulle)\~english Defines the textual expression for calculating the target (if no operator, the target is achieved if the value is non-zero).
	virtual void setCondFct(char const*op,float value)=0;//!<\~french Dfinit une comparaison de la valeur courante avec un rel\~english Defines a comparison of the current value with a real number.
	virtual TOperator::EOp op()const=0;//!<\~french Retourne l'oprateur de la comparaison\~english Returns the operator of the comparison.
	virtual char const*opStr()const=0;//!<\~french Retourne l'oprateur de la comparaison sous forme textuelle\~english Returns the comparison operator as a text string.
	virtual float limit()const=0;//!<\~french Retourne la limite de la comparaison\~english Returns the comparison limit.
	virtual bool bValue()const=0;//!<\~french Retourne la valeur du test de cible\~english Returns the target test value.
};

/**\~french Interface commune  l'ensemble des variables et des rfrences.
Cette interface permet de dfinir les callbacks.
Un callback est une mthode du composant parent de la `IVarBase`.

exemple:
\~english Common interface for all variables and references.
This interface allows defining callbacks.
A callback is a method of the parent component of `IVarBase`.

example:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
    void cb(){
	    //Callback processing
	}
    CComp(){
		IVarBase*var = addVariable("VarName",TVarType::t_double,1.);
		var->addCallback("cb",&CComp::cb);
	}
};
~~~~
*/
class PYC_PUBLIC IVarBase:public CNamedComp{
	friend class CSystemState;
	friend void IVarBase_addCallback(IVarBase*, char const*, PyObject*, int);
	virtual void addCallback(char const*name,void(CSysNamedComp::*method)(),PyObject*pyMth, int sens)=0;
protected:
	CNX_ID m_N;//!<\~french Nombre maximum de liens accepts par la variable.\~english Maximum number of links accepted by the variable.
	IVarBase() :CNamedComp() {}
	IVarBase(CNamedComp&parent,char const*name):CNamedComp(parent,name),m_N(0xffffffff){}
	~IVarBase(){}
public:
	/**\~french Retourne le type de la valeur courante de la variable.\~english Returns the type of the current value of the variable.*/
	virtual std::type_info const&typeInfo()const{return typeid(bool);}
	/**\~french Retourne le nombre maximum de connexion sur la variable.\~english Returns the maximum number of connections for the variable.*/
	virtual CNX_ID cnctMax()const{return m_N;}
	/**\~french Ajoute une mthode sensible aux modifications de la variable.
	@param name nom de la mthode
	@param method mthode sensible  ajouter
	@param sens sens de sensibilit de la mthode (0 tous, 1 accroissement, -1 rduction)
	\~english Adds a method sensitive to modifications of the variable.
	@param name name of the method
	@param method sensitive method to add
	@param sens sensitivity direction of the method (0 for all, 1 for increase, -1 for decrease)
	*/
	template<typename Type>void addCallback(char const* name, void(Type::* method)(), int sens = 0) { checkMethodAddrSize(sizeof(method)); addCallback(name, static_cast<void(CSysNamedComp::*)()>(method), NULL, sens); }
	template<typename Type>DEPRECATED void addSensitiveMethod(char const* name, void(Type::* method)(), int sens = 0) { ILogManager::glLogManager().msgObsolete("IVarBase::addSensitiveMethod", "addCallback"); addCallback(name, method, sens); }
	/**\~french Supprime la mthode de nom name des mthodes sensible aux modifications de la variable\~english Removes the method named *name* from the methods sensitive to variable modifications.*/
	virtual void removeCallback(char const* name) = 0;
	DEPRECATED void removeSensitiveMethod(char const* name);
	virtual void clearCallbacks() = 0;//!<Supprime toutes les mthodes sensibles\~english Removes all sensitive methods.
	DEPRECATED void clearSensitiveMethods();//!<Supprime toutes les mthodes sensibles\~english Removes all sensitive methods.
	/**\~french Modifie le nombre maximum de connexions sur la variable\~english Modifies the maximum number of connections for the variable.*/
	void setCnctMax(CNX_ID N){m_N=N;}
};

/**\~french Structure de dfinition de l'numr de dfinition du type d'une variable.\~english Definition structure of the enumeration for defining a variable type.*/
struct TVarType{
	/**\~french Enumr de dfinition du type d'une variable.\~english Enumeration for defining a variable type.*/
	typedef enum EVarType{
		t_unknown=0,//!<\~french Type inconnu\~english Unknown type.
		t_bool,//!<\~french Variable boolenne\~english Boolean variable.
		t_int,//!<\~french Variable entire\~english Integer variable.
		t_float,//!<\~french Variable relle\~english Real variable.
		t_double,//!<\~french Variable relle double prcision\~english Double precision real variable.
		t_complex,//!<\~french Variable complexe double prcision\~english Double precision complex variable.
		t_string,//!<\~french Variable chane de caractres\~english String variable.
		t_abool,//!<\~french Variable tableau de boolens\~english Boolean array variable.
		t_aint,//!<\~french Variable tableau d'entiers\~english Integer array variable.
		t_afloat,//!<\~french Variable tableau de rels\~english Real array variable.
		t_adouble,//!<\~french Variable tableau de rels double prcision\~english Double precision real array variable.
		t_acomplex,//!<\~french Variable tableau de complexes double prcision\~english Double precision complex array variable.
		t_astring//!<\~french Variable tableau de chanes de caractres\~english String array variable.
	}EVarType;
	static char const*st_Names[13];
	static EVarType fromStr(char const*value);//!<\~french Lit le type de variable  partir d'une chane au format du fichier XML de paramtres\~english Reads the variable type from a string in the XML parameter file format.
};

/**\~french Structure de dfinition de l'numr de dfinition des distributions de probabilit prdfinies\~english Definition structure of the enumeration for predefined probability distributions.*/
struct PYC_PUBLIC TDistributionType{
	/**\~french Enumr de dfinition des distributions de probabilit prdfinies dans le coeur de Pycatshoo\~english Enumeration of predefined probability distributions in the core of Pycatshoo.*/
	typedef enum EDistributionType{
		none=0,//!<\~french Pas de distribution\~english No distribution.
		expo,//!<\~french Loi exponentielle (1 paramtre : lambda)\~english Exponential distribution (1 parameter: lambda).
		frechet,//!<\~french Loi de Frchet (2 paramtres : alpha, beta)\~english Frchet distribution (2 parameters: alpha, beta).
		gamma,//!<\~french Loi gamma (2 paramtres : alpha, beta)\~english Gamma distribution (2 parameters: alpha, beta).
		gumbel,//!<\~french Loi de Gumbel (2 paramtres : mu, sigma)\~english Gumbel distribution (2 parameters: mu, sigma).
		lognormal,//!<\~french Loi log normale (2 paramtres : mu, sigma)\~english Log-normal distribution (2 parameters: mu, sigma).
		normal,//!<\~french Loi normale (2 paramtres : mu, sigma)\~english Normal distribution (2 parameters: mu, sigma).
		pareto,//!<\~french Loi de Pareto (3 paramtres : alpha, beta, delta)\~english Pareto distribution (3 parameters: alpha, beta, delta).
		triangle,//!<\~french Loi triangulaire (3 paramtres : min, interm, max)\~english Triangular distribution (3 parameters: min, mode, max).
		uniforme,//!<\~french Loi uniforme (2 paramtres : min, max)\~english Uniform distribution (2 parameters: min, max).
		uniform=uniforme,//!<\~french Loi uniforme (2 paramtres : min, max)\~english Uniform distribution (2 parameters: min, max).
		weibull,//!<\~french Loi de Weibull (2 paramtres : alpha, beta)\~english Weibull distribution (2 parameters: alpha, beta).
		loguniform//!<\~french Loi log uniforme (2 paramtres : min, max)\~english Log-uniform distribution (2 parameters: min, max).
	}EDistributionType;
	static char const*st_Names[12];
	static EDistributionType fromStr(char const*value);//!<\~french Lit le type d'indicateur  partir d'une chane au format du fichier XML de paramtres\~english Reads the indicator type from a string in the XML parameter file format.
};

/**\~french Structure de dfinition de l'numr de dfinition de la causalit d'une variable\~english Definition structure of the enumeration for defining the causality of a variable.*/
struct PYC_PUBLIC TCausality{
	/**\~french Enumr de dfinition de la causalit d'une variable.
	Cette notion n'est utilise que lors de la construction d'un FMU  partir d'un systme PyCATSHOO.

	C'est un sous ensemble des valeurs admises par FMU2.
	- "calculated parameter" est dduit de "parameter" lorsque l'initialisabilit n'est pas "exact".
	- "local" n'est pas distingu de "output"
	- "independent" n'est pas pris en compte
	\~english Enumeration for defining the causality of a variable.
	This concept is used only when constructing an FMU from a PyCATSHOO system.

	It is a subset of the values allowed by FMU2.
	- "calculated parameter" is derived from "parameter" when initializability is not "exact."
	- "local" is not distinguished from "output."
	- "independent" is not considered.
	*/
	typedef enum ECausality{
		unknown=0,//!<\~french Causalit indfinie : la variable ne sera pas prsente  l'extrieur\~english Undefined causality: the variable will not be exposed externally.
		structural,//!<\~french Paramtre structurel (peut tre fixe ou modifiable de manire discrte)\~english Structural Parameter (can be fixed or modified discretely).
		parameter,//!<\~french Paramtre (peut tre fixe ou modifiable de manire discrte)\~english Parameter (can be fixed or modified discretely).
		input,//!<\~french Variable d'entre du modle (en principe non initializable)\~english Model input variable (typically non-initializable).
		output//!<\~french Variable de sortie du modle (non modifiable sauf ventuellement la valeur initiale)\~english Model output variable (non-modifiable except possibly the initial value).
	}ECausality;
	static char const*st_Names[5];
};

/**\~french Structure de dfinition de l'numr de dfinition de la variabilit d'une variable\~english Definition structure of the enumeration for defining the variability of a variable.*/
struct PYC_PUBLIC TVariability{
	/**\~french Enumr de dfinition de la variabilit d'une variable.
	Cette notion n'est utilise que lors de la construction d'un FMU  partir d'un systme PyCATSHOO.
	C'est un sous ensemble des valeurs admises par FMU2 o "fixed" et "tunable" sont regroups sous "constant" et "discrete" respectivement
	\~english Enumeration for defining the variability of a variable.
	This concept is used only when constructing an FMU from a PyCATSHOO system.
	It is a subset of the values allowed by FMU2, where "fixed" and "tunable" are grouped under "constant" and "discrete," respectively.
	*/
	typedef enum EVariability{
		unknown=0,
		constant,//!<\~french Invariable aprs l'initialisation (transform en "fixed" pour les paramtres)\~english Invariable after initialization (converted to "fixed" for parameters).
		discrete,//!<\~french Variable uniquement lors des vnements (transform en "tunable" pour les paramtres)\~english Variable only during events (converted to "tunable" for parameters).
		continuous//!<\~french Continuement variable\~english Continuously variable.
	}EVariability;
	static char const*st_Names[4];
};

/**\~french Structure de dfinition de l'numr de dfinition de l'initialisabilit d'une variable\~english Definition structure of the enumeration for defining the initializability of a variable.*/
struct PYC_PUBLIC TInitializability{
	/**\~french Enumr de dfinition de l'initialisabilit d'une variable.
	Cette notion n'est utilise que lors de la construction d'un FMU  partir d'un systme PyCATSHOO.
	Elle reprend les valeurs admises par FMU2.
	\~english Enumeration for defining the initializability of a variable.
	This concept is used only when constructing an FMU from a PyCATSHOO system.
	It uses the values allowed by FMU2.
	*/
	typedef enum EInitializability{
		unknown=0,
		exact,//!<\~french Une valeur initiale exacte peut tre fournie\~english An exact initial value can be provided.
		approx,//!<\~french Une valeur initiale approche peut tre fournie, mais la valeur exacte est calcule\~english An approximate initial value can be provided, but the exact value is calculated.
		calculated//!<\~french La valeur initiale est calcule\~english The initial value is calculated.
	}EInitializability;
	static char const*st_Names[4];
};

/**\~french Interface de gestion des variables.
Les variables peuvent tre des variables d'tat, des automates ou des tats.
Cette interface permet d'accder  la valeur initiale et la la valeur courante.
Lorsqu'il s'agit d'une variable d'tat, les lments suivants sont accessibles :
- elle peut tre qualifie de rinitialise (sa valeur courante est rinitialise  la valeur initiale aprs chaque tirage de transitions) : setReinitialized()
- elle peut tre qualifie d'interpole (sa valeur en analyse sera interpole entre les points de mmorisation) : setInterpolated()
- sa drive peut tre renseigne (si la variable participe  une ODE) : setDvdtODE()
- sa valeur peut tre modifie

La valeur d'une variable a un type dfini lors de sa cration et non modifiable.
Mais la valeur d'une variable peut tre convertie entre tous les types disponibles grce aux mthodes d'accs.
Ainsi, `sValue` appel pour une variable relle retourne la reprsentation textuelle de la valeur relle et `setValue("5.1")` appele sur une variable entire lui donnera la valeur 5.

exemple:
\~english Variable management interface.
Variables can be state variables, automata, or states.
This interface provides access to the initial and current values.
For a state variable, the following elements are accessible:
- it can be designated as reinitialized (its current value is reset to the initial value after each transition draw): setReinitialized()
- it can be designated as interpolated (its analysis value will be interpolated between memory points): setInterpolated()
- its derivative can be set (if the variable is part of an ODE): setDvdtODE()
- its value can be modified

The type of a variable is defined at creation and is not modifiable.
However, the value of a variable can be converted between all available types using access methods.
For example, `sValue` called on a real variable returns the textual representation of the real value, and `setValue("5.1")` called on an integer variable assigns it the value 5.

example:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
	CComp(){
		IVariable*var1 = addVariable("VarName1",TVarType::t_double,1.);// Real variable
		var1->setUncertainty(TDistributionType::normal,3.,5.);// Uncertain variable
		var1->setUncertaintyLimits(0.,10);
		IVariable*var2 = addVariable("VarName2",TVarType::t_double,0.);
		var2->setDelay(5.);// Delayed variable
		var2->setValue(3.);
		var2->setValue(-5,1.);
		var2->setInterpolated(true);
		IVariable*var3 = addVariable("VarName3",TVarType::t_int,4.);// Integer variable
	}
};
~~~~
*/
class PYC_PUBLIC IVariable : public IVarBase,public IMonitored{
	bool value(bool const*)const { return bValue(); }
	int value(int const*)const { return iValue(); }
	float value(float const*)const { return fValue(); }
	double value(double const*)const { return dValue(); }
	std::complex<double> value(std::complex<double> const*)const { return cValue(); }
	std::string value(std::string const*)const { return sValue(); }
	bool value(int const*pt, bool const*)const { return bValue(pt); }
	int value(int const* pt, int const*)const { return iValue(pt); }
	float value(int const* pt, float const*)const { return fValue(pt); }
	double value(int const* pt, double const*)const { return dValue(pt); }
	std::complex<double> value(int const* pt, std::complex<double> const*)const { return cValue(pt); }
	std::string value(int const* pt, std::string const*)const { return sValue(pt); }
	std::vector<bool> values(bool const*)const { return bValues(); }
	std::vector<int> values(int const*)const { return iValues(); }
	std::vector<float> values(float const*)const { return fValues(); }
	std::vector<double> values(double const*)const { return dValues(); }
	std::vector < std::complex<double>> values(std::complex<double> const*)const { return cValues(); }
	std::vector < std::string> values(std::string const*)const { return sValues(); }
	virtual void* valuePtr()const { notArrayVariable(); return NULL; }
	virtual void* initValuePtr()const { notArrayVariable(); return NULL; }
protected:
	void notArrayVariable()const { ILogManager::glLogManager().throwError("Lar variable %s n'est pas une variable tableau", name()); }
	template<typename T>void setValue(T& val, T const& newVal, int& sens) {
		if(val == newVal)
			return;
		if(!sens)
			sens = val < newVal ? 1 : -1;
		val = newVal;
	}
	IVariable() :IVarBase() {}
	IVariable(CNamedComp&parent,char const*name):IVarBase(parent,name){}
	~IVariable(){}
public:
	char const*basename()const override{return CNamedComp::basename();}
	char const*name()const override{return CNamedComp::name();}
	CNamedComp*parent()const override{return CNamedComp::parent();}
	virtual CVariable const*variable()const{return NULL;}
	/**\~french  \fn setTrace
	modifie le niveau de trace
	@param trace : niveau souhait = cumul de :
	- 1 : trace les modifications de la variable aprs stabilisation des callback
	- 2 : trace chaque affectation
	- 4 : trace  chaque pas d'intgration ODE
	\~english  \fn setTrace  
	modifies the trace level  
	@param trace: desired level = combination of:  
	- 1 : traces variable modifications after callback stabilization  
	- 2 : traces each assignment  
	- 4 : traces at each ODE integration step*/
	inline void setTrace(int trace){IMonitored::setTrace(trace);}//Uniquement pour valider la doc
	/**\~french Retourne vrai si la variable est une variable retarde\~english Returns true if the variable is a delayed variable.*/
	virtual bool isDelayed()const { return false; }
	/**\~french Modifie la rinitialisation de la variable\~english Modifies the reinitialization of the variable.*/
	virtual void setReinitialized(bool toReinit){ILogManager::glLogManager().throwError("Mthode setReinitialized inutilisable sur cette variable");}
	/**\~french Retourne la rinitialisation de la variable\~english Returns the reinitialization status of the variable.*/
	virtual bool isReinitialized()const{return false;}
	/**\~french Modifie la gestion de l'interpolation des valeurs de la variable\~english Modifies the interpolation handling of the variable's values.*/
	virtual void setInterpolated(bool interp);
	/**\~french Retourne vrai si les valeurs de la variable doivent tre interpoles\~english Returns true if the variable's values should be interpolated.*/
	virtual bool isInterpolated()const{return false;}
	/**\~french Retourne la drive de la variable pour les ODE implicites\~english Returns the derivative of the variable for implicit ODEs.*/
	virtual double dvdtODE()const=0;
	/**\~french Modifie la drive de la valeur pour les ODE\~english Modifies the value's derivative for ODEs.*/
	virtual void setDvdtODE(double dvdt)=0;
	/**\~french Modifie la valeur rsiduelle de rsolution des quations diffrentielles algbriques\~english Modifies the residual value for solving differential-algebraic equations.*/
	virtual void setResidualValue(double value)=0;
	/**\~french Modifie la valeur de la drive de la variable par rapport  var\~english Modifies the derivative value of the variable with respect to var.*/
	virtual void setPartial(IVariable&var,double value)=0;
	/**\~french Modifie la valeur de la drive de la variable par rapport  la drive de var\~english Modifies the derivative value of the variable with respect to the derivative of var.*/
	virtual void setPartialP(IVariable&var,double value)=0;
	/**\~french Modifie le fait que la variable soit modifiable de l'extrieur\~english Modifies whether the variable can be modified externally.*/
	virtual void setModifiable(bool res)=0;
	/**\~french Modifie la causalit de la variable (pour la gnration du XML FMU)\~english Modifies the causality of the variable (for FMU XML generation).*/
	virtual void setCausality(TCausality::ECausality causality){ILogManager::glLogManager().throwError("Mthode setCausality inutilisable sur cette variable");}
	virtual TCausality::ECausality causality()const{return TCausality::unknown;}//!<\~french Retourne la causalit de la variable (par dfaut unknown)\~english Returns the causality of the variable (default is unknown).
	/**\~french Modifie la variabilit de la variable (pour la gnration du XML FMU)\~english Modifies the variability of the variable (for FMU XML generation).*/
	virtual void setVariability(TVariability::EVariability variability){ILogManager::glLogManager().throwError("Mthode setVariability inutilisable sur cette variable");}
	virtual TVariability::EVariability variability()const{return TVariability::unknown;}//!<\~french Retourne la variabilit de la variable (par dfaut unknown)\~english Returns the variability of the variable (default is unknown).
	/**\~french Modifie l'initialisabilit de la variable (pour la gnration du XML FMU)*/
	virtual void setInitializability(TInitializability::EInitializability initializability){ILogManager::glLogManager().throwError("Mthode setCausality inutilisable sur cette variable");}
	virtual TInitializability::EInitializability initializability()const{return TInitializability::unknown;}//!<\~french Retourne l'initialisabilit de la variable (par dfaut unknown)
	/**\~french Modifie la causalit, la variabilit et l'initialisabilit de la variable (pour la gnration du XML FMU)\~english Modifies the initializability of the variable (for FMU XML generation).*/
	virtual void setFMUProperties(TCausality::ECausality causality,TVariability::EVariability variability,TInitializability::EInitializability initializability){ILogManager::glLogManager().throwError("Mthode setCausality inutilisable sur cette variable");}
	/**\~french Retourne la drive de la valeur de la variable (uniquement pour les variables double) avec un pas de calcul de dt\~english Returns the derivative of the variable's value (only for double variables) with a calculation step of dt.*/
	virtual double derivative(double dt)const=0;
	/**\~french Retourne la drive de la valeur de la variable (uniquement pour les variables double) avec un pas de calcul de dt\~english Returns the derivative of the variable's value (only for double variables) with a calculation step of dt.*/
	virtual double dtDerivative(double dt)const{return derivative(dt);/*Pour obtenir le message d'erreur par dfaut\~english To obtain the default error message.*/}
	/**\~french Dsigne la variable dont la variable courante est l'intgrale.\~english Designates the variable of which the current variable is the integral.*/
	virtual void integrate(IVariable*var);
	/**\~french Dsigne la variable dont la variable courante est l'intgrale.\~english Designates the variable of which the current variable is the integral.*/
	virtual void integrate(IVariable&var){integrate(&var);}
	/**\~french Retourne la variable intgre ou NULL s'il n'y en a pas\~english Returns the integrated variable or NULL if there is none.*/
	virtual IVariable* integratedVariable()const { return nullptr; }
	/**\~french Retourne vrai si la variable est intgrale\~english Returns true if the variable is an integral.*/
	virtual bool isIntegral()const { return false; }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) in complex form.*/
	virtual void setValue(std::complex<double>const&val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) in complex form.*/
	inline void setCValue(std::complex<double>const&val){setValue(val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) in double form.*/
	virtual void setValue(double val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) in double form.*/
	inline void setDValue(double val){setValue(val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) in float form.*/
	virtual void setValue(float val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) in float form.*/
	inline void setFValue(float val){setValue(val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) in int form.*/
	virtual void setValue(int val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) in int form.*/
	inline void setIValue(int val){setValue(val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) in bool form.*/
	virtual void setValue(bool val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) in bool form.*/
	inline void setBValue(bool val){setValue(val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) in string form.*/
	virtual void setValue(char const*val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) in string form.*/
	inline void setValue(std::string const&val) { setValue(val.c_str()); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) in string form.*/
	inline void setSValue(char const* val) { setValue(val); }
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de complexe.\~english Modifies the initial value of a delayed variable in complex form if the simulation is not started.*/
	virtual void setValue(PyTime time, std::complex<double>const&val);
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de complexe.\~english Modifies the initial value of a delayed variable in complex form if the simulation is not started.*/
	inline void setCValue(PyTime time, std::complex<double>const&val){setValue(time, val);}
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de double.\~english Modifies the initial value of a delayed variable in double form if the simulation is not started.*/
	virtual void setValue(PyTime time, double val);
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de double.\~english Modifies the initial value of a delayed variable in double form if the simulation is not started.*/
	inline void setDValue(PyTime time, double val){setValue(time, val);}
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de float.\~english Modifies the initial value of a delayed variable in float form if the simulation is not started.*/
	virtual void setValue(PyTime time, float val);
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de float.\~english Modifies the initial value of a delayed variable in float form if the simulation is not started.*/
	inline void setFValue(PyTime time, float val){setValue(time, val);}
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de int.\~english Modifies the initial value of a delayed variable in int form if the simulation is not started.*/
	virtual void setValue(PyTime time, int val);
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de int.\~english Modifies the initial value of a delayed variable in int form if the simulation is not started.*/
	inline void setIValue(PyTime time, int val){setValue(time, val);}
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de bool.\~english Modifies the initial value of a delayed variable in bool form if the simulation is not started.*/
	virtual void setValue(PyTime time, bool val);
	/**\~french Modifie la valeur initiale d'une variable retarde si la simulation n'est pas lance sous forme de bool.\~english Modifies the initial value of a delayed variable in bool form if the simulation is not started.*/
	inline void setBValue(PyTime time, bool val){setValue(time, val);}
	/**\~french Modifie le script d'initialisation de la variable (uniquement pour les variables).\~english Modifies the initialization script of the variable (for variables only).*/
	virtual void setInitScript(char const*) { ILogManager::glLogManager().throwError("La fonctionnalit n'est disponible que pour les variables"); }
	/**\~french Retourne le script d'initialisation de la variable\~english Returns the initialization script of the variable.*/
	virtual std::string initScript()const { return ""; }
	/**\~french Nettoie l'historique des valeurs initiales pour une variable retarde\~english Clears the initial value history for a delayed variable.*/
	virtual void clearInitValues(){};
	/**\~french Retourne la valeur courante sous forme de complexe.\~english Returns the current value in complex form.*/
	virtual std::complex<double> cValue()const=0;
	/**\~french Retourne la valeur courante sous forme de double.\~english Returns the current value in double form.*/
	virtual double dValue()const=0;
	/**\~french Retourne la valeur courante sous forme de float.\~english Returns the current value in float form.*/
	virtual float fValue()const=0;
	/**\~french Retourne la valeur courante sous forme de int.\~english Returns the current value in int form.*/
	virtual int iValue()const=0;
	/**\~french Retourne la valeur courante sous forme de bool.\~english Returns the current value in bool form.*/
	virtual bool bValue()const=0;
	/**\~french Retourne la valeur courante sous forme de string.\~english Returns the current value in string form.*/
	virtual std::string sValue()const=0;
	/**\~french Retourne la valeur courante.\~english Returns the current value.*/
	template<typename T>T value()const { return value((T*)NULL); }
	/**\~french Retourne toutes les valeurs courantes.\~english Returns all the current values.*/
	template<typename T>std::vector<T> values()const { return values((T*)NULL); }
	/**\~french Retourne vrai si la valeur initiale est la valeur par dfaut de la variable.\~english Returns true if the initial value is the default value of the variable.*/
	virtual bool isDefault()const = 0;
	/**\~french Retourne la valeur initiale sous forme de complexe.\~english Returns the initial value in complex form.*/
	virtual std::complex<double> cInitValue()const = 0;
	/**\~french Retourne la valeur initiale sous forme de double.\~english Returns the initial value in double form.*/
	virtual double dInitValue()const = 0;
	/**\~french Retourne la valeur initiale sous forme de float.\~english Returns the initial value in float form.*/
	virtual float fInitValue()const override = 0;
	/**\~french Retourne la valeur initiale sous forme de int.\~english Returns the initial value in int form.*/
	virtual int iInitValue()const = 0;
	/**\~french Retourne la valeur initiale sous forme de bool.\~english Returns the initial value in bool form.*/
	virtual bool bInitValue()const = 0;
	/**\~french Retourne la valeur initiale sous forme de string.\~english Returns the initial value in string form.*/
	virtual std::string sInitValue()const = 0;
	/**\~french Retourne la valeur initiale retarde sous forme de complexe.\~english Returns the delayed initial value in complex form.*/
	virtual std::complex<double> cInitValue(PyTime time)const { return cInitValue(); }
	/**\~french Retourne la valeur initiale retarde sous forme de double.\~english Returns the delayed initial value in double form.*/
	virtual double dInitValue(PyTime time)const { return dInitValue(); }
	/**\~french Retourne la valeur initiale retarde sous forme de float.\~english Returns the delayed initial value in float form.*/
	virtual float fInitValue(PyTime time)const { return fInitValue(); }
	/**\~french Retourne la valeur initiale retarde sous forme de int.\~english Returns the delayed initial value in int form.*/
	virtual int iInitValue(PyTime time)const { return iInitValue(); }
	/**\~french Retourne la valeur initiale retarde sous forme de bool.\~english Returns the delayed initial value in bool form.*/
	virtual bool bInitValue(PyTime time)const { return bInitValue(); }

	/**\~freanch Retourne vrai si la variable est une variable tableau.\~english Return true if variable is array variable.*/
	virtual bool isArray()const { return false; }
	/**\~french Modifie la dimension d'une variable tableau.\~english Modify the dimension of an array variable.*/
	virtual void setDimensions(std::vector<int> const& dims) { notArrayVariable(); }
	/**\~french Retourne la dimension d'une variable tableau.\~english Return the dimension of an array variable.*/
	virtual std::vector<int>dimensions()const { return std::vector<int>(); }
	/**\~french Retourne les coordonnes du point correspondant  la place dans le tableau monodimensionnel d'une variable tableau.\~english Return the coordinates of the point corresponding to the place in the one dimention array of an array variable.*/
	virtual std::vector<int>point(size_t place)const{notArrayVariable(); return std::vector<int>();}
	/**\~french Retourne la place dans le tableau monodimensionnel correspondant au point pt d'une variable tableau.\~english Return the place in the one dimension array correponding to the point pt of an array variable.*/
	virtual size_t place(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la place dans le tableau monodimensionnel correspondant au point pt d'une variable tableau.\~english Return the place in the one dimension array correponding to the point pt of an array variable.*/
	size_t place(std::vector<int>const&pt)const { return place(pt.data()); }
	/**\~french Retourne toutes les valeurs sous forme de complexe.\~english Return all values in complex form.*/
	virtual std::vector<std::complex<double>>cValues()const { return std::vector<std::complex<double>>(1, cValue()); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in complex form.*/
	virtual void setValue(int const*pt,std::complex<double>const& val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in complex form.*/
	void setValue(std::vector<int>const& pt, std::complex<double>const& val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in complex form.*/
	inline void setCValue(int const* pt, std::complex<double>const& val) { setValue(pt, val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in complex form.*/
	inline void setCValue(std::vector<int>const& pt, std::complex<double>const& val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) in complex form.*/
	virtual void setCValues(std::vector<std::complex<double>>const& val) { setCValues(val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) in complex form.*/
	inline void setValues(std::vector<std::complex<double>>const& val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in double form.*/
	virtual void setValue(int const*pt,double val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in double form.*/
	inline void setValue(std::vector<int>const& pt, double val) { setValue(pt.data(), val); }
	/**\~french Retourne toutes les valeurs sous forme de double.\~english Return all values in double form.*/
	virtual std::vector<double>dValues()const { return std::vector<double>(1, dValue()); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in double form.*/
	inline void setDValue(int const*pt,double val) { setValue(pt, val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in double form.*/
	inline void setDValue(std::vector<int>const& pt, double val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) in double form.*/
	virtual void setDValues(std::vector<double>const& val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) in double form.*/
	inline void setValues(std::vector<double>const& val) { setDValues(val); }
	/**\~french Retourne toutes les valeurs sous forme de float.\~english Return all values in float form.*/
	virtual std::vector<float>fValues()const { return std::vector<float>(1, fValue()); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in float form.*/
	virtual void setValue(int const* pt, float val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in float form.*/
	inline void setValue(std::vector<int>const& pt, float val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in float form.*/
	inline void setFValue(int const* pt, float val) { setValue(pt, val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in float form.*/
	inline void setFValue(std::vector<int>const& pt, float val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de rel.\~english Modifies the current value (or the initial value if the simulation is not started) in float form.*/
	virtual void setFValues(std::vector<float>const& val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de rel.\~english Modifies the current value (or the initial value if the simulation is not started) in float form.*/
	inline void setValues(std::vector<float>const& val) { setFValues(val); }
	/**\~french Retourne toutes les valeurs sous forme de int.\~english Return all values in float int.*/
	virtual std::vector<int>iValues()const { return std::vector<int>(1, iValue()); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in int form.*/
	virtual void setValue(int const* pt, int val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in int form.*/
	inline void setValue(std::vector<int>const& pt, int val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in int form.*/
	inline void setIValue(int const* pt, int val) { setValue(pt, val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in int form.*/
	inline void setIValue(std::vector<int>const& pt, int val) { setValue(pt, val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme d'entier.\~english Modifies the current value (or the initial value if the simulation is not started) in integer form.*/
	virtual void setIValues(std::vector<int>const& val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme d'entier.\~english Modifies the current value (or the initial value if the simulation is not started) in integer form.*/
	inline void setValues(std::vector<int>const& val) { setIValues(val); }
	/**\~french Retourne toutes les valeurs sous forme de boolen.\~english Return all values in bolean form.*/
	virtual std::vector<bool>bValues()const { return std::vector<bool>(1, bValue()); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in bool form.*/
	virtual void setValue(int const* pt, bool val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in bool form.*/
	inline void setValue(std::vector<int>const& pt, bool val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in bool form.*/
	inline void setBValue(int const* pt, bool val) { setValue(pt, val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in bool form.*/
	inline void setBValue(std::vector<int>const& pt, bool val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de boolen.\~english Modifies the current value (or the initial value if the simulation is not started) in bolean form.*/
	virtual void setBValues(std::vector<bool>const& val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de boolen.\~english Modifies the current value (or the initial value if the simulation is not started) in bolean form.*/
	inline void setValues(std::vector<bool>const& val) { setBValues(val); }
	/**\~french Retourne toutes les valeurs sous forme de string.\~english Return all values in string form.*/
	virtual std::vector<std::string>sValues()const { return std::vector<std::string>(1, sValue()); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in string form.*/
	virtual void setValue(int const* pt, char const* val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in string form.*/
	inline void setValue(std::vector<int>const& pt, char const* val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in string form.*/
	inline void setSValue(int const* pt, char const* val) { setValue(pt, val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) au point pt sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) at point pt in string form.*/
	inline void setSValue(std::vector<int>const& pt, char const* val) { setValue(pt.data(), val); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) in string form.*/
	virtual void setSValues(std::vector<std::string>const& val) { notArrayVariable(); }
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) in string form.*/
	inline void setValues(std::vector<std::string>const& val) { setSValues(val); }
	/**\~french Retourne la valeur courante sous forme de complexe.\~english Returns the current value at pt point in complex form.*/
	virtual std::complex<double> cValue(int const* pt)const { notArrayVariable(); return std::complex<double>(0., 0.); }
	/**\~french Retourne la valeur courante sous forme de complexe.\~english Returns the current value at pt point in complex form.*/
	std::complex<double> cValue(std::vector<int>const& pt)const { return cValue(pt.data()); }
	/**\~french Retourne la valeur courante au point pt sous forme de double.\~english Returns the current value at pt point in double form.*/
	virtual double dValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur courante au point pt sous forme de double.\~english Returns the current value at pt point in double form.*/
	double dValue(std::vector<int>const& pt)const { return dValue(pt.data()); }
	/**\~french Retourne la valeur courante au point pt sous forme de float.\~english Returns the current value at pt point in float form.*/
	virtual float fValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur courante au point pt sous forme de float.\~english Returns the current value at pt point in float form.*/
	float fValue(std::vector<int>const& pt)const { return fValue(pt.data()); }
	/**\~french Retourne la valeur courante au point pt sous forme de int.\~english Returns the current value at pt point in int form.*/
	virtual int iValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur courante au point pt sous forme de int.\~english Returns the current value at pt point in int form.*/
	int iValue(std::vector<int>const& pt)const { return iValue(pt.data()); }
	/**\~french Retourne la valeur courante au point pt sous forme de bool.\~english Returns the current value at pt point in bool form.*/
	virtual bool bValue(int const* pt)const { notArrayVariable(); return false; }
	/**\~french Retourne la valeur courante au point pt sous forme de bool.\~english Returns the current value at pt point in bool form.*/
	bool bValue(std::vector<int>const& pt)const { return bValue(pt.data()); }
	/**\~french Retourne la valeur courante au point pt sous forme de string.\~english Returns the current value at pt point in string form.*/
	virtual std::string sValue(int const* pt)const { notArrayVariable(); return ""; }
	/**\~french Retourne la valeur courante au point pt sous forme de string.\~english Returns the current value at pt point in string form.*/
	std::string sValue(std::vector<int>const& pt)const { return sValue(pt.data()); }
	/**\~french Retourne la valeur courante.\~english Returns the current value.*/
	template<typename T>T value(int const* pt)const { return value(pt, (T*)NULL); }
	/**\~french Retourne la valeur courante.\~english Returns the current value.*/
	template<typename T>T value(std::vector<int>const& pt)const {return value(pt.data());}
	/**\~french Retourne toutes les valeurs courantes sous forme du type template (attention, NULL si la variable n'est pas du type demand ou si elle est boolenne).\~english Returns all current values in template type form (attention, NULL if the variable is not of asked type or it is boolean).*/
	template<typename T>T const* valuesPtr()const { return typeInfo()==typeid(T) && typeid(T)!=typeid(bool)?reinterpret_cast<T*>(valuePtr()):NULL; }
	/**\~french Retourne la valeur initiale sous forme de complexe.\~english Returns the initial value in complex form.*/
	virtual std::complex<double> cInitValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur initiale sous forme de complexe.\~english Returns the initial value in complex form.*/
	std::complex<double> cInitValue(std::vector<int>const& pt)const { return cInitValue(pt.data()); }
	/**\~french Retourne la valeur initiale au point pt sous forme de double.\~english Returns the initial at pt point value in double form.*/
	virtual double dInitValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur initiale au point pt sous forme de double.\~english Returns the initial at pt point value in double form.*/
	double dInitValue(std::vector<int>const& pt)const { return dInitValue(pt.data()); }
	/**\~french Retourne la valeur initiale au point pt sous forme de float.\~english Returns the initial at pt point value in float form.*/
	virtual float fInitValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur initiale au point pt sous forme de float.\~english Returns the initial at pt point value in float form.*/
	float fInitValue(std::vector<int>const& pt)const { return fInitValue(pt.data()); }
	/**\~french Retourne la valeur initiale au point pt sous forme de int.\~english Returns the initial at pt point value in int form.*/
	virtual int iInitValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur initiale au point pt sous forme de int.\~english Returns the initial at pt point value in int form.*/
	int iInitValue(std::vector<int>const& pt)const { return iInitValue(pt.data()); }
	/**\~french Retourne la valeur initiale au point pt sous forme de bool.\~english Returns the initial at pt point value in bool form.*/
	virtual bool bInitValue(int const* pt)const { notArrayVariable(); return 0; }
	/**\~french Retourne la valeur initiale au point pt sous forme de bool.\~english Returns the initial at pt point value in bool form.*/
	bool bInitValue(std::vector<int>const& pt)const { return bInitValue(pt.data()); }
	/**\~french Retourne la valeur initiale au point pt sous forme de string.\~english Returns the initial at pt point value in string form.*/
	virtual std::string sInitValue(int const* pt)const { notArrayVariable(); return ""; }
	/**\~french Retourne la valeur initiale au point pt sous forme de string.\~english Returns the initial at pt point value in string form.*/
	std::string sInitValue(std::vector<int>const& pt)const { return sInitValue(pt.data()); }
	/**\~french Retourne toutes les valeurs initiales sous forme du type template (attention, NULL si la variable n'est pas du type demand).\~english Returns all initial values in template type form (attention, NULL if the variable is not of asked type).*/
	template<typename T>T const* initValuesPtr()const { return typeInfo() == typeid(T) && typeid(T)!=typeid(bool) ? reinterpret_cast<T*>(initValuePtr()) : NULL; }
	/**\~french Retourne 0 si la valeur courante est indentique  la valeur initiale et + ou -1 suivant le sens de variation.\~english Returns 0 if the current value is identical to the initial value, and +1 or -1 depending on the direction of change.*/
	virtual int valueChanged()const=0;
	/**\~french Applique la valeur de redmarrage  la valeur courante.\~english Applies the restart value to the current value.*/
	virtual void reset()=0;
	/**\~french Demande la gestion du monitoring de la variable.\~english Requests monitoring management for the variable.*/
	virtual void monitorValue()const=0;
	/**\~french Retourne le type de la variable\~english Returns the type of the variable.*/
	virtual TVarType::EVarType variableType()const;
	DEPRECATED TVarType::EVarType valueType()const { ILogManager::glLogManager().msgObsolete("IVariable::valueType", "variableType"); return variableType(); }
    /**\~french Modifie l'incertitude sur la valeur initiale de la variable.
    Les paramtres sont fournis dans l'ordre prcis pour chaque type de distribution.
    @param type : type de la distribution de probabilit
	\~english Modifies the uncertainty on the initial value of the variable.
	Parameters are provided in the specified order for each distribution type.
	@param type: type of probability distribution
    */
	virtual void setUncertainty(TDistributionType::EDistributionType type,...);
    /**\~french Modifie l'incertitude sur la valeur initiale de la variable.
    Les paramtres sont fournis dans l'ordre prcis pour chaque type de distribution.
    @param type : type de la distribution de probabilit
	@param params : liste des paramtres
	\~english Modifies the uncertainty on the initial value of the variable.
	Parameters are provided in the specified order for each distribution type.
	@param type: type of probability distribution
	@param params: list of parameters
    */
	virtual void setUncertainty(TDistributionType::EDistributionType type,std::vector<double>params);
    /**\~french Modifie les bornes de l'incertitude sur la valeur initiale de la variable.\~english Modifies the bounds of the uncertainty on the initial value of the variable.*/
	virtual void setUncertaintyLimits(double min, double max);
	/**\~french Retourne la distribution utilise pour l'incertitude sur la valeur initiale\~english Returns the distribution used for the uncertainty on the initial value.*/
	virtual TDistributionType::EDistributionType uncertaintyDistribution()const{return TDistributionType::none;}
	/**\~french Retourne le ime paramtre de la loi dans l'ordre standard\~english Returns the ith parameter of the distribution in standard order.*/
	virtual double uncertaintyParameter(size_t i)const;
	/**\~french Retourne le nombre des paramtres de la loi d'incertitude dfinis\~english Returns the number of defined parameters of the uncertainty distribution.*/
	virtual size_t uncertaintyParametersCount()const{return 0;}
	/**\~french Retourne le minimum de l'incertitude\~english Returns the minimum of the uncertainty.*/
	virtual double uncertaintyMin()const{return uncertaintyParameter(0);}//Juste pour le message
	/**\~french Retourne le maximum de l'incertitude\~english Returns the maximum of the uncertainty.*/
	virtual double uncertaintyMax()const{return uncertaintyParameter(0);}//Juste pour le message
	/**\~french Modifie le retard de la variable (uniquement pour les variables retardes)\~english Modifies the delay of the variable (for delayed variables only).*/
	virtual void setDelay(PyTime delay);
	/**\~french Retourne le retard de la variable\~english Returns the delay of the variable.*/
	virtual PyTime delay()const{return 0;}
	/**\~french Retourne le retard initial de la variable\~english Returns the initial delay of the variable.*/
	virtual PyTime initDelay()const{return 0;}
};

/**\~french Interface de gestion des rfrences  des variables.
	Les rfrences sont le reflet de variables situes dans d'autres composants.

	La connexion entre une rfrence et une variable peut tre effectue  l'aide de bind(), mais il est prfrable d'utiliser les botes de messages.
	La connexion par bote de messages s'effectue en ajoutant une rfrence comme import  une bote de messages.

	L'interface permet d'accder au nombre de connexions (cnctCount()) et aux diffrentes valeurs des variables connectes.

	Il est aussi possible d'accder aux variables connectes : reference(i), variable(i), varBase(i).

	exemple1 (definition):
	\~english Interface for managing references to variables.
	References are reflections of variables located in other components.

	The connection between a reference and a variable can be made using bind(), but it is preferable to use message boxes.
	The connection via message box is done by adding a reference as an import to a message box.

	The interface provides access to the number of connections (cnctCount()) and the different values of the connected variables.

	It is also possible to access the connected variables: reference(i), variable(i), varBase(i).

	example1 (definition):
	\~
	~~~~{.cpp}
	class CComp :public CComponent{
	public:
		CComp(char const*name,CSysNamedComp&parent):CComponent(name,parent){
			IReference*ref=addReference("ReferenceName");
			IMessageBox*mb=addMessageBox("MessageBoxName");
			mb->addImport(ref,"AliasName");
		}
	};
	~~~~
	\~french exemple2 (use of references):
	\~english example2 (use of references):
	\~
	~~~~{.cpp}
	IReference*ref=sys.reference("ReferenceName");
	if(ref->cnctCount()>0
		ref->sValue(0);//Get the value of the first connected element as a string
	ref->sumValue(10);//Get the sum of the connected values (return 10 if no connection)
	IVariable*var=ref->cnctCount()>0?ref->variable(0):NULL;
	if(var)
		var->setValue(4.);//Modify the value of the first referenced variable
	CComponent*cpt=ref->cnctCount()>2?ref->component(2):NULL;
	if(cpt)
		cpt->variable("VarName")->dValue();//Get the value of another variable of the third connected component
	~~~~
*/
class PYC_PUBLIC IReference : public IVarBase{
	bool value(CNX_ID i,bool const*)const { return bValue(i); }
	int value(CNX_ID i,int const*)const { return iValue(i); }
	float value(CNX_ID i,float const*)const { return fValue(i); }
	double value(CNX_ID i,double const*)const { return dValue(i); }
	std::complex<double> value(CNX_ID i,std::complex<double>const*)const { return cValue(i); }
	std::string value(CNX_ID i,std::string const*)const { return sValue(i); }
	bool value(CNX_ID i, int const* pt,bool const*)const { return bValue(i, pt); }
	int value(CNX_ID i, int const* pt,int const*)const { return iValue(i, pt); }
	float value(CNX_ID i, int const* pt,float const*)const { return fValue(i, pt); }
	double value(CNX_ID i, int const* pt,double const*)const { return dValue(i, pt); }
	std::complex<double> value(CNX_ID i, int const* pt,std::complex<double>const*)const { return cValue(i, pt); }
	std::string value(CNX_ID i, int const* pt,std::string const*)const { return sValue(i, pt); }
	using IVarBase::typeInfo;
protected:
	IReference(CNamedComp&parent,char const*name):IVarBase(parent,name){}
public:
	/**\~french Retourne le type de la ime connexion\~english Returns the type of the ith connection.*/
	virtual std::type_info const&typeInfo(CNX_ID i)const=0;
	/**\~french Retourne le nombre de valeurs connues par la rfrence (connexions relles ou valeurs fixes).\~english Returns the number of values known by the reference (actual connections or fixed values).*/
	virtual CNX_ID cnctCount()const=0;
	virtual CNX_ID nbCnx()const{return cnctCount();}
	/**\~french Retourne le composant correspondant  la ime connexion s'il peut tre dtermin\~english Returns the component corresponding to the ith connection if it can be determined.*/
	virtual CComponent*component(CNX_ID i)const=0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de complexe.\~english Returns the current value of the ith connection in complex form.*/
	virtual std::complex<double>cValue(CNX_ID i)const = 0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de double.\~english Returns the current value of the ith connection in double form.*/
	virtual double dValue(CNX_ID i)const = 0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de float.\~english Returns the current value of the ith connection in float form.*/
	virtual float fValue(CNX_ID i)const = 0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de int.\~english Returns the current value of the ith connection in int form.*/
	virtual int iValue(CNX_ID i)const = 0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de bool.\~english Returns the current value of the ith connection in bool form.*/
	virtual bool bValue(CNX_ID i)const = 0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de string.\~english Returns the current value of the ith connection in string form.*/
	virtual std::string sValue(CNX_ID i)const = 0;
	/**\~french Retourne la valeur courante de la ime connexion caste au type T.\~english Returns the current value of the ith connection cast to type T.*/
	template<typename T>T value(CNX_ID i)const { return value(i, (T*)NULL); }
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de complexe.\~english Returns the current value at pt point of the ith connection in complex form.*/
	virtual std::complex<double>cValue(CNX_ID i, int const* pt)const = 0;
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de complexe.\~english Returns the current value at pt point of the ith connection in complex form.*/
	std::complex<double>cValue(CNX_ID i, std::vector<int>const&pt)const { return cValue(i, pt.data()); }
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de double.\~english Returns the current value at pt point of the ith connection in double form.*/
	virtual double dValue(CNX_ID i, int const* pt)const = 0;
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de double.\~english Returns the current value at pt point of the ith connection in double form.*/
	double dValue(CNX_ID i, std::vector<int>const& pt)const { return dValue(i, pt.data()); }
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de float.\~english Returns the current value at pt point of the ith connection in float form.*/
	virtual float fValue(CNX_ID i, int const* pt)const = 0;
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de float.\~english Returns the current value at pt point of the ith connection in float form.*/
	float fValue(CNX_ID i, std::vector<int>const& pt)const { return fValue(i, pt.data()); }
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de int.\~english Returns the current value at pt point of the ith connection in int form.*/
	virtual int iValue(CNX_ID i, int const* pt)const = 0;
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de int.\~english Returns the current value at pt point of the ith connection in int form.*/
	int iValue(CNX_ID i, std::vector<int>const& pt)const { return iValue(i, pt.data()); }
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de bool.\~english Returns the current value at pt point of the ith connection in bool form.*/
	virtual bool bValue(CNX_ID i, int const* pt)const = 0;
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de bool.\~english Returns the current value at pt point of the ith connection in bool form.*/
	bool bValue(CNX_ID i, std::vector<int>const& pt)const { return bValue(i, pt.data()); }
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de string.\~english Returns the current value at pt point of the ith connection in string form.*/
	virtual std::string sValue(CNX_ID i, int const* pt)const = 0;
	/**\~french Retourne la valeur courante au point pt de la ime connexion sous forme de string.\~english Returns the current value at pt point of the ith connection in string form.*/
	std::string sValue(CNX_ID i, std::vector<int>const& pt)const { return sValue(i, pt.data()); }
	/**\~french Retourne la valeur courante au point pt de la ime connexion caste au type T.\~english Returns the current value at pt point of the ith connection cast to type T.*/
	template<typename T>T value(CNX_ID i, int const* pt)const { return value(i, pt, (T*)NULL); }
	/**\~french Retourne les valeurs courantes des connexions sous forme de complexe.\~english Returns the current values of the connections in complex form.*/
	virtual CNX_ID cValues(std::complex<double>*tab)const=0;
	/**\~french Retourne les valeurs courantes des connexions sous forme de double.\~english Returns the current values of the connections in double form.*/
	virtual CNX_ID dValues(double*tab)const=0;
	/**\~french Retourne les valeurs courantes des connexions sous forme de float.\~english Returns the current values of the connections in float form.*/
	virtual CNX_ID fValues(float*tab)const=0;
	/**\~french Retourne les valeurs courantes des connexions sous forme d'entiers.\~english Returns the current values of the connections in integer form.*/
	virtual CNX_ID iValues(int*tab)const=0;
	/**\~french Retourne les valeurs courantes des connexions sous forme de boolens.\~english Returns the current values of the connections in boolean form.*/
	virtual CNX_ID bValues(bool*tab)const=0;
	/**\~french Retourne vrai si la ime connexion est associe  une valeur\~english Returns true if the ith connection is associated with a value.*/
	virtual bool isValue(CNX_ID i)const=0;
	/**\~french Retourne la rfrence associe  la ime connexion.\~english Returns the reference associated with the ith connection.*/
	virtual IReference*reference(CNX_ID i)const=0;
	/**\~french Retourne la variable associe  la ime connexion.\~english Returns the variable associated with the ith connection.*/
	virtual IVariable*variable(CNX_ID i)const=0;
	/**\~french Retourne la IVarBase associe  la ime connexion.\~english Returns the IVarBase associated with the ith connection.*/
	virtual IVarBase*varBase(CNX_ID i)const=0;
	/**\~french Prend une quantit res de ressource  la ime variable connecte.\~english Takes a quantity *res* of resource from the ith connected variable.*/
	virtual void takeResource(CNX_ID i,double res)=0;
	/**\~french Restitue une quantit res de ressource  la ime variable connecte.\~english Returns a quantity *res* of resource to the ith connected variable.*/
	virtual void restoreResource(CNX_ID i,double res)=0;
	/**\~french Modifie la valeur de la ime variable connecte.\~english Modifies the value of the ith connected variable.*/
	virtual void setValue(CNX_ID i,double val)=0;
	/**\~french Modifie la drive de la ime variable connecte (uniquement si c'est une variable double).\~english Modifies the derivative of the ith connected variable (only if it is a double variable).*/
	virtual void setDvdt(CNX_ID i,double val)=0;
	/**\~french Lie la rfrence  une autre rfrence ou variable vb\~english Binds the reference to another reference or variable vb.*/
	virtual void bind(IVarBase&vb)=0;
	/**\~french Dlie la rfrence de la rfrence ou de la variable vb\~english Unbinds the reference from the reference or variable vb.*/
	virtual void unbind(IVarBase&vb)=0;
	/**\~french Dlie la ime connexion\~english Unbinds the ith connection.*/
	virtual void unbind(CNX_ID i)=0;
	/**\~french Calcule un ou sur l'ensemble des connexions (def est la valeur en l'absence de connexion)\~english Computes an OR over all connections (def is the value in the absence of connections).*/
	virtual bool orValue(bool def=false)const=0;
	/**\~french Calcule un et sur l'ensemble des connexions (def est la valeur en l'absence de connexion)\~english Computes an AND over all connections (def is the value in the absence of connections).*/
	virtual bool andValue(bool def=true)const=0;
	/**\~french Calcule une somme de l'ensemble des connexions (def est la valeur en l'absence de connexion)\~english Computes a sum of all connections (def is the value in the absence of connections).*/
	virtual double sumValue(double def=0)const=0;
	/**\~french Calcule un produit l'ensemble des connexions (def est la valeur en l'absence de connexion)\~english Computes a product of all connections (def is the value in the absence of connections).*/
	virtual double productValue(double def=1)const=0;
	/**\~french Remplit le vecteur avec les indices qui vrifient la condition\~english Fills the vector with indices that satisfy the condition.*/
	virtual int valuesThat(ICondFctFloat const&cf, std::vector<int> & indexes)const=0;
	virtual int valuesThat(bool(condition)(float), std::vector<int> & indexes)const{std::unique_ptr<ICondFctFloat>cff(ICondFctFloat::newCondFctFloat(condition));return valuesThat(*cff,indexes);}
};

/**\~french Interface de gestion des informations de connexion entre 2 botes de messages.

Les informations de connexion sont un ensemble de proprits associes  des cls textuelles et accessibles  partir des 2 botes de messages connectes.

L'information de connexion est vide lors de la connexion d'une bote demessage  une autre.
Elle peut tre peuple de valeurs de type boolen, entier, double, complexe ou texte.
\~english Interface for managing connection information between two message boxes.

Connection information is a set of properties associated with text keys and accessible from the two connected message boxes.

The connection information is empty when one message box connects to another.
It can be populated with values of boolean, integer, double, complex, or text types.
*/
class PYC_PUBLIC ICnctInfo{
	void value(bool&res,char const*name)const{res=bValue(name);}
	void value(int&res,char const*name)const{res=iValue(name);}
	void value(float&res,char const*name)const{res=fValue(name);}
	void value(double&res,char const*name)const{res=dValue(name);}
	void value(std::complex<double>&res,char const*name)const{res=cValue(name);}
	void value(std::string&res,char const*name)const{res=sValue(name);}
public:
	/**\~french Retourne le type de la proprit de nom name\~english Returns the type of the property named *name*.*/
	virtual std::type_info const&typeInfo(char const*name)const=0;
	/**\~french Retourne le type des valeurs de la proprit de nom name\~english Returns the type of the values of the property named *name*.*/
	virtual TVarType::EVarType variableType(char const* name)const = 0;
	DEPRECATED TVarType::EVarType valueType(char const* name)const { ILogManager::glLogManager().msgObsolete("ICnctInfo::valueType", "variableType"); return variableType(name); }
	/**\~french Retourne la valeur courante de la ime connexion sous forme de complexe.\~english Returns the current value of the ith connection in complex form.*/
	virtual std::complex<double>cValue(char const*name)const=0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de double.\~english Returns the current value of the ith connection in double form.*/
	virtual double dValue(char const*name)const=0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de float.\~english Returns the current value of the ith connection in float form.*/
	virtual float fValue(char const*name)const=0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de int.\~english Returns the current value of the ith connection in int form.*/
	virtual int iValue(char const*name)const=0;
	/**\~french Retourne la valeur courante de la ime connexion sous forme de bool.\~english Returns the current value of the ith connection in bool form.*/
	virtual bool bValue(char const*name)const=0;
	/**\~french Retourne la valeur courante de la connexion caste au type T.\~english Returns the current value of the connection cast to type T.*/
	template<typename T>T value(char const*name)const{
		T res;
		value(res);
		return res;
	}
	/**\~french Retourne la valeur courante de la ime connexion sous forme de string.\~english Returns the current value of the ith connection in string form.*/
	virtual std::string sValue(char const*name)const=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation has not started) in complex form.*/
	virtual void setValue(char const*name,std::complex<double>const&val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de complexe.\~english Modifies the current value (or the initial value if the simulation is not started) in complex form.*/
	inline void setCValue(char const*name,std::complex<double>const&val){setValue(name,val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) in double form.*/
	virtual void setValue(char const*name,double val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de double.\~english Modifies the current value (or the initial value if the simulation is not started) in double form.*/
	inline void setDValue(char const*name,double val){setValue(name,val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) in float form.*/
	virtual void setValue(char const*name,float val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de float.\~english Modifies the current value (or the initial value if the simulation is not started) in float form.*/
	inline void setFValue(char const*name,float val){setValue(name,val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) in int form.*/
	virtual void setValue(char const*name,int val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de int.\~english Modifies the current value (or the initial value if the simulation is not started) in int form.*/
	inline void setIValue(char const*name,int val){setValue(name,val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) in bool form.*/
	virtual void setValue(char const*name,bool val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de bool.\~english Modifies the current value (or the initial value if the simulation is not started) in bool form.*/
	inline void setBValue(char const*name,bool val){setValue(name,val);}
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) in string form.*/
	virtual void setValue(char const*name,char const*val)=0;
	/**\~french Modifie la valeur courante (ou la valeur initiale si la simulation n'est pas lance) sous forme de string.\~english Modifies the current value (or the initial value if the simulation is not started) in string form.*/
	inline void setSValue(char const*name,char const*val){setValue(name,val);}
	/**\~french Supprime la proprit de nom name\~english Deletes the property named *name*.*/
	virtual void delValue(char const*name)=0;
};

/**\~french Interface de gestion des botes de messages.
Les botes de message permettent de connecter les composants entre eux.

Une bote de messages permet d'exporter des variables interne  son composant et d'importer  travers des rfrences (IReference) des valeurs de variables d'autres composants.

La connexion entre 2 botes de message n'est contrle qu' travers le fait que les rfrences importes par une bote de messages peuvent bien tre connectes  des variables exportes par l'autre bote de messages.

Pour augmenter la tolrance de connexion, il est possible de dclarer une importation avec une valeur par dfaut. Cette valeur par dfaut sera attribue  la rfrence si elle n'a pas pu tre connecte.

Les importation et exportation sont dfinies avec un alias. La connexion entre variables exportes et importes est effectue sur la base de l'identit de l'alias.
Chaque connexion entre botes de messages peut tre munie d'un poid qui vaut 0 par dfaut.
Ce poids n'est pas utilis par Pycatsho, mais peut tre restitu par la mthode weight().

Une bote de messages donne accs au nombre de connexions (cnctCount()), botes de messages connectes (cnct(i)) et aux composants connects (component(i))

Une bote de messages permet aussi d'accder directement aux valeurs importes  travers le nom d'alias.

exemple1:
\~english Message box management interface.
Message boxes allow components to connect with each other.

A message box can export internal variables of its component and import values from other components through references (IReference).

The connection between two message boxes is only controlled by the fact that the references imported by one message box can indeed be connected to the variables exported by the other message box.

To increase connection tolerance, it is possible to declare an import with a default value. This default value will be assigned to the reference if it could not be connected.

Imports and exports are defined with an alias. The connection between exported and imported variables is based on the identity of the alias.
Each connection between message boxes can have a weight, which is 0 by default.
This weight is not used by Pycatsho but can be retrieved using the weight() method.

A message box provides access to the number of connections (cnctCount()), connected message boxes (cnct(i)), and connected components (component(i)).

A message box also allows direct access to imported values through the alias name.

example1:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
	CComp(){
		IMessageBox*mb = addMessageBox("MBName");
		IVariable*var = addVariable("VarName",TVarType::t_double,0.);
		IReference*ref = addReference("RefName");
		mb->addExport(var,"AliasExportName");
		mb->addOptionalImport(ref,TVarType::t_double,1.3,"AliasImportName");
	}
};
~~~~
\~french exemple2 flow propagation:
\~english example2 flow propagation:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
	CComp(){
		IMessageBox*mbUp = addMessageBox("UpComponents");
		IMessageBox*mbDown = addMessageBox("DownComponents");
		IVariable*var = addVariable("flow",TVarType::t_double,0.);
		IReference*ref = addReference("input");
		mbDown->addExport(var,"input");
		mbUp->addImport(ref);
		ref->addCallback("propagation",&CComp::propagation);
	}
	void propagation(){
		variable("flow")->setValue(reference("input")->sumValue()); 
	}
};
~~~~
*/
class PYC_PUBLIC IMessageBox:public CNamedComp{
protected:
	IMessageBox(CNamedComp&parent,char const*name):CNamedComp(parent,name){}
public:
	/**\~french Ajoute une variable  exporter et dfinit son alias.\~english Adds a variable to export and defines its alias.*/
	virtual void addExport(IVarBase*p,char const*aliasName=NULL)=0;
	void addExport(IVarBase&p,char const*aliasName=NULL){addExport(&p,aliasName);}
	/**\~french Retourne la liste des noms des alias des variables exportes\~english Returns the list of names of the aliases of the exported variables.*/
	virtual std::vector<std::string>exportedAliases()const=0;
	/**\~french Retourne la variable exporte sous l'alias aliasName (exception si c'est une rfrence).\~english Returns the exported variable under the alias *aliasName* (exception if it is a reference).*/
	virtual IVariable*exportedVariable(char const*aliasName)const=0;
	/**\~french Ajoute une rfrence  importer et dfinit le nom de la variable  lui associer.\~english Adds a reference to import and defines the name of the variable to associate with it.*/
	virtual void addImport(IReference*p,char const*varName=NULL)=0;
	void addImport(IReference&p,char const*varName=NULL){addImport(&p,varName);}
	/**\~french Ajoute une rfrence  connecter facultativement et dfinit sa valeur par dfaut si elle n'est pas connecte.
	@param p : rfrence  importer
	@param type : type de la valeur par dfaut parmi `TVarType`
	@param defaultValue : valeur par dfaut
	@param alias de la variable exporte correspondante
	\~english Adds an optional reference to connect and defines its default value if it is not connected.
	@param p: reference to import
	@param type: type of the default value among `TVarType`
	@param defaultValue: default value
	@param alias: corresponding name of the exported variable
	*/
	virtual void addOptionalImport(IReference*p,int type, double defaultValue, char const*alias)=0;
	void addOptionalImport(IReference&p,int type, double defaultValue, char const*alias){addOptionalImport(&p,type,defaultValue,alias);}
	/**\~french Retourne la liste des noms des alias des rfrences importes\~english Returns the list of names of the aliases of the imported references.*/
	virtual std::vector<std::string>importedAliases()const=0;
	/**\~french Retourne la liste des rfrences importes avec l'alias aliasName\~english Returns the list of imported references with the alias *aliasName*.*/
	virtual std::vector<IReference*>importedReferences(char const*aliasName)const=0;
	/**\~french Retourne vrai si la connexion peut tre effectue avec la bote de messages other\~english Returns true if the connection can be made with the message box *other*.*/
	virtual bool canConnectTo(IMessageBox&other)const=0;
	/**\~french Retourne vrai si la connexion existe dj avec la bote de messages other\~english Returns true if a connection already exists with the message box *other*.*/
	virtual bool isConnectedTo(IMessageBox&other)const=0;
	/**\~french Connecte la bote de messages  une autre bote de messages et dfinit le poids de la connexion.\~english Connects the message box to another message box and sets the weight of the connection.*/
	virtual ICnctInfo&connectTo(IMessageBox&other,double weight)=0;
	/**\~french Connecte la bote de messages  une autre bote de messages avec un poids nul.\~english Connects the message box to another message box with a weight of zero.*/
	ICnctInfo&connectTo(IMessageBox&other){return connectTo(other,0);}//Pour le python
	/**\~french deconnecte la bote de messages d'une autre bote de messages.\~english Disconnects the message box from another message box.*/
	virtual void disconnectFrom(IMessageBox&other)=0;
	/**\~french deconnecte la bote de messages des autres botes de messages.\~english Disconnects the message box from the other message boxes.*/
	virtual void disconnect()=0;
	/**\~french Retourne le nombre de connexions\~english Returns the number of connections.*/
	virtual CNX_ID cnctCount()const=0;
	/**\~french Retourne les informations de la ime connexion\~english Returns the information of the ith connection.*/
	virtual ICnctInfo&cnctInfo(CNX_ID i)const=0;
	/**\~french Retourne les informations de la 1re connexion avec la bote de messages msb\~english Returns the information of the first connection with the message box *msb*.*/
	virtual ICnctInfo&cnctInfo(IMessageBox&msb)const=0;
	/**\~french Retourne le poids de la ime connexion\~english Returns the weight of the ith connection.*/
	virtual double weight(CNX_ID i)const=0;
	/**\~french Retourne le ime composant connect\~english Returns the ith connected component.*/
	virtual CComponent*component(CNX_ID i)const=0;
	/**\~french Retourne les composants connects dans l'ordre des connexions\~english Returns the connected components in the order of the connections.*/
	virtual std::vector<CComponent*>components()const=0;
	/**\~french Retourne la ime bote de message connecte\~english Returns the ith connected message box.*/
	virtual IMessageBox*cnct(CNX_ID i)const=0;
	/**\~french Retourne sous forme de complexe la valeur de la ime variable connecte exporte sous l'alias aliasName.\~english Returns the value of the ith connected exported variable under the alias *aliasName* in complex form.*/
	virtual std::complex<double> cValue(CNX_ID i,char const*aliasName)const=0;
	/**\~french Retourne sous forme de double la valeur la valeur de la ime variable connecte exporte sous l'alias aliasName.\~english Returns the value of the ith connected exported variable under the alias *aliasName* in double form.*/
	virtual double dValue(CNX_ID i,char const*aliasName)const=0;
	/**\~french Retourne sous forme de float la valeur la valeur de la ime variable connecte exporte sous l'alias aliasName.\~english Returns the value of the ith connected exported variable under the alias *aliasName* in float form.*/
	virtual float fValue(CNX_ID i,char const*aliasName)const=0;
	/**\~french Retourne sous forme de int la valeur la valeur de la ime variable connecte exporte sous l'alias aliasName.\~english Returns the value of the ith connected exported variable under the alias *aliasName* in int form.*/
	virtual int iValue(CNX_ID i,char const*aliasName)const=0;
	/**\~french Retourne sous forme de bool la valeur la valeur de la ime variable connecte exporte sous l'alias aliasName.\~english Returns the value of the ith connected exported variable under the alias *aliasName* in bool form.*/
	virtual bool bValue(CNX_ID i,char const*aliasName)const=0;
	/**\~french Retourne sous forme de string la valeur la valeur de la ime variable connecte exporte sous l'alias aliasName.\~english Returns the value of the ith connected exported variable under the alias *aliasName* in string form.*/
	virtual std::string sValue(CNX_ID i,char const*aliasName)const=0;
};

struct SBindingMM;
struct SBindingM;
struct SBinding;

/**\~french Structure de dfinition de l'numr de dfinition des types de loi prdfinis\~english Definition structure of the enumeration for defining predefined distribution types.*/
struct PYC_PUBLIC TLawType{
	/**\~french Enumr de dfinition des types de loi prdfinis dans le coeur de Pycatshoo\~english Enumeration for defining predefined distribution types in the core of Pycatshoo.*/
	typedef enum ELawType{
		unknown=-1,//!<\~french Loi certaine (attention ! pas de paramtre)\~english Certain distribution (note! no parameters).
		inst,//!<\~french Loi instantane (nombre de paramtres compris entre le nombre de sorties -1 et le nombre de sorties)\~english Instantaneous distribution (number of parameters between the number of outputs - 1 and the number of outputs).
		expo,//!<\~french Loi exponentielle (1 paramtre : lambda)\~english Exponential distribution (1 parameter: lambda).
		cstt,//!<\~french Loi temps constant (1 paramtre : temps d'attente)\~english Constant time distribution (1 parameter: waiting time).
		defer=cstt,//!<\~french Loi temps constant (1 paramtre : temps d'attente)\~english Constant time distribution (1 parameter: waiting time).
		cstd,//!<\~french Loi date constante (1 paramtre : date de tirage)\~english Constant date distribution (1 parameter: draw date).
		weib,//!<\~french Loi de Weibull (3 paramtres positifs : chelle, forme par dfaut 1, dcalage temporel par dfaut 0)\~english Weibull distribution (3 positive parameters: scale, shape default 1, time shift default 0).
		Weibull=weib,
		uniform,//!<\~french Loi uniforme (3 paramtres positifs : t min (par dfaut 0), tmax (par dfaut 0), probabilit globale (<=1))\~english Uniform distribution (3 positive parameters: t min (default 0), t max (default 0), global probability (<=1)).
		normal//!<\~french Loi normale (5 paramtres dont 3 positifs : moyenne (par dfaut 0), cart type (par dfaut 0), t min (par dfaut -inf), t max (par dfaut inf), probabilit globale (<=1))\~english Normal distribution (5 parameters including 3 positive: mean (default 0), standard deviation (default 0), t min (default -inf), t max (default inf), global probability (<=1)).
	}ELawType;
	static char const*st_Names[];
	static ELawType fromStr(char const*value);
};
/**\~french Interface de gestion des lois de probabilit.

Cette interface est utilise par les lois prdfinies dans Pycatshoo et doit tre utilise comme base pour la conception de nouvelles lois par l'utilisateur.

Il n'est pas utile de construire une loi de probabilit directement, il faut appeler IDistLaw::newLaw ou ITransition::setDistLaw.

Une loi peut grer nbIndex sorties (si nbIndex > 1 alors le dlai d'attente doit tre nul).
Une loi peut comprendre plusieurs paramtres qui peuvent tre ajouts grce  addParameter().

- Lors de la cration d'une loi grant une unique sortie, il faut surcharger : delay, delayBeyond, delayMax, delayMin (le dlai est tjrs calcul  partir de start_time). Il est utile de surcharger invDistribution pour utiliser l'chantillonnage.
- Lors de la cration d'une loi grant plusieurs sorties, il faut surcharger : index

exemple (cration d'une nouvelle loi):
\~english Probability law management interface.

This interface is used by predefined laws in Pycatshoo and should be used as a basis for designing new laws by the user.

It is not necessary to construct a probability law directly; you should call IDistLaw::newLaw or ITransition::setDistLaw.

A law can manage *nbIndex* outputs (if *nbIndex* > 1, then the waiting time must be zero). A law can have multiple parameters that can be added using addParameter().

- When creating a law managing a single output, you must override: delay, delayBeyond, delayMax, delayMin (the delay is always calculated from start_time). It is useful to override invDistribution to use sampling.
- When creating a law managing multiple outputs, you must override: index.

example (creating a new law):
\~
~~~~{.cpp}
class CUniform :public IDistLaw{
	char const* name()const override { return "Uniform"; }
	int type()const override { return 40; }
	CUniform(CNamedComp& parent, double proba) :IDistLaw(parent), m_NbSamples(1) { setParameter(0., 0); setParameter(0., 1); }

	PyTime delayMin()const override{
		PyTime t = system().currentTime();
		return t <= parameter(0) ? parameter(0) - t : t > parameter(1) ? HUGE_VAL : 0;
	}
	PyTime delayMax()const override{
		PyTime t = system().currentTime();
		return t <= parameter(1) ? parameter(1) - t : HUGE_VAL;
	}
	PyTime delay(unsigned sample)const override{
		return invDistribution((sample + 0.5) / m_NbSamples) - system().currentTime();
	}
	PyTime delayBeyond(PyTime dMin)const {//On suppose que dMin=currentTime - startTime
		PyTime ct = system().currentTime(), tmax = parameter(1);
		PyTime del = ct <= tmax ? system().RNG()->uniform(std::max(parameter(), ct) , tmax) : HUGE_VAL;
		return del < std::max(parameter(), ct) ? HUGE_VAL : del - ct + dMin;
	}
	double density(PyTime t)const override{
		PyTime tmin = parameter();
		if(t < tmin || t>parameter(1) || parameter(1) == tmin)
			return 0;
		return 1 / (parameter(1) - tmin);
	}
	double distribution(PyTime t)const override{
		PyTime tmin = parameter();
		if(t <= tmin)
			return 0;
		if(t > parameter(1))
			return 1;
		return (t - tmin) / (parameter(1) - tmin);
	}
	PyTime CDistLawUniform::invDistribution(double x)const {
		if(x <= 0)
			return parameter(0);
		if(x > 1)
			return HUGE_VAL;
		return parameter(0) + x * (parameter(1) - parameter(0));
	}
	void verify()override{
		if(nbParam() == 2 && parameter() >= 0 && parameter(1) > parameter(0))
			return;//Not to build context
		ILogManager::CCtx ctx(formatMsg("While verifying uniform law of component %s", parent().name()).c_str());
		if(nbParam() != 2)
			ILogManager::glLogManager().throwError("The number of parameters of uniform law is different of 2");
		if(parameter() < 0)
			ILogManager::glLogManager().throwError("The minimal instant is less than 0");
		if(parameter(1) < parameter())
			ILogManager::glLogManager().throwError("The maximal instant is less than the minimal one");
	}
};
~~~~
*/
class PYC_PUBLIC IDistLaw{
	friend class CDistLawInst;
	friend class ITransition;
	friend class CTransition;
	friend class CImportanceSampling;
	friend class CTreeNode;
	friend class CSystemP;
	friend IDistLaw* IDistLaw_newLaw(CNamedComp&, TLawType::ELawType, char const*, PyObject*);
	friend void IDistLaw_setParameter(IDistLaw*, char const*, PyObject*,LAWP_ID);
	LAWP_ID m_NbParam;
	CNamedComp&m_Parent;
	SBindingMM*m_Parameters;//!<\~french Dfinition premire des paramtres permettant de mettre  jour les valeurs numriques\~english Initial definition of parameters allowing the update of numerical values.
	double*m_ParamValues;//!<\~french Valeurs des paramtres utilises pour le calcul du temps d'attente\~english Parameter values used for calculating the waiting time.
	void setParameter(CSysNamedComp& comp, double(CSysNamedComp::* fct)(), LAWP_ID place);
	void setParameter(CSysNamedComp& comp, char const*name,PyObject*pyObj, LAWP_ID place);
	SBindingMM const&parameterBinding(LAWP_ID place)const;//!<\~french Retourne le paramtre ou lance une exception\~english Returns the parameter or throws an exception.
	void reset();//!<\~french Replace les valeurs initiales\~english Replaces the initial values.
	void insertAt(LAWP_ID place,bool bAdd=false);//!<\~french Cre une place en place\~english Creates a place in place.
	void incRef();
	unsigned int m_NbCppRef;//!<\~french Nombre de rfrences d'un objet C++\~english Number of references of a C++ object.
	PyObject*m_Self;//!<\~french Rfrence  l'objet python correspondant (NULL, si c'est un objet purement C++).\~english Reference to the corresponding Python object (NULL if it is a purely C++ object).
	IDistLaw(IDistLaw const&);//Interdit
	IDistLaw&operator=(IDistLaw const&);//Interdit
	/**\~french Lance une exception si les paramtres sont invalides (ne doit tre appel qu'aprs upToDate ou upToDateForModifiers)\~english Throws an exception if the parameters are invalid (should only be called after upToDate or upToDateForModifiers).*/
	virtual void verify();
protected:
	virtual~IDistLaw(void);
	IDistLaw(PyObject*self,CNamedComp&parent);
	static IDistLaw* newLaw(CNamedComp& parent, TLawType::ELawType type, char const* name, PyObject* pVoid) { IDistLaw* dl = newLaw(parent, type, 0.); dl->setParameter(parent,name, pVoid, 0); return dl; }
	CNamedComp&parent()const{return m_Parent;}
	CSystem const&system()const{return m_Parent.system();}
public:
	explicit IDistLaw(CNamedComp&parent);
	/**\~french Donne le nom du type de distribution\~english Returns the name of the distribution type.*/
	virtual char const*name()const{return NULL;}
	/**\~french Retourne le type de loi (parmi TLawType pour les type standard)\~english Returns the type of law (from TLawType for standard types).*/
	virtual int type()const{return TLawType::unknown;}
	void deleteDLaw();//!<\~french Detruit la loi sauf si m_Self est non null\~english Destroys the law unless m_Self is not null.
	/**\~french Dfinit le nombre de paramtres de la loi (par dfaut 1)\~english Sets the number of parameters for the law (default is 1).*/
	void setNbParameters(LAWP_ID nb){if(nb>m_NbParam)insertAt(nb-1);else m_NbParam=nb;}
	/**\~french Insert un paramtre sous forme de valeur.\~english Inserts a parameter as a value.*/
	void insertParameter(double value,LAWP_ID place=0);
	/**\~french Insert un paramtre sous forme de variable.\~english Inserts a parameter as a variable.*/
	void insertParameter(IVariable*var,LAWP_ID place=0);
	void insertParameter(IVariable&var,LAWP_ID place=0){insertParameter(&var,place);}
	/**\~french Dfinit un paramtre par connexion d'une variable.\~english Sets a parameter by connecting a variable.*/
	void setParameter(IVariable*var,LAWP_ID place=0);
	void setParameter(IVariable&var,LAWP_ID place=0){setParameter(&var,place);}
	/**\~french Dfinit un paramtre par une mthode\~english Sets a parameter through a method.*/
	template<typename Type>void setParameter(double (Type::* fct)(), LAWP_ID place = 0) { CSysNamedComp::checkMethodAddrSize(sizeof(fct)); setParameter(m_Parent, static_cast<double (CSysNamedComp::*)()>(fct), place); }
	/**\~french Dfinit un paramtre par une valeur.\~english Sets a parameter by a value.*/
	void setParameter(double value,LAWP_ID place=0);
	/**\~french Retourne la variable associe au paramtre si elle existe (null sinon)\~english Returns the variable associated with the parameter if it exists (null otherwise).*/
	IVariable*parameterVar(LAWP_ID place)const;
	/**\~french Dfinit un coefficient de modification pour un paramtre existant\~english Sets a modification coefficient for an existing parameter.*/
	void setParameterModifier(double coef,LAWP_ID place);
	/**\~french Retourne le modificateur du ime paramtre\~english Returns the modifier of the ith parameter.*/
	double parameterModifier(LAWP_ID place)const;
	/**\~french Retourne la valeur d'un paramtre\~english Returns the value of a parameter.*/
	virtual double parameter(LAWP_ID place=0)const{return m_ParamValues[place];}
	/**\~french Retourne le temps d'attente tir alatoirement avec les paramtres courants de telle sorte que le dlai soit suprieur  dMin.
	* Utile uniquement pour les transitions modifiables ou utilisant des date absolues.
	Pour les lois utilisant des dates absolues, on suppose que dMin = tCourant - tDebutAttenteTransition
	\~english Returns the randomly drawn wait time with the current parameters such that the delay is greater than dMin.  
	* Useful only for modifiable transitions or those using absolute dates.  
	For laws using absolute dates, it is assumed that dMin = tCurrent - tStartWaitTransition.
	*/
	virtual PyTime delayBeyond(PyTime dMin)const{return dMin;}
	/**\~french Retourne le temps d'attente tir alatoirement avec les paramtres courants\~english Returns the waiting time randomly drawn with the current parameters.*/
	virtual PyTime delay()const{return delayBeyond(0.);}
	/**\~french Retourne le temps minimal d'attente\~english Returns the minimum waiting time.*/
	virtual PyTime delayMin()const{return 0.;}
	/**\~french Retourne le temps minimal d'attente\~english Returns the minimum waiting time.*/
	virtual PyTime delayMax()const{return HUGE_VAL;}
	/**\~french Retourne la valeur de la densit de probabilit pour un tirage  partir de la date 0\~english Returns the value of the probability density function for a draw starting from date 0.*/
	virtual double density(PyTime t)const{return 0.;}
	/**\~french Retourne la valeur de la fonction de rpartition pour un tirage  partir de la date 0\~english Returns the value of the cumulative distribution function for a draw starting from date 0.*/
	virtual double distribution(PyTime t)const{return 0.;}
	/**\~french Retourne la valeur de la fonction de rpartition inverse pour un tirage  partir de la date 0\~english Returns the value of the inverse cumulative distribution function for a draw starting from date 0.*/
	virtual PyTime invDistribution(double x)const {return 0.;}
	/**\~french Retourne l'indexe de la transition de sortie tir alatoirement avec les paramtres courants ( partir de 0 jusqu' nbIndex-1)
	@param inhibited liste des indices des tats inhibs
	\~english Returns the index of the output transition randomly drawn with the current parameters (from 0 to nbIndex-1).
	@param inhibited list of indices of the inhibited states.
	*/
	virtual TGT_ID index(std::vector<TGT_ID>inhibited)const{return 0;}
	/**\~french Retourne la probabilit de tirer la sortie d'indice ind
	@param ind indice de la sortie souhaite
	@param inhibited liste des indices des tats inhibs
	\~english Returns the probability of selecting the output with index *ind*.
	@param ind: index of the desired output
	@param inhibited: list of indices of the inhibited states.
	*/
	virtual double outputProbability(TGT_ID ind,std::vector<TGT_ID>inhibited)const{return 1;}
	/**\~french Modifie le nombre d'chantillons de la loi utiliss pour l'exploration systmatique de squences.
	Dclenche une exception pour les lois qui ne grent pas cet chantillonnage.
	\~english Modifies the number of samples of the law used for the systematic exploration of sequences.  
	Throws an exception for laws that do not handle this sampling.*/
	virtual void setNbSamples(unsigned nb);
	/**\~french Retourne le nombre d'chantillons de la loi utiliss pour l'exploration systmatique de squences.\~english Returns the number of samples of the law used for the systematic exploration of sequences.*/
	virtual unsigned nbSamples()const{return 1;}
	/**\~french Retourne le dlai de tirage du sample me chantillon (dbut  0) : retourne le dlai standard par dfaut\~english Returns the draw delay of the *n*th sample (starting from 0): returns the default standard delay.*/
	virtual PyTime delay(unsigned sample)const{return delay();}
	/**\~french Retourne le nombre de paramtres de la loi\~english Returns the number of parameters of the law.*/
	inline LAWP_ID nbParam()const{return m_NbParam;}
	/**\~french Retourne le nombre de sorties gres par la loi.\~english Returns the number of outputs managed by the law.*/
	virtual TGT_ID nbIndex()const{return 1;}
	/**\~french Retourne faux si les paramtres ont chang\~english Returns false if the parameters have changed.*/
	bool upToDate();
	/**\~french Retourne faux si les paramtres ont chang (pas de recalcul de la valeur de base)\~english Returns false if the parameters have changed (no recalculation of the base value).*/
	bool upToDateForModifiers();
	/**\~french Construit une des lois prdfinies et dfinit le premier paramtre par une variable\~english Constructs one of the predefined laws and sets the first parameter with a variable.*/
	static IDistLaw*newLaw(CNamedComp&parent,TLawType::ELawType type,IVariable&var);
	static IDistLaw*newLaw(CNamedComp&parent,TLawType::ELawType type,IVariable*var){return newLaw(parent,type,*var);}
	/**\~french Construit une des lois prdfinies et dfinit le premier paramtre par une valeur\~english Constructs one of the predefined laws and sets the first parameter with a value.*/
	static IDistLaw*newLaw(CNamedComp&parent,TLawType::ELawType type,double param);
	/**\~french Construit une des lois prdfinies et dfinit le premier paramtre par une fonction\~english Constructs one of the predefined laws and sets the first parameter with a function.*/
	template<typename Type>static IDistLaw*newLaw(CNamedComp&parent,TLawType::ELawType type,double (Type::*fct)()){CSysNamedComp::checkMethodAddrSize(sizeof(fct));IDistLaw*dl=newLaw(parent,type,0.);dl->setParameter(fct);return dl;}
};

/**\~french Structure de dfinition de l'numr de dfinition de l'aspect modifiable de la transition.\~english Definition structure of the enumeration for defining the modifiable aspect of the transition.*/
struct TModificationMode{
	/**\~french Enumr de dfinition de l'aspect modifiable de la transition.\~english Enumeration for defining the modifiable aspect of the transition.*/
	typedef enum EModificationMode{
		not_modifiable,//!<\~french Transition non modifiable\~english Non-modifiable transition.
		not_modif=not_modifiable,//!<\~french Transition non modifiable\~english Non-modifiable transition.
		discrete_modification,//!<\~french Transition modifiable  chaque pas\~english Transition modifiable at each step.
		modif=discrete_modification,//!<\~french Transition modifiable  chaque pas\~english Transition modifiable at each step.
		modifiable=modif,//!<\~french Transition modifiable  chaque pas\~english Transition modifiable at each step.
		continuous_modification,//!<\~french Transition modifiable continment (dans les ODE)\~english Continuously modifiable transition (in ODEs).
		cont_modif=continuous_modification,//!<\~french Transition modifiable continment (dans les ODE)\~english Continuously modifiable transition (in ODEs).
		cont_modifiable=continuous_modification//!<\~french Transition modifiable continment (dans les ODE)\~english Continuously modifiable transition (in ODEs).
	}EModificationMode;
};
/**\~french Structure de dfinition de l'numr de dfinition de type de franchissement de la transition.\~english Definition structure of the enumeration for defining the transition crossing type.*/
struct PYC_PUBLIC TTransType{
	/**\~french Enumr de dfinition de type de franchissement de la transition.\~english Enumeration for defining the transition crossing type.*/
	typedef enum ETransType{
		trans=1,//!<\~french Transition standard\~english Standard transition.
		fault=2,//!<\~french Dfaillance\~english Failure.
		rep=3//!<\~french Rparation\~english Repair.
	}ETransType;
	static char const*st_Names[];
	static ETransType fromStr(char const*value);
};
/**\~french Structure de dfinition de l'numr de dfinition de type d'vnement de base.\~english Definition structure of the enumeration for defining the basic event type.*/
struct PYC_PUBLIC TBEType{
	/**\~french Enumr de dfinition de type d'vnement de base.\~english Enumeration for defining the basic event type.*/
	typedef enum EBEType{
		unknown=0,//!<\~french Inconnu : permet de supprimer le modle d'EB\~english Unknown: allows removing the EB model.
		probability,//!<\~french Simple probabilit\~english Simple probability.
		exponential,//!<\~french Exponentielle avec temps de mission\~english Exponential with mission time.
		GLM,//!<\~french Gamma Lambda Mu avec temps de mission\~english Gamma Lambda Mu with mission time.
		Weibull,//!<\~french Weibull avec temps de mission\~english Weibull with mission time.
		periodic4,//!<\~french Test priodiquement avec 4 paramtres (cf OpenPSA)\~english Periodically tested with 4 parameters (see OpenPSA).
		periodic5,//!<\~french Test priodiquement avec 5 paramtres (cf OpenPSA)\~english Periodically tested with 5 parameters (see OpenPSA).
		periodic11//!<\~french Test priodiquement avec 11 paramtres (cf OpenPSA)\~english Periodically tested with 11 parameters (see OpenPSA).
	}EBEType;
	static char const*st_Names[];
	static EBEType fromStr(char const*value);
};
/**\~french Interface de gestion des transitions d'un tat  un autre.
	Une transition permet de contrler le passage d'un tat  un autre.
	Une transition gre :
	- une condition par l'intermdiaire d'une rfrence m_Condition,
	- des paramtres de loi par l'intermdiaire de la rfrence m_Parameters.
	Une transition connat son tat de dpart et ses tats d'arrive potentiels.
	Il ne peut y avoir plusieurs tats d'arrive que pour une transition INS.

	Les modles actuellement reconnus sont :
	- INS : les paramtres sont les proba de passage dans les tats cibles (le dernier paramtre peut tre ignor)
	- EXP : l'unique paramtre est le taux de passage par unit de temps (seconde)
	- TC  : l'unique paramtre est la dure d'attente en seconde

	La transition peut tre qualife de :
	- trans : transition standard,
	- def : dfaillance,
	- rep : rparation.

	La transition peut tre :
	- interruptible (si la condition est invalide pendant l'attente, l'attente est interrompue),
	- modifiable (si les paramtres sont modifis pendant l'attente, celle-ci est recalcule).

	La transition peut tre utilise en gnration d'ADD (par dfaut avec une condition vraie) sa condition peut tre modifie par setFTCondition
	Dans ce cas, il est possible de modifier l'vnement de base reprsentatif de la transition par setBEType et setBEParameter
	Une transition peut tre rserve  l'unique gnration d'ADD en utilisant une condition sous la forme d'une variable NULL.

exemple:
\~english Interface for managing transitions from one state to another.
A transition controls the passage from one state to another.
A transition manages:
- a condition through the reference *m_Condition*,
- law parameters through the reference *m_Parameters*.

A transition knows its starting state and its potential target states.
There can only be multiple target states for an INS transition.

Currently recognized models are:
- **INS**: the parameters are the probabilities of transitioning to the target states (the last parameter can be ignored),
- **EXP**: the only parameter is the rate of transition per unit time (second),
- **TC**: the only parameter is the waiting time in seconds.

The transition can be classified as:
- **trans**: standard transition,
- **def**: failure,
- **rep**: repair.

The transition can be:
- **interruptible** (if the condition is invalidated during the wait, the wait is interrupted),
- **modifiable** (if the parameters are modified during the wait, it is recalculated).

The transition can be used in the generation of ADD (by default with a true condition). Its condition can be modified with *setFTCondition*.
In this case, it is possible to modify the basic event representing the transition using *setBEType* and *setBEParameter*.
A transition can be reserved exclusively for ADD generation by using a condition in the form of a NULL variable.

example:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
	CComp(char const*name,CSysNamedComp&parent):CCompoenet(name,parent){
		IState*st1 = addState("StateName1");
		IState*st2 = addState("StateName2");
		IState*st3 = addState("StateName3");
		ITransition*trDef = st1->addTransition("TransNameDef");
		trDef->setDistLaw(TLawType::expo,0.001);
		trDef->addTarget(st2,TTransType::fault);//Failure st1 => st2
		ITransition*trRep = st2->addTarget("TransNameRep");
		trRep->setDistLaw(TLawType::inst,0.01);
		trRep->addTarget(st3,TTransType::fault);//Failure to repaire st2 => st3 probability 0.01
		trRep->addTarget(st1,TTransType::rep);//Repaire st2 => st1 probability 1 - 0.01
	}
};
~~~~
*/
class PYC_PUBLIC ITransition : public CNamedComp,public IMonitored{
	friend class CTreeNode;
	friend void ITransition_addCallback(ITransition*, char const*, PyObject*, int);
	friend void ITransition_setCondition(ITransition*, char const*, PyObject*, bool);
	friend void ITransition_setBEParameter(ITransition*, char const*, PyObject*, BEP_ID);
	friend IDistLaw* ITransition_setDistLaw(ITransition*, TLawType::ELawType, char const*, PyObject*);
protected:
	SBindingM&m_Condition;//!<\~french Dfinition de la condition de franchissement de la transition\~english Definition of the transition crossing condition.
	bool m_NCond;//!<\~french Si vrai, la condition est utilise  l'envers\~english If true, the condition is used in reverse.
	bool m_NCondInit;//!<\~french Si vrai, la condition est initialement utilise  l'envers\~english If true, the condition is initially used in reverse.
	bool m_Interruptible;//!<\~french Si vrai, la transition peut tre interrompue si la condition n'est plus valide (par dfaut faux)\~english If true, the transition can be interrupted if the condition is no longer valid (default is false).
	TModificationMode::EModificationMode m_Modifiable;//!<\~french Si 1 la transition peut tre modifie aprs son lancement, si 2 elle est continuement modifiable (intgration) (par dfaut non modifiable)\~english If 1, the transition can be modified after it starts; if 2, it is continuously modifiable (integration) (default is non-modifiable).
	IDistLaw*m_DistLaw;//!<\~french Loi de distribution des probabilits\~english Probability distribution law.
	virtual void addCallback(char const* name, void(CSysNamedComp::* method)(), PyObject* pyMth, int sens) = 0;
	void setCondition(bool(CNamedComp::*fct)(),bool negate);
	void setBEParameter(double(CNamedComp::*fct)(),BEP_ID place);
	virtual SBinding const&BEParameter(BEP_ID place)const=0;//!<\~french Retourne le lien du paramtre ou lance une exception s'il n'existe pas\~english Returns the parameter link or throws an exception if it does not exist.
	virtual SBinding&BEParameter(BEP_ID place)=0;//!<\~french Retourne le lien du paramtre en signalant une modification du modle ou lance une exception s'il n'existe pas\~english Returns the parameter link while signaling a model modification, or throws an exception if it does not exist.
	virtual void setBEParameter(char const* name, PyObject* pyObj, BEP_ID place);
	IDistLaw* setDistLaw(TLawType::ELawType type, char const* name, PyObject* pVoid) { setDistLaw(IDistLaw::newLaw(*parent(), type, name, pVoid)); return distLaw(); }
	void setCondition(char const* name, PyObject* pyObj, bool negate = false);
	ITransition(CNamedComp&parent,char const*name);
	virtual~ITransition();
public:
	char const*basename()const override{return CNamedComp::basename();}
	char const*name()const override{return CNamedComp::name();}
	CNamedComp*parent()const override{return CNamedComp::parent();}
	/** \fn setTrace
	\~french
	modifie le niveau de trace
	@param trace : niveau souhait :
	- 1 : trace les franchissements de la transition
	- 2 : trace en plus les dcisions d'activation de la transition
	- 3 : trace en plus les dcisions de non activation ou de conservation de l'activation
	- >3 : trace en plus le dtail du tirage de l'index de sortie
	\~english  
	Modifies the trace level.  
	@param trace: desired level:  
	- 1: traces the transitions' crossings  
	- 2: additionally traces the activation decisions of the transition  
	- 3: additionally traces the decisions of non-activation or preservation of activation  
	- >3: additionally traces the details of the output index drawing.*/
	inline void setTrace(int trace){IMonitored::setTrace(trace);}//Uniquement pour valider la doc
	/**\~french Retourne la loi de probabilit de la transition\~english Returns the probability law of the transition.*/
	IDistLaw*distLaw()const{return m_DistLaw;}
	/**\~french Modifie la loi de probabilit de la transition\~english Modifies the probability law of the transition.*/
	void setDistLaw(IDistLaw*distLaw);
	/**\~french Modifie la loi de probabilit de la transition
	@param type type de la nouvelle loi
	@param param valeur du premier paramtre de la loi
	\~english Modifies the probability law of the transition.  
	@param type: type of the new law  
	@param param: value of the first parameter of the law.*/
	IDistLaw*setDistLaw(TLawType::ELawType type,double param){setDistLaw(IDistLaw::newLaw(*parent(),type,param));return distLaw();}
	/**\~french Modifie la loi de probabilit de la transition
	@param type type de la nouvelle loi
	@param param variable donnant la valeur du premier paramtre de la loi
	\~english Modifies the probability law of the transition.  
	@param type: type of the new law  
	@param param: variable providing the value of the first parameter of the law.*/
	IDistLaw*setDistLaw(TLawType::ELawType type,IVariable&param){setDistLaw(IDistLaw::newLaw(*parent(),type,param));return distLaw();}
	IDistLaw*setDistLaw(TLawType::ELawType type,IVariable*param){setDistLaw(IDistLaw::newLaw(*parent(),type,*param));return distLaw();}
	/**\~french Modifie la loi de probabilit de la transition
	@param type type de la nouvelle loi
	@param fct fonction donnant la valeur du premier paramtre de la loi
	\~english Modifies the probability law of the transition.  
	@param type: type of the new law  
	@param fct: function providing the value of the first parameter of the law.*/
	template<typename Type>IDistLaw*setDistLaw(TLawType::ELawType type,double(Type::*fct)()){checkMethodAddrSize(sizeof(fct));setDistLaw(IDistLaw::newLaw(*parent(),type,fct));return distLaw();}
	/**\~french Modifie le caractre interruptible de la transition (par dfaut faux).\~english Modifies the interruptible nature of the transition (default is false).*/
	void setInterruptible(bool interruptible){m_Interruptible=interruptible;}
	/**\~french Retourne le caractre interruptible de la transition.\~english Returns the interruptible nature of the transition.*/
	bool interruptible()const{return m_Interruptible;}
	/**\~french Modifie le caractre modifiable de la transition (par dfaut non modifiable).\~english Modifies the modifiable nature of the transition (default is non-modifiable).*/
	void setModifiable(TModificationMode::EModificationMode modifiable){m_Modifiable=modifiable;}
	/**\~french Retourne le caractre modifiable de la transition.\~english Returns the modifiable nature of the transition.*/
	TModificationMode::EModificationMode modifiable()const{return m_Modifiable;}
	/**\~french Retourne l'tat de dpart de la transition\~english Returns the starting state of the transition.*/
	virtual IState*startState()const=0;
	/**\~french Retourne le nombre de cibles de la transition\~english Returns the number of target states of the transition.*/
	virtual TGT_ID targetCount()const=0;
	/**\~french Insert un tat cible  la transition et dfinit le type de la transition vers cet tat.\~english Inserts a target state into the transition and defines the type of transition to that state.*/
	virtual void insertTarget(IState*end,int type,int place)=0;
	void insertTarget(IState&end,int type,int place){insertTarget(&end,type,place);}
	/**\~french Ajoute un tat cible  la transition et dfinit le type de la transition vers cet tat.\~english Adds a target state to the transition and defines the type of transition to that state.*/
	virtual void addTarget(IState*end,int type=TTransType::trans)=0;
	void addTarget(IState&end,int type=TTransType::trans){addTarget(&end,type);}
	DEPRECATED IState*getTarget(TGT_ID ind)const;//!<\~french Retourne l'tat cible d'indice i (obsolte).\~english Returns the target state at index i (obsolete).
	virtual IState*target(TGT_ID i)const=0;//!<\~french Retourne l'tat cible d'indice i.\~english Returns the target state at index i.
	virtual char const*targetName(TGT_ID ind)const=0;//!<\~french Retourne le nom de l'tat cible d'indice i ou NULL\~english Returns the name of the target state at index i, or NULL.
	DEPRECATED int getTargetType(TGT_ID ind)const;//!<\~french Retourne le type de franchissement effectu pour atteindre le ime tat cible (obsolte).\~english Returns the type of crossing made to reach the ith target state (obsolete).
	virtual int targetType(TGT_ID ind)const=0;//!<\~french Retourne le type de franchissement effectu pour atteindre le ime tat cible.\~english Returns the type of crossing made to reach the ith target state.
	virtual char const* targetDesc(TGT_ID ind)const = 0;//!<\~french Retourne la description de la cible d'indice i ou NULL\~english Returns the description of the target at index i, or NULL.
	virtual void setTargetDesc(char const* desc, TGT_ID ind) = 0;//!<\~french Modifie la description de la cible d'indice i\~english Modifies the description of the target at index i.
	/**\~french Inhibe ou dsinhibe les sorties vers l'tat\~english Inhibits or uninhibits the outputs to the state.*/
	virtual void inhibateTarget(IState*st,bool inhibate)=0;
	void inhibateTarget(IState&st,bool inhibate){inhibateTarget(&st,inhibate);}
	/**\~french Inhibe ou dsinhibe la sortie vers les cibles de type type\~english Inhibits or uninhibits the output to targets of type *type*.*/
	virtual void inhibateTarget(int type,bool inhibate)=0;
	/**\~french Retourne vrai si la 1re cible correspondant  l'tat st est inhibe\~english Returns true if the first target corresponding to the state *st* is inhibited.*/
	virtual bool isTargetInhibited(IState*st)const=0;
	/**\~french Retourne vrai si la cible d'indice ind est inhibe\~english Returns true if the target at index *ind* is inhibited.*/
	virtual bool isTargetInhibited(TGT_ID ind)const=0;
	/**\~french Renseigne la condition par connexion  une variable ou  un tat.\~english Sets the condition by connecting to a variable or a state.*/
	void setCondition(IVariable*var,bool negate=false);
	void setCondition(IVariable&var,bool negate=false){setCondition(&var,negate);}
	/**\~french Renseigne la condition par une fonction boolenne.\~english Sets the condition with a boolean function.*/
	template<typename FctType> void setCondition(FctType fct,bool negate=false){checkMethodAddrSize(sizeof(fct));setCondition(static_cast<bool(CNamedComp::*)()>(fct),negate);}
	/**\~french Renseigne la condition par une valeur boolenne.\~english Sets the condition with a boolean value.*/
	void setCondition(bool val);
	/**\~french Retourne vrai si la transition est utilise en simulation\~english Returns true if the transition is used in simulation.*/
	bool usedInSimulation()const;
	/**\~french Retourne la valeur actuelle de la condition.\~english Returns the current value of the condition.*/
	DEPRECATED bool getCondition()const;//!<\~french Retourne la valeur actuelle de la condition (obsolte).\~english Returns the current value of the condition (obsolete).
	bool conditionValue()const;//!<\~french Retourne la valeur actuelle de la condition.\~english Returns the current value of the condition.
	/**\~french Retourne la date de franchissement prvue (ou -1 s'il n'y a pas de date prvue).\~english Returns the expected crossing date (or -1 if no expected date is available).*/
	virtual PyTime endTime()const=0;
	/**\~french Retourne la date de dcision d'activation de la transition\~english Returns the activation decision date of the transition.*/
	virtual PyTime startTime()const=0;
	/**\~french Modifie le dlai d'attente\~english Modifies the waiting time.*/
	virtual void setDelay(PyTime delay)=0;
	/**\~french Retourne l'indice de l'tat de sortie choisi ou -1 s'il n'y a pas d'tat prvu\~english Returns the index of the chosen exit state, or -1 if no state is planned.*/
	virtual int indOutState()const=0;
	/**\~french Modifie l'indice de l'tat de sortie\~english Modifies the index of the exit state.*/
	virtual void setIndOutState(unsigned ind)=0;
	/**\~french Ajoute une mthode sensible au tirage de la transition
	@param name nom de la mthode
	@param method adresse de la mthode (la mthode doit faire partie de l'objet propritaire de la transition)
	@param outState numro de sortie sensible (si -1, toutes)
	\~english Adds a method sensitive to the transition drawing.  
	@param name: name of the method  
	@param method: address of the method (the method must be part of the object owning the transition)  
	@param outState: output number that is sensitive (if -1, all).*/
	template<typename FctType>void addCallback(char const* name, FctType method, int outState = -1) { checkMethodAddrSize(sizeof(method)); addCallback(name, static_cast<void(CSysNamedComp::*)()>(method), NULL, outState); }
	template<typename FctType>DEPRECATED void addSensitiveMethod(char const* name, FctType method, int outState = -1) { ILogManager::glLogManager().msgObsolete("ITransition::addSensitiveMethod", "addCallback");  addCallback(name, method, outState); }
	/**\~french Supprime une mthode sensible au tirage de la transition\~english Removes a method sensitive to the transition drawing.*/
	virtual void removeCallback(char const* name) = 0;
	DEPRECATED void removeSensitiveMethod(char const* name);
	/**\~french Retourne l'indice de l'tat de sortie atteint lors du tirage de la transition (ou -1 si la transition n'a pas t tire)\~english Returns the index of the exit state reached during the transition drawing (or -1 if the transition was not drawn).*/
	virtual int firedState()const=0;
	/**\~french Force  recalculer le dlai d'attente\~english Forces the recalculation of the waiting time.*/
	virtual void invalidate()=0;
	/**\~french Modifie le masque des tats de sortie  monitorer
	@param mask : expression rgulire adapte aux noms des tats de sortie monitors
	\~english Modifies the mask for the exit states to be monitored.  
	@param mask: regular expression adapted to the names of the monitored exit states.*/
	virtual void setMonitoredOutStateMask(char const*mask)=0;
	/**\~french Retourne le masque des tats de sortie  monitorer\~english Returns the mask for the exit states to be monitored.*/
	virtual char const*monitoredOutStateMask()const=0;
	/**\~french Modifie la condition textuelle (ddie  la gnration d'ADD)\~english Modifies the textual condition (dedicated to ADD generation).*/
	virtual void setConditionFT(char const*cond)=0;
	/**\~french Retourne la condition textuelle (ddie  la gnration d'ADD)\~english Returns the textual condition (dedicated to ADD generation).*/
	virtual std::string conditionFT()const=0;
	/**\~french Demande l'interprtation de la condition\~english Requests the interpretation of the condition.*/
	DEPRECATED virtual bool parseConditionFT()=0;
	/**\~french Modifie le type de l'vnement de base.
	Si le type de l'vnement de base est inconnu, le modle de l'EB est dduit de la loi de probabilit de la transition.
	\~english Modifies the type of the basic event.  
	If the type of the basic event is unknown, the EB model is inferred from the probability law of the transition.*/
	virtual void setBEType(TBEType::EBEType type)=0;
	/**\~french Retourne le type de l'vnement de base utilis en gnration d'ADD\~english Returns the type of the basic event used in ADD generation.*/
	virtual TBEType::EBEType BEType()const=0;
	/**\~french Retourne la valeur du placeme paramtre de l'vnement de base\~english Returns the value of the placeholder parameter of the basic event.*/
	double BEParameterValue(BEP_ID place)const;
	/**\~french Dfinit un paramtre de l'EB par connexion d'une variable.\~english Sets a parameter of the basic event (EB) by connecting a variable.*/
	void setBEParameter(IVariable*var,BEP_ID place);
	void setBEParameter(IVariable&var,BEP_ID place){setBEParameter(&var,place);}
	/**\~french Dfinit un paramtre par une mthode\~english Sets a parameter through a method.*/
	template<typename Type>void setBEParameter(double (Type::*fct)(),BEP_ID place){checkMethodAddrSize(sizeof(fct));setBEParameter(static_cast<double (CNamedComp::*)()>(fct),place);}
	/**\~french Dfinit un paramtre par une valeur.\~english Sets a parameter by a value.*/
	virtual void setBEParameter(double value,BEP_ID place);
	/**\~french Dfinit une stratgie par dfaut pour cette transition\~english Sets a default strategy for this transition.*/
	virtual void setDefaultStrategy(std::vector<int>strat)=0;
	virtual std::vector<int> defaultStrategy()const=0;//!<\~french Retourne la stratgie par dfaut de la transition\~english Returns the default strategy of the transition.
	/**\~french Dfinit une stratgie pour cette transition\~english Sets a strategy for this transition.*/
	virtual void setStrategy(std::vector<int>strat)=0;
	virtual std::vector<int> strategy()const=0;//!<\~french Retourne la stratgie courante de la transition\~english Returns the current strategy of the transition.
};

/**\~french Interface de gestion des tats.
	Les tats permettent de construire le diagramme d'tats d'un composant  l'aide des transitions.
	Un tat appartient toujours  un unique automate. Cet automate peut tre cr explicitement par l'utilisateur ou construit automatiquement par Pycatshoo.
	Un tat peut tre actif ou non (valeur boolenne vrai ou faux). Un tat actif active ses transitions de sortie.
	Un tat peut tre export dans une bote de messages comme une variable.
	Un tat possde un indice entier qui peut ne pas tre unique au sein d'un automate.

exemple:
\~english State management interface.
States are used to build the state diagram of a component using transitions.
A state always belongs to a unique automaton. This automaton can either be explicitly created by the user or automatically built by Pycatshoo.
A state can be active or inactive (boolean value true or false). An active state activates its exit transitions.
A state can be exported to a message box as a variable.
A state has an integer index, which may not be unique within an automaton.

example:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
	CComp(char const*name,CSysNamedComp&parent):CComponent(name,parent){
		IState*st1 = addState("StateName1");//Add a state in an automaton built by PyCATSHOO
		IState*st2 = st1->automaton()->addState("StateName2",3);//Add a state in the same automaton as st1
		IState*st3 = addState("StateName3");//Add a state in an automaton built by PyCATSHOO
		ITransition*tr = st1->addTransition("TransName");
		tr->addTarget(st3);//Since st1 and st3 are related, the automata of st1 and st3 will be merged
		setInitState(st1);//Set the initial state of the automaton groupiing st1, st2 and st3
	}
};
~~~~
*/
class PYC_PUBLIC IState:public virtual IVariable{
protected:
	~IState(){}
public:
	/**\~french Ajoute une transition de sortie de l'tat.\~english Adds an exit transition from the state.*/
	virtual ITransition*addTransition(char const*name)=0;
	/**\~french Retourne vrai si l'tat est actif.\~english Returns true if the state is active.*/
	virtual bool isActive()const{return bValue();}
	/**\~french Retourne la date de sortie de l'tat la plus proche (en seconde).\~english Returns the closest exit time of the state (in seconds).*/
	virtual double exitTime()const=0;
	/**\~french Retourne l'automate contenant l'tat.\~english Returns the automaton containing the state.*/
	virtual IAutomaton*automaton()const=0;
	/**\~french Retourne l'indice de l'tat (doit tre unique dans l'automate).\~english Returns the index of the state (must be unique within the automaton).*/
	virtual unsigned int index()const=0;
	/**\~french Retourne la liste des transitions au dpart de l'tat.\~english Returns the list of transitions originating from the state.*/
	virtual std::vector<ITransition*>transitions()const=0;
};

/**\~french Interface de gestion des automates d'tats.
	Un automate gre un diagramme d'tats d'un composant. Il peut y avoir plusieurs automates dans un composant, mais une transition ne peut pas relier des tats d'automates diffrents.
	Il existe toujours un unique tat actif dans un automate.
	Un automate peut tre export dans une bote de messages comme une variable.
	Un automate peut restituer l'tat courant ou son indice.
	La valeur courante d'un automate est l'indice de l'tat actif.

exemple:
\~english Automaton state management interface.
An automaton manages a state diagram of a component. There can be multiple automatons within a component, but a transition cannot connect states from different automatons.
There is always a unique active state within an automaton.
An automaton can be exported to a message box as a variable.
An automaton can return the current state or its index.
The current value of an automaton is the index of the active state.

example:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
	CComp(char const*name,CSysNamedComp&parent):CComponent(name,parent){
		IAutomaton*aut = addAutomaton("AutomatonName");
		IState*st1 = aut->addState("StateName1", 2);//Add a state of index 1 in the automaton AutomatonName
		IState*st2 = st1->automaton()->addState("StateName2",3);//Add a state in the same automaton as st1
		IState*st3 = addState("StateName3");//Add a state in an automaton built by PyCATSHOO
		ITransition*tr = st1->addTransition("TransName");
		tr->addTarget(st3);//Since st1 and st3 are related, the automata of st1 and st3 will be merged (PyCATSHOO will keep the named automaton : AutomatonName)
		aut->setInitState(st3);//Modify the initial state of the automaton
	}
};
~~~~
*/
class PYC_PUBLIC IAutomaton:public virtual IVariable{
public:
	/** \fn setTrace
	\~french modifie le niveau de trace
	@param trace : niveau souhait = cumul de :
	- 1 : trace les modifications de l'indice de l'tat courant
	\~english Modifies the trace level.  
	@param trace: desired level = combination of:  
	- 1: traces the changes in the index of the current state.*/
	inline void setTrace(int trace){IMonitored::setTrace(trace);}//Uniquement pour valider la doc
	virtual size_t nbStates()const=0;//!<\~french Retourne le nombre d'tats de l'automate\~english Returns the number of states in the automaton.
	virtual std::vector<IState*>states()const=0;//!<\~french Retourne la liste des tats de l'automate
	/**\~french Modifie l'tat initial de l'automate.\~english Returns the list of states in the automaton.*/
	virtual void setInitState(IState*st)=0;
	void setInitState(IState&st){setInitState(&st);}
	/**\~french Retourne l'tat initial par dfaut ou NULL.\~english Returns the default initial state or NULL.*/
	virtual IState*defaultState()const=0;
	/**\~french Retourne l'tat initial ou NULL.\~english Returns the initial state or NULL.*/
	virtual IState*initState()const=0;
	/**\~french Retourne l'indice de l'tat courrent ou -1.\~english Returns the index of the current state or -1.*/
	virtual int currentIndex()const=0;
	/**\~french Retourne l'tat courant ou NULL.\~english Returns the current state or NULL.*/
	virtual IState*currentState()const=0;
	/**\~french Ajoute un tat  l'automate.
	@param name nom de l'tat ajout
	@param index indice de l'tat ajout
	\~english Adds a state to the automaton.  
	@param name: name of the added state  
	@param index: index of the added state.*/
	virtual IState*addState(char const*name,int index)=0;
	/**\~french Retourne le premier tat d'indice index\~english Returns the first state with index *index*.*/
	virtual IState*state(int index)const=0;
	/**\~french Ecrit le contenu de l'automate dans un fichier interprtable par graphviz\~english Writes the content of the automaton to a file that can be interpreted by Graphviz.*/
	virtual bool writeGraph(char const* fileName, bool bTrNames)const = 0;
};

/**\~french Interface de manipulation des tapes.
Une tape permet de regrouper des mthodes sensibles de manire  ordonner leur appel prioritairement  l'instant d'activation.
Une liste d'tapes peut tre constitue au sein du systme : CSystem::addStep()
Le systme contient toujours une tape par dfaut qui contient toutes les mthodes qui ne sont pas assignes  une autre tape.
Les mthodes appartenant  l'tape sont dfinie par la donne de l'adresse de leur objet propritaire et de leur nom (l'objet peut tre NULL : toutes les mthode du nom sont dans l'tape).
Chaque tape peut avoir une utilisation conditionne
Les tapes sont actives dans l'ordre de dclaration. L'tape par dfaut est active en dernier.
Au sein de l'tape, les mthodes sont appeles dans l'ordre d'activation.
Chaque tape utilisateur peut tre conditionne. La condition de l'tape n'est value que juste avant l'activation de l'tape. Une fois la condition valide,toutes les mthodes de l'tape sont actives.

exemple:
\~english Step manipulation interface.
A step allows grouping sensitive methods in such a way that their call is prioritized at the activation time.
A list of steps can be created within the system using *CSystem::addStep()*.
The system always contains a default step that holds all methods not assigned to another step.
The methods belonging to a step are defined by providing the address of their owning object and their name (the object can be NULL: all methods with that name are in the step).
Each step can have a conditional usage.
Steps are activated in the order of declaration. The default step is activated last.
Within a step, methods are called in the order of activation.
Each user step can be conditional. The condition of the step is evaluated just before the step is activated. Once the condition is validated, all methods in the step are activated.

example:
\~
~~~~{.cpp}
class CComp:public CComponent{
public:
	CComp(char const*name,CSysNamedComp&parent):CComponent(name,parent){
		IStep*step2 = addStep("StepName1");//Add a step
		step2->addMethod(this, "MethodName");//Add the method MethodName in the step
		IStep*step1 = system().insertStep("StepName2",0);//Insert the step at the first place
		step1->setCondition(*this,&CComp::condStep1);//Step1 is conditionned
	}
	bool condStep1();//Condition function
};
~~~~

*/
class PYC_PUBLIC IStep : public CSysNamedComp{
	friend class PySystem;
	friend void IStep_setCondition(IStep*, CSysNamedComp&, char const*, PyObject*, bool);
protected:
	SBindingM&m_Condition;//!<\~french Dfinition de la condition d'utilisation de l'tape\~english Definition of the step's usage condition.
	bool m_NCond;//!<\~french Si vrai, la condition est utilise  l'envers\~english If true, the condition is used in reverse.
	bool m_NCondInit;//!<\~french Si vrai, la condition est initialement utilise  l'envers\~english If true, the condition is initially used in reverse.
	IStep(CSystem&sys, char const*name);
	virtual ~IStep();
	void setCondition(CSysNamedComp&nc, bool(CSysNamedComp::*fct)(),bool negate);
	void setCondition(CSysNamedComp& nc, char const* name, PyObject* pyObj, bool negate = false);
public:
	/**\~french Ajoute une mthode  l'tape\~english Adds a method to the step.*/
	virtual void addMethod(CSysNamedComp const* cmp, char const* methodName) = 0;
	virtual void addMethod(CSysNamedComp const& cmp, char const* methodName) { addMethod(&cmp, methodName); };
	/**\~french Renseigne la condition par connexion  une variable ou  un tat.\~english Sets the condition by connecting to a variable or a state.*/
	void setCondition(IVariable*var,bool negate=false);
	void setCondition(IVariable&var,bool negate=false){setCondition(&var,negate);}
	/**\~french Renseigne la condition par une fonction boolenne.
	@param nc objet propritaire de la mthode
	@param fct mthode boolenne constituant la condition
	@param negate si vrai, la condition est valide si la fonction prend la valeur faux
	\~english Sets the condition with a boolean function.  
	@param nc: object owning the method  
	@param fct: boolean method constituting the condition  
	@param negate: if true, the condition is validated if the function returns false.*/
	template<typename Type> void setCondition(Type&nc,bool(Type::*fct)(),bool negate=false){checkMethodAddrSize(sizeof(fct));setCondition(nc, static_cast<bool(CNamedComp::*)()>(fct),negate);}
	/**\~french Renseigne la condition par une fonction boolenne.
	@param nc objet propritaire de la mthode
	@param name nom de la mthode de la mthode
	@param fct mthode boolenne constituant la condition
	@param negate si vrai, la condition est valide si la fonction prend la valeur faux
	\~english Sets the condition with a boolean function.  
	@param nc: object owning the method  
	@param name: name of the method  
	@param fct: boolean method constituting the condition  
	@param negate: if true, the condition is validated if the function returns false.*/
	template<typename Type> void setCondition(Type&nc,char const*name,bool(Type::*fct)(),bool negate=false){checkMethodAddrSize(sizeof(fct));setCondition(nc, static_cast<bool(CNamedComp::*)()>(fct),negate);}
	/**\~french Renseigne la condition par une valeur boolenne.\~english Sets the condition with a boolean value.*/
	void setCondition(bool val);
	/**\~french Retourne la valeur actuelle de la condition.\~english Returns the current value of the condition.*/
	DEPRECATED bool getCondition()const;//!<\~french Retourne la valeur actuelle de la condition (obsolte).\~english Returns the current value of the condition (obsolete).
	bool conditionValue()const;//!<\~french Retourne la valeur actuelle de la condition.\~english Returns the current value of the condition.
};

/**\~french Structure de dfinition de l'numr de choix de l'algorithme d'intgration.\~english Definition structure of the enumeration for choosing the integration algorithm.*/
struct PYC_PUBLIC TSchemaType{
	/**\~french Enumr de choix de l'algorithme d'intgration.\~english Enumeration for choosing the integration algorithm.*/
	typedef enum ESchemaType{
		euler=1,//!<\~french Algorithme d'euler\~english Euler's algorithm.
		runge_kutta4,//!<\~french Runge kutta d'ordre 4\~english Fourth-order Runge-Kutta method.
		runge_kutta_cash_karp54,//!<\~french Runge kutta d'ordre 4/5\~english Fourth/fifth-order Runge-Kutta method.
		runge_kutta_dopri5,//!<\~french Runge kutta d'ordre 5\~english Fifth-order Runge-Kutta method.
		runge_kutta_fehlberg78,//!<\~french Runge kutta fehlberg\~english Runge-Kutta-Fehlberg method.
		modified_midpoint,//!<\~french Point milieu modifi\~english Modified midpoint method.
		controlled_rk_cash_karp54,//!<\~french Runge kutta  pas variable\~english Runge-Kutta method with variable step size.
		controlled_rk_dopri5,//!<\~french Runge kutta  pas variable\~english Runge-Kutta method with variable step size.
		controlled_rk_fehlberg78,//!<\~french Runge kutta  pas variable\~english Runge-Kutta method with variable step size.
		controlled_bulirsch_stoer,//!<\~french Bulirsch Stoer naturellement  pas variable\~english Bulirsch-Stoer method with variable step size.
		dense_rk_dopri5,//!<\~french Runge kutta dense et  pas variable\~english Dense Runge-Kutta method with variable step size.
		dense_bulirsch_stoer,//!<\~french Bulirsch Stoer dense et  pas variable\~english Dense Bulirsch-Stoer method with variable step size.
		backward_differentiation_formulas,//!<\~french BDF avec la bibliothque Sundials\~english BDF (Backward Differentiation Formula) method with the Sundials library.
		adams_moulton,//!<\~french Adams Moulton avec la bibliothque Sundials\~english Adams-Moulton method with the Sundials library.
		implicit_differential_algebraic,//!<\~french Equations diffrentielles algbriques implicites\~english Implicit algebraic differential equations.
		newton_krilov,//!<\~french Equations algbriques implicites\~english Implicit algebraic equations.
		IDA_NK//!<\~french Equations diffrentielles algbriques implicites ou quations algbriques implicites\~english Implicit differential-algebraic equations or implicit algebraic equations.
	}ESchemaType;
	static char const*st_Names[];
	static ESchemaType fromStr(char const*value);//!<\~french Lit le type d'algorithme dans une chane au format du fichier XML de paramtres\~english Reads the algorithm type from a string in the XML parameter file format.
};

/**\~french Structure de dfinition de l'numr de choix d'une contrainte sur une variable intgre.\~english Definition structure of the enumeration for choosing a constraint on an integrated variable.*/
struct PYC_PUBLIC TVarConstraintType{
	/**\~french Enumr de choix d'une contrainte sur une variable intgre\~english Enumeration for choosing a constraint on an integrated variable.*/
	typedef enum EVarConstraintType{
		negative_strict=-2,//!<\~french Strictement ngative\~english Strictly negative.
		negative,//!<\~french Ngative ou nulle\~english Negative or zero.
		none,//!<\~french Nulle\~english Zero.
		positive,//!<\~french Positive ou nulle\~english Positive or zero.
		positive_strict//!<\~french Strictement positive\~english Strictly positive.
	}EVarConstraintType;
};

/**\~french Interface de gestion des solveurs ODE de Pycatshoo.
Une ODE Pycatshoo est constitue d'un ensemble de variables modifies pendant l'intgration et d'un ensemble d'quations d'intgration.

Une quation d'intgration est une mthode sans argument qui modifie les variables de 2 manires :
- en modifiant la drive des variables d'ODE grce IVariable::setDvdtODE()
- en modifiant la valeur des variables  explicites
Par dfaut, les quations sont appeles dans l'ordre alphabtique, mais il est possible de leur attribuer un ordre (par dfaut 0).
Les quations sont alors effectue dans l'ordre croissant de l'ordre.

L'intgration est automatiquement stoppe lors des tirages de transitions, mais il est possible d'ajouter des conditions d'arrt sous 2 formes :
- des mthodes relles prenant une valeur positive pour demander l'arrt de l'intgration
- des transitions dont la condition est utilise comme condition d'arrt

Les transitions continuement variables actives sont automatiquement prises en compte par le gestionnaire de PDMP courant.

Il est possible de demander l'appel de mthodes avant le dbut et aprs l'arrt de l'intgration addBeginMethod(), addEndMethod()

exemple1:
\~english Pycatshoo ODE solver management interface.
A Pycatshoo ODE consists of a set of variables that are modified during integration and a set of integration equations.

An integration equation is a method with no arguments that modifies variables in two ways:
- by modifying the derivative of the ODE variables using *IVariable::setDvdtODE()*
- by modifying the values of explicit variables

By default, the equations are called in alphabetical order, but they can be assigned a specific order (default is 0).
The equations are then performed in ascending order of their specified order.

Integration is automatically stopped during transition draws, but stop conditions can be added in two forms:
- real methods that return a positive value to request the stop of integration
- transitions whose condition is used as a stop condition

Continuously active variable transitions are automatically accounted for by the current PDMP manager.

It is possible to request the call of methods before the start and after the stop of the integration using *addBeginMethod()* and *addEndMethod()*.

example1:
\~
~~~~{.cpp}
class CComp:public CComponent{
	IVariable*level,*surface;
	IReference*flow;
public:
	CComp(char const*name,CSysNamedComp&parent):CComponent(name,parent){
		surface = addVariable("Surface",TVarType::t_double,10.);
		level = addVariable("level",TVarType::t_double,0.);
		flow = addReference("flow");
		IMessageBox*inputs=addMessageBox("inputs);
		inputs->addImport(flow,"flow");
		IState*initial = addState("initial");
		IState*overflow = addState("overflow");
		ITransition*trOverflow = initial->addTransition("overflow");
		trOverflow->addTarget(overflow);
		IPDMPManager*pdmp = addPDMPManager("PDMP1");//Add a PDMP manager (may be already created)
		pdmp->addDifferentialVariable(level);
		pdmp->addEquationMethod(this,"levelODE",&CComp::levelODE);
	}
	void levelODE(){//Equation method
		level->setDvDtODE(surface->dValue()*flow->sumValue());
	}
};
~~~~
\~french exemple2 (explicit equations):
\~english example2 (explicit equations):
\~
~~~~{.cpp}
class Planete :public CComponent
{
	IVariable*x,*y,*vx,*vy,*m;//Variables (coordonnes x,y, vitesses vx,vy et masse)
	IReference*ix,*iy,*im;//Rfrences coordonnes x,y et masse des autres plantes
	//Constructeur de la classe
	Planete(char const*name,CSystem&system):CComponent(name,system){
		//Construction des variables
		x=addVariable("X",t_double,0);
		y= addVariable("Y",t_double,0);
		vx= addVariable("VX",t_double,0);
		vy= addVariable("VY",t_double,0);
		m= addVariable("M",t_double,1);
		//Construction des rfrences
		ix=addReference("X");
		iy= addReference("Y");
		im= addReference("M");
		//Construction de la bote de messages permettant de relier les diffrentes plantes
		IMessageBox*mb=addMessageBox("Gravit");
		//Ajout des variables locales comme lments exports
		mb->addExport(x,"X");
		mb->addExport(y,"Y");
		mb->addExport(m,"M");
		//Ajout des rfrences comme lments imports (elles seront connectes aux elements imports correspondants des autres plantes)
		mb->addImport(ix,"X");
		mb->addImport(iy,"Y");
		mb->addImport(im,"M");
		//Construction du gestionnaire d'ODE si ce n'est dj fait
		IPDMPManager*p=addPDMPManager("Trajectoire") ;
		//Ajout des variables x et y comme variables gres par les ODE
		p->addODEVariable(*x) ;
		p-> addODEVariable (*y) ;
		//Ajout de la fonction de calcul des drives
		p->addEquationMethod("attraction",*this,&Planete::attraction,0) ;
		//Ajout de la fonction condition de poursuite du calcul
		p->addCondition("collision",*this,&Planete::collision) ;
	}
	void attraction(){
		double ax=0,ay=0,ldx,ldy,r;
		//Calcul de l'acclration
		for(int i=0 ;i<ix->nbCnx() ;i++){
			ldx=ix->dValue(i)-x->dValue() ;
			ldy=iy->dValue(i)-y->dValue() ;
			r=sqrt(ldx*ldx+ldy*ldy) ;
			ax+=im->dValue(i)*ldx/r/r/r ;
			ay+=im->dValue(i)*ldy/r/r/r ;
		}
		//Mise  jour des drives
		vx->setDvdtODE(ax) ;
		vy->setDvdtODE(ay) ;
		x->setDvdtODE(vx->dValue());
		x->setDvdtODE(vy->dValue());
	}
	double collision(){
		double ldx,ldy,r,rmin=10000;
		//Calcul de la distance minimale avec les autres plantes
		for(int i=0 ;i<ix->nbCnx() ;i++){
			ldx=ix->dValue(i)-x->dValue() ;
			ldy=iy->dValue(i)-y->dValue() ;
			r=sqrt(ldx*ldx+ldy*ldy) ;
			if(r<rmin)
				rmin=r ;
		}
		//Retour de la valeur de la condition (>0 si la distance min est infrieure  10)
		return 10-rmin ;
	}
};
~~~~
\~french exemple2 (implicit equations):
\~english example2 (implicit equations):
\~
~~~~{.cpp}
class Planete :public CComponent
{
	IVariable*x,*y,*vx,*vy,*m;//Variables (coordonnes x,y, vitesses vx,vy et masse)
	IReference*ix,*iy,*im;//Rfrences coordonnes x,y et masse des autres plantes
	//Constructeur de la classe
	Planete(char const*name,CSystem&system):CComponent(name,system){
		//Construction des variables
		x=addVariable("X",t_double,0);
		y= addVariable("Y",t_double,0);
		vx= addVariable("VX",t_double,0);
		vy= addVariable("VY",t_double,0);
		m= addVariable("M",t_double,1);
		//Construction des rfrences
		ix=addReference("X");
		iy= addReference("Y");
		im= addReference("M");
		//Construction de la bote de messages permettant de relier les diffrentes plantes
		IMessageBox*mb=addMessageBox("Gravit");
		//Ajout des variables locales comme lments exports
		mb->addExport(x,"X");
		mb->addExport(y,"Y");
		mb->addExport(m,"M");
		//Ajout des rfrences comme lments imports (elles seront connectes aux elements imports correspondants des autres plantes)
		mb->addImport(ix,"X");
		mb->addImport(iy,"Y");
		mb->addImport(im,"M");
		//Construction du gestionnaire d'ODE si ce n'est dj fait
		IPDMPManager*p=addPDMPManager("Trajectoire") ;
		//Ajout des variables x et y comme variables gres par les ODE
		p->addDifferentialVariable(*x) ;
		p-> addDifferentialVariable (*y) ;
		//Ajout de la fonction de calcul des rsidus
		p->addEquationMethod("attraction",*this,&Planete::attraction,0) ;
		//Ajout de la fonction condition de poursuite du calcul
		p->addCondition("collision",*this,&Planete::collision) ;
		//Choix du schma adapt
		p->setSchema(TSchemaType::implicit_differential_algebraic) ;
	}
	void attraction(){
		double ax=0,ay=0,ldx,ldy,r;
		//Calcul de l'acclration
		for(int i=0 ;i<ix->nbCnx() ;i++){
			ldx=ix->dValue(i)-x->dValue() ;
			ldy=iy->dValue(i)-y->dValue() ;
			r=sqrt(ldx*ldx+ldy*ldy) ;
			ax+=im->dValue(i)*ldx/r/r/r ;
			ay+=im->dValue(i)*ldy/r/r/r ;
		}
		//Mise  jour des rsidus
		vx->setResidualValue(ax-vx->dvdtODE()) ;
		vy->setResidualValue(ay-vy->dvdtODE()) ;
		x->setResidualValue(vx->dValue()-x->dvdtODE());
		y->setResidualValue(vy->dValue()-y->dvdtODE());
	}
	double collision(){
		double ldx,ldy,r,rmin=10000;
		//Calcul de la distance minimale avec les autres plantes
		for(int i=0 ;i<ix->nbCnx() ;i++){
			ldx=ix->dValue(i)-x->dValue() ;
			ldy=iy->dValue(i)-y->dValue() ;
			r=sqrt(ldx*ldx+ldy*ldy) ;
			if(r<rmin)
				rmin=r ;
		}
		//Retour de la valeur de la condition (>0 si la distance min est infrieure  10)
		return 10-rmin ;
	}
};
~~~~
*/
class PYC_PUBLIC IPDMPManager:public CNamedComp
{
	friend class CComponent;
	friend void IPDMPManager_addBeginMethod(IPDMPManager*, char const*, CSysNamedComp&, PyObject*);
	friend void IPDMPManager_addEndMethod(IPDMPManager*, char const*, CSysNamedComp&, PyObject*);
	friend void IPDMPManager_addPreStepMethod(IPDMPManager*, char const*, CSysNamedComp&, PyObject*);
	friend void IPDMPManager_addPostStepMethod(IPDMPManager*, char const*, CSysNamedComp&, PyObject*);
	friend void IPDMPManager_addEquationMethod(IPDMPManager*, char const*, CSysNamedComp&, PyObject*, int);
	friend void IPDMPManager_addJacobianMethod(IPDMPManager*, char const*, CSysNamedComp&, PyObject*, int);
	friend void IPDMPManager_addCondition(IPDMPManager*, char const*, CSysNamedComp&, PyObject*);
protected:
	IPDMPManager(CSysNamedComp&system,char const*name):CNamedComp(system,name),m_Trace(0){}

	virtual void addEquationMethod(CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(),PyObject*pyMth,int order)=0;//!<\~french Ajoute une fonction de calcul des drives des variables\~english Adds a function to calculate the derivatives of the variables.
	virtual void addJacobianMethod(CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(),PyObject*pyMth,int order)=0;//!<\~french Ajoute une fonction de calcul du jacobien variables\~english Adds a function to calculate the Jacobian of the variables.
	virtual void addBeginMethod(CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(),PyObject*pyMth)=0;//!<\~french Ajoute une fonction  appeler avant l'intgration\~english Adds a function to be called before the integration.
	virtual void addEndMethod(CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(char const*),PyObject*pyMth)=0;//!<\~french Ajoute une fonction  appeler aprs l'intgration\~english Adds a function to be called after the integration.
	virtual void addPreStepMethod(CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(char const*),PyObject*pyMth)=0;//!<\~french Ajoute une fonction  appeler avant l'intgration\~english Adds a function to be called before the integration.
	virtual void addPostStepMethod(CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(char const*),PyObject*pyMth)=0;//!<\~french Ajoute une fonction  appeler aprs l'intgration\~english Adds a function to be called after the integration.
	virtual void addCondition(CSysNamedComp*comp,char const*name,double(CSysNamedComp::*method)(),PyObject*pyMth)=0;//!<\~french Ajoute une fonction de calcul de condition d'arrt de l'intgration\~english Adds a function to calculate the stop condition for the integration.
	virtual void addCondition(CTransition*tr)=0;//!<\~french Ajoute une transition utilise pour sa condition\~english Adds a transition used for its condition.
	int m_Trace;
public:
	/**\~french Modifie l'algorithme d'intgration (par dfaut Runge kutta d'ordre 4).\~english Modifies the integration algorithm (default is Fourth-order Runge-Kutta).*/
	virtual void setSchema(TSchemaType::ESchemaType schema)=0;
	/**\~french Retourne l'algorithme d'intgration utilis.\~english Returns the integration algorithm used.*/
	virtual TSchemaType::ESchemaType schema()const=0;
	/**\~french Modifie le pas d'intgration initial (par dfaut 0.01).
	Ce pas peut tre modifi au cours d'une intgration  pas variable, mais est repris aprs chaque arrt de l'intgration
	\~english Modifies the initial integration step (default is 0.01).  
	This step can be modified during a variable-step integration, but it is reset after each stop of the integration.*/
	virtual void setDt(double dt)=0;
	DEPRECATED double getDt()const;//!<\~french Retourne le pas d'intgration initial (obsolte)\~english Returns the initial integration step (obsolete).
	virtual double dt()const=0;//!<\~french Retourne le pas d'intgration initial\~english Returns the initial integration step.
	/**\~french Modifie le pas d'intgration minimum (par dfaut 0).\~english Modifies the minimum integration step (default is 0).*/
	virtual void setDtMin(double dtMin)=0;
	DEPRECATED double getDtMin()const;//!<\~french Retourne le pas minimum d'intgration (obsolte)\~english Returns the minimum integration step (obsolete).
	virtual double dtMin()const=0;//!<\~french Retourne le pas minimum d'intgration\~english Returns the minimum integration step.
	/**\~french Modifie le pas d'intgration maximum (par dfaut l'infini).\~english Modifies the maximum integration step (default is infinity).*/
	virtual void setDtMax(double dtMax)=0;
	DEPRECATED double getDtMax()const;//!<\~french Retourne le pas maximum d'intgration (obsolte)\~english Returns the maximum integration step (obsolete).
	virtual double dtMax()const=0;//!<\~french Retourne le pas maximum d'intgration\~english Returns the maximum integration step.
	/**\~french Modifie la prcision de recherche des intersections (par dfaut 0.001).\~english Modifies the precision of intersection search (default is 0.001).*/
	virtual void setDtCond(double dt)=0;
	DEPRECATED double getDtCond()const;//!<\~french Retourne la prcision de recherche des intersections (obsolte)\~english Returns the precision of intersection search (obsolete).
	virtual double dtCond()const=0;//!<\~french Retourne la prcision de recherche des intersections\~english Returns the precision of intersection search.
	/**\~french Modifie la priodicit de mmorisation des variables (par dfaut 0.1).
	Le pas de mmorisation effectu est toujours plus grand que le pas d'intgration (attention au intgrations  pas variable)
	\~english Modifies the periodicity of variable memorization (default is 0.1).  
	The memorization step taken is always larger than the integration step (be cautious with variable-step integrations).*/
	virtual void setDtMem(double dt)=0;
	DEPRECATED double getDtMem()const;//!<\~french Retourne le pas de mmorisation des variables (obsolte)\~english Returns the variable memorization step (obsolete).
	virtual double dtMem()const=0;//!<\~french Retourne le pas de mmorisation des variables\~english Returns the variable memorization step.
	/**\~french Modifie l'option de stockage systmatique des variables (par dfaut false).\~english Modifies the option for systematic storage of variables (default is false).*/
	virtual void setAlwaysMem(bool bAlways)=0;
	virtual bool alwaysMem()const=0;//!<\~french Retourne l'option de stockage systmatique des variables\~english Returns the option for systematic storage of variables.
	/**\~french Modifie l'option d'utilisation d'un jacobien anlytique pour rsoudre les systmes non linaires\~english Modifies the option to use an analytical Jacobian to solve nonlinear systems.*/
	virtual void setUseJacobian(bool bUse)=0;
	virtual bool useJacobian()const=0;//!<\~french Retourne vrai si un jacobien analytique est utilisable par Sundials pour rsoudre les systmes non linaires\~english Returns true if an analytical Jacobian can be used by Sundials to solve nonlinear systems.
	/**\~french Modifie le paramtre de nom name\~english Modifies the parameter named *name*.*/
	virtual void setParameter(char const*name,double value)=0;
	/**\~french Retourn la valeur du paramtre de nom name\~english Returns the value of the parameter named *name*.*/
	virtual double parameter(char const*name)const=0;
	/**\~french Ajoute une variable  grer par le systme d'ODE.\~english Adds a variable to be managed by the ODE system.*/
	virtual void addODEVariable(IVariable*var)=0;
	void addODEVariable(IVariable&var){addODEVariable(&var);}
	/**\~french Ajoute une variable  grer par le systme diffrentiel (idem addODEVariable)\~english Adds a variable to be managed by the differential system (same as addODEVariable).*/
	void addDifferentialVariable(IVariable&var){addODEVariable(&var);}
	void addDifferentialVariable(IVariable*var){addODEVariable(var);}
	/**\~french Ajoute une variable  grer par le systme algbrique\~english Adds a variable to be managed by the algebraic system.*/
	virtual void addAlgebraicVariable(IVariable*var)=0;
	void addAlgebraicVariable(IVariable&var){addAlgebraicVariable(&var);}
	/**\~french Ajoute une variable  calculer en mme temps que le systme d'ODE.\~english Adds a variable to be computed alongside the ODE system.*/
	virtual void addExplicitVariable(IVariable*var)=0;
	void addExplicitVariable(IVariable&var){addExplicitVariable(&var);}
	/**\~french Ajoute un FMU utilis en mme temps que le systme d'ODE (l'ordre est donn en concurrence avec les mthodes quation).\~english Adds an FMU to be used alongside the ODE system (the order is given in parallel with the equation methods).*/
	virtual void addFMU(IFunctionalMockupUnit*FMU)=0;
	void addFMU(IFunctionalMockupUnit&FMU){addFMU(&FMU);}
	/**\~french Modifie les contraintes sur la variable var\~english Modifies the constraints on the variable *var*.*/
	virtual void setVariableConstraint(IVariable*var,TVarConstraintType::EVarConstraintType constraint)=0;
	void setVariableConstraint(IVariable&var,TVarConstraintType::EVarConstraintType constraint){setVariableConstraint(&var,constraint);}
	/**\~french Retourne la contrainte  laquelle est soumise la variable var\~english Returns the constraint to which the variable *var* is subjected.*/
	virtual TVarConstraintType::EVarConstraintType variableConstraint(IVariable const*var)const=0;
	/**\~french Ajoute une fonction  appeler avant l'intgration\~english Adds a function to be called before the integration.*/
	template<typename Type> void addBeginMethod(char const*name,Type&comp,void(Type::*beginFct)()){checkMethodAddrSize(sizeof(beginFct));addBeginMethod(&comp,name,static_cast<void(CSysNamedComp::*)()>(beginFct),NULL);}
	/**\~french Ajoute une fonction  appeler aprs l'intgration\~english Adds a function to be called after the integration.*/
	template<typename Type> void addEndMethod(char const*name,Type&comp,void(Type::*endFct)(char const*)){checkMethodAddrSize(sizeof(endFct));addEndMethod(&comp,name,static_cast<void(CSysNamedComp::*)(char const*)>(endFct),NULL);}
	/**\~french Ajoute une fonction  appeler avant chaque pas d'intgration\~english Adds a function to be called before each integration step.*/
	template<typename Type> void addPreStepMethod(char const*name,Type&comp,void(Type::*fct)()){checkMethodAddrSize(sizeof(fct));addPreStepMethod(&comp,name,static_cast<void(CSysNamedComp::*)()>(fct),NULL);}
	/**\~french Ajoute une fonction  appeler aprs chaque pas d'intgration\~english Adds a function to be called after each integration step.*/
	template<typename Type> void addPostStepMethod(char const*name,Type&comp,void(Type::*fct)()){checkMethodAddrSize(sizeof(fct));addPostStepMethod(&comp,name,static_cast<void(CSysNamedComp::*)()>(fct),NULL);}
	/**\~french Ajoute une fonction d'quation diffrentielle
	@param name nom de la mthode
	@param comp composant propritaire de la mthode
	@param odeFct mthode de calcul de l'ODE
	@param order ordre d'appel de la mthode
	\~english Adds a differential equation function.  
	@param name: name of the method  
	@param comp: owner component of the method  
	@param odeFct: method for calculating the ODE  
	@param order: order of calling the method.*/
	template<typename Type> void addEquationMethod(char const*name,Type&comp,void(Type::*odeFct)(),int order=0){checkMethodAddrSize(sizeof(odeFct));addEquationMethod(&comp,name,static_cast<void(CSysNamedComp::*)()>(odeFct),NULL,order);}
	/**\~french Ajoute une fonction de calcul de condition\~english Adds a function to calculate the condition.*/
	template<typename Type> void addCondition(char const*name,Type&comp,double(Type::*condFct)()){checkMethodAddrSize(sizeof(condFct));addCondition(&comp,name,static_cast<double(CSysNamedComp::*)()>(condFct),NULL);}
	template<typename Type> void addBoundaryCheckerMethod(char const*name,Type&comp,double(Type::*condFct)()){checkMethodAddrSize(sizeof(condFct));addCondition(&comp,name,static_cast<double(CSysNamedComp::*)()>(condFct),NULL);}
	/**\~french Ajoute une fonction de calcul du jacobien
	@param name nom de la mthode
	@param comp composant propritaire de la mthode
	@param jacobianFct mthode de calcul du jacobien
	@param order ordre d'appel de la mthode
	\~english Adds a function to calculate the Jacobian.  
	@param name: name of the method  
	@param comp: owner component of the method  
	@param jacobianFct: method for calculating the Jacobian  
	@param order: order of calling the method.*/
	template<typename Type> void addJacobianMethod(char const*name,Type&comp,void(Type::*jacobianFct)(),int order=0){checkMethodAddrSize(sizeof(jacobianFct));addJacobianMethod(&comp,name,static_cast<void(CSysNamedComp::*)()>(jacobianFct),NULL,order);}
	/**\~french Ajoute une transition utilise pour sa condition\~english Adds a transition used for its condition.*/
	void addWatchedTransition(ITransition&tr);
	/**\~french Modifie la trace du PDMP
	@param trace niveau de trace : 0 aucune, !=0 informations sur le droulement de l'intgration
	\~english Modifies the trace of the PDMP.  
	@param trace: trace level: 0 for none, !=0 for information about the integration process.*/
	void setTrace(int trace){m_Trace=trace;}
	int trace()const{return m_Trace;}
	/**\~french Retourne vrai si la variable fait partie des variable explicites du PDMP\~english Returns true if the variable is part of the explicit variables in the PDMP.*/
	virtual bool isExplicitVariable(IVariable const*var)const=0;
	/**\~french Retourne vrai si la variable fait partie des variable d'ODE du PDMP\~english Returns true if the variable is part of the ODE variables in the PDMP.*/
	virtual bool isODEVariable(IVariable const*var)const=0;
	/**\~french Retourne vrai si la variable fait partie des variable algbriques du PDMP\~english Returns true if the variable is part of the algebraic variables in the PDMP.*/
	virtual bool isAlgebraicVariable(IVariable const*var)const=0;
};

/**\~french Classe de calcul d'une quation d'un systme linaire
Les quations doivent tre remplies par les mthodes de calcul communiques au systme d'quation.
Une quation est constitue d'un ensemble de couples (variable,coefficient) et d'une constante pour former une quation de la forme :
\~english Class for calculating an equation of a linear system.
The equations must be filled using the calculation methods provided to the equation system.
An equation consists of a set of pairs (variable, coefficient) and a constant, forming an equation of the following form:
\~ c1 var1 + c2 var2 + ... = const
*/
class PYC_PUBLIC IEquation{
protected:
	virtual~IEquation(){}
public:
	/**\~french Ajoute un coefficient associ  une variable
	@param var variable
	@param cR partie relle du coefficient
	@param cI partie imaginaire du coefficient
	\~english Adds a coefficient associated with a variable.  
	@param var: variable  
	@param cR: real part of the coefficient  
	@param cI: imaginary part of the coefficient.*/
	virtual void setCoefficient(IVariable*var,double cR,double cI=0)=0;
	virtual void setCoefficient(IVariable&var,double cR,double cI=0){setCoefficient(&var,cR,cI);}
	/**\~french Dfinit la constate du second membre
	@param cR partie relle de la constante
	@param cI partie imaginaire de la constante
	\~english Defines the constant of the right-hand side.  
	@param cR: real part of the constant  
	@param cI: imaginary part of the constant.*/
	virtual void setConstant(double cR,double cI=0)=0;
};

/**\~french Classe de dfinition et de rsolution de systmes d'quations linaires.
Un systme linaire est constitu par un ensemble de mthodes de dfinition d'quations dclares par addEquation().
Lors d'une rsolution du systme, chaque mthode est appele une fois pour dfinir son quation.
Il doit y avoir au moins autant d'quations que de variables dclares dans le systme par addVariable().
La rsolution du systme est obtenue en appelant la mthode solve() qui modifie la valeur des variables en fonction des rsultats de la rsolution.

exemple:
~~~~{.cpp}
class Equation:public CComponent{
	IVariable*terme;//Variable correspondant au terme  valuer
	IVariable*cst;// Variable correspondant au second membre
	IReference*it;//Rfrence d'accs aux termes des voisins
	//Constructeur de la classe
	Equation(char const*name,CSystem&system):CComponent(name,system){
		//Cration des variables
		terme=addVariable ("Terme",t_double,1);
		cst= addVariable ("Cst",t_double,1);
		//Cration de la rfrence
		it=addReference("It");
		//Cration de la bote de messages de connexion entre voisins
		IMessageBox*mb=addMessageBox("Voisins");
		//Exportation de la variable terme
		mb->addExport(terme,"Terme");
		//Importation des variables terme des voisins dans la rfrence it
		mb->addImport(it,"Terme");
		//Cration du systme d'quations s'il n'existe pas dj
		addSLEManager("Voisinage");
		//Ajout de terme comme variable du systme d'quations
		addSLEVariable("Voisinage",*terme);
		//Ajout de l'quation dans le systme d'quation
		addSLEEquation("Voisinage","equation",equation);
		//Dclaration de la mthode d'initialisation
		addStartMethod("start",&Equation::start);
	}
	//Dfinition de la mthode de remplissage de l'quation
	void equation(IEquation&equ){
		//On crit les coefficients associs aux termes des voisins
		for(int i=0;i<it->nbCnx();it++)
			equ.addCoefficient(it->variable(i),0.1,0) ;
		//On crit le coefficient associ  terme
		equ.addCoefficient(terme,1,0);
		//On renseigne la valeur du second membre
		equ.setConstant(cst->dValue(),0);
	}
	//Dfinition de la mthode d'initialisation
	void start()override{
		//Rsolution du systme linaire
		getSLEManager("Voisinage")->solve();
	}
};
~~~~
\~english Class for defining and solving linear systems of equations.
A linear system consists of a set of equation definition methods declared by *addEquation()*.
During the solution of the system, each method is called once to define its equation.
There must be at least as many equations as there are variables declared in the system by *addVariable()*.
The solution of the system is obtained by calling the *solve()* method, which modifies the value of the variables based on the results of the solution.

example:
~~~~{.cpp}
class Equation: public CComponent {
	IVariable* terme; // Variable corresponding to the term to evaluate
	IVariable* cst;   // Variable corresponding to the right-hand side
	IReference* it;   // Reference to access neighbor terms
	// Constructor of the class
	Equation(char const* name, CSystem& system): CComponent(name, system) {
		// Create variables
		terme = addVariable("Terme", t_double, 1);
		cst = addVariable("Cst", t_double, 1);
		// Create reference
		it = addReference("It");
		// Create message box for connection between neighbors
		IMessageBox* mb = addMessageBox("Voisins");
		// Export the terme variable
		mb->addExport(terme, "Terme");
		// Import neighbor terms into the it reference
		mb->addImport(it, "Terme");
		// Create equation system if it doesn't already exist
		addSLEManager("Voisinage");
		// Add terme as a variable in the equation system
		addSLEVariable("Voisinage", *terme);
		// Add the equation to the equation system
		addSLEEquation("Voisinage", "equation", equation);
		// Declare the initialization method
		addStartMethod("start", &Equation::start);
	}

	// Define the method for filling the equation
	void equation(IEquation& equ) {
		// Write coefficients for neighbor terms
		for (int i = 0; i < it->nbCnx(); it++)
			equ.addCoefficient(it->variable(i), 0.1, 0);
		// Write the coefficient for terme
		equ.addCoefficient(terme, 1, 0);
		// Set the constant value
		equ.setConstant(cst->dValue(), 0);
	}

	// Define the initialization method
	void start() override {
		// Solve the linear system
		getSLEManager("Voisinage")->solve();
	}
};
~~~~
*/
class PYC_PUBLIC ISLEManager:public CNamedComp{
	friend class CComponent;
	friend void ISLEManager_addEquation(ISLEManager*, char const*, CSysNamedComp&, PyObject*);
protected:
	ISLEManager(CSysNamedComp&system,char const*name):CNamedComp(system,name){}
	~ISLEManager(){}
	virtual void addEquation(char const*name,CSysNamedComp&comp,PyObject*pVoid)=0;//Composant ncessaire lors de la destruction d'un composant
public:
	virtual void addEquation(char const*name,CSysNamedComp&comp,void(CSysNamedComp::*equation)(IEquation&equ))=0;
	/**\~french Ajoute une quation
	@param name nom de l'quation
	@param comp composant propritaire de la mthode
	@param equation mthode de dfinition de l'quation
	\~english Adds an equation.  
	@param name: name of the equation  
	@param comp: owner component of the method  
	@param equation: method for defining the equation.*/
	template<typename Type>void addEquation(char const*name,Type&comp,void(Type::*equation)(IEquation&equation)){checkMethodAddrSize(sizeof(equation));addEquation(name,comp,static_cast<void(CNamedComp::*)(IEquation&equation)>(equation));}
	/**\~french Ajoute une variable  calculer par le systme d'quations.\~english Adds a variable to be calculated by the equation system.*/
	virtual void addVariable(IVariable*var)=0;
	void addVariable(IVariable&var){addVariable(&var);}
	/**\~french Demande la rsolution du systme (retourne VRAI si la rsolution s'est bien passe)\~english Requests the solution of the system (returns TRUE if the solution was successful).*/
	virtual bool solve()=0;
	/**\~french Demande l'impression du systme pour vrifier sa forme\~english Requests the printing of the system to verify its form.*/
	virtual void print(char const*fileName)const=0;
};

/**\~french Interface de manipulation des ingalits des systmes d'optimisation.
Une ingalit est dfinie par un ensemble de couples (variable, coefficient) et des bornes min et max.
L'ngalit est de la forme :
min < var1 coef1 + var2 coef2 + ... < max
Les ingalits doivent tre remplies par les mthodes ingalit dclares dans les systmes d'optimisation.
\~english Interface for manipulating inequalities in optimization systems.
An inequality is defined by a set of pairs (variable, coefficient) and minimum and maximum bounds.
The inequality is of the form:
min < var1 * coef1 + var2 * coef2 + ... < max
The inequalities must be filled using the inequality methods declared in the optimization systems.*/
class PYC_PUBLIC IInequality{
protected:
	virtual~IInequality(){}
public:
	/**\~french Ajoute un coefficient associ  une variable
	@param var variable
	@param cR coefficient
	\~english Adds a coefficient associated with a variable.  
	@param var: variable  
	@param cR: coefficient.*/
	virtual void setCoefficient(IVariable*var,double cR)=0;
	virtual void setCoefficient(IVariable&var,double cR){setCoefficient(&var,cR);}
	/**\~french Dfinit la borne min (-HUGE_VAL pour ne pas limiter infrieurement)\~english Sets the minimum bound (-HUGE_VAL to not impose a lower limit).*/
	virtual void setMin(double Min)=0;
	/**\~french Dfinit la borne max (HUGE_VAL pour ne pas limiter suprieurement)\~english Sets the maximum bound (HUGE_VAL to not impose an upper limit).*/
	virtual void setMax(double Max)=0;
	/**\~french Dfinit les limites min et max\~english Sets the minimum and maximum limits.*/
	virtual void setLimits(double Max,double Min){setMin(Min),setMax(Max);}
};

/**\~french Structure de dfinition de l'numrr de dfinition des messages trace de l'optimisation de systme\~english Definition structure of the enumeration for defining trace messages in system optimization.*/
struct TMILPMsgLevel{
	/**\~french Enumrr de dfinition des messages trace de l'optimisation de systme\~english Enumeration for defining trace messages in system optimization.*/
	typedef enum EMILPMsgLevel{
		off=0,	//!<\~french Pas de restitution\~english No restitution.
		error,	//!<\~french Uniquement les erreurs\~english Only errors.
		on,		//!<\~french Erreurs et warnings\~english Errors and warnings.
		all		//!<\~french Tous\~english All.
	}EMILPMsgLevel;
};
/**\~french Structure de dfinition de l'numr de dfinition du mode de mise  l'chelle de l'optimisation de systme\~english Definition structure of the enumeration for defining the scaling mode in system optimization.*/
struct TMILPScalingType{
	/**\~french Enumr de dfinition du mode de mise  l'chelle de l'optimisation de systme\~english Enumeration for defining the scaling mode in system optimization.*/
	typedef enum EMILPScalingType{
		geom_mean=1,	//!<\~french Moyenne gomtrique\~english Geometric mean.
		equilib=16,		//!<\~french Equilibrage\~english Balancing.
		near_p2=32,		//!<\~french Plus proche en distance quadratique\~english Nearest in quadratic distance.
		none=64,		//!<\~french Aucune mise  l'chelle\~english No scaling.
		automatic=128	//!<\~french Choix automatique du mode\~english Automatic selection of the mode.
	}EMILPScalingType;
};
/**\~french Structure de dfinition de l'numr de dfinition de l'algorithme d'optimisation\~english Definition structure of the enumeration for defining the optimization algorithm.*/
struct TMILPAlgorithmType{
	/**\~french Enumr de dfinition de l'algorithme d'optimisation\~english Enumeration for defining the optimization algorithm.*/
	typedef enum EMILPAlgorithmType{
		simplex=0,		//!<Simplex
		interior_point,	//!<\~french Point intrieur\~english Interior point.
		milp			//!<MILP
	}EMILPAlgorithmType;
};
/**\~french Structure de dfinition de l'numr de dfinition des options d'optimisation de systme\~english Definition structure of the enumeration for defining system optimization options.*/
struct TMILPOptionsType{
	/**\~french Enumr de dfinition des options d'optimisation de systme
	Les options sont cumulables
	\~english Enumeration for defining system optimization options.  
	The options are combinable.*/
	typedef enum EMILPOptionsType{
		none=0,				//!<\~french Aucune option\~english No option.
		pr_steepest_edge=1,	//!<\~french Projection suivant la plus grande pente (Simplex)\~english Projection along the steepest slope (Simplex).
		hr_ratio_test=2,	//!<\~french Test de rapport en 2 passes de Harris (Simplex)\~english Two-pass ratio test of Harris (Simplex).
		presolve=4,			//!<\~french Utilisation du prsolveur (Simplex, MILP)\~english Use of the presolver (Simplex, MILP).
		binarize=8,			//!<\~french Binairisation : uniquement avec prsolveur (MILP)\~english Binarization: only with presolver (MILP).
		sr_heuristic=16,	//!<\~french Heuristique d'arrondi simple (MILP)\~english Simple rounding heuristic (MILP).
		fp_heuristic=32,	//!<\~french Heuristique de pompe de faisabilit (MILP)\~english Feasibility pump heuristic (MILP).
		ps_heuristic=64,	//!<\~french Heuristique de recherche de proximit (MILP)\~english Proximity search heuristic (MILP).
		gmi_cuts=128,		//!<\~french Coupure d'entier mixte de Gomory (MILP)\~english Gomory mixed integer cut (MILP).
		mir_cuts=256,		//!<\~french Arrondi d'entiers mixte (MIR) (MILP)\~english Mixed Integer Rounding (MIR) (MILP).
		cov_cuts=512,		//!<\~french Mixed cover vut (MILP)\~english Mixed cover cut (MILP).
		clq_cuts=1024		//!<\~french Coupe clique (MILP)\~english Clique cut (MILP).
	}EMILPOptionsType;
};
/**\~french Structure de dfinition de l'numr de spcialisation de l'algorithme du simplex\~english Definition structure of the enumeration for specializing the Simplex algorithm.*/
struct TMILPSimplexType{
	/**\~french Enumr de spcialisation de l'algorithme du simplex\~english Enumeration for specializing the Simplex algorithm.*/
	typedef enum EMILPSimplexType{
		primal=0,		//!<Primal
		dual,			//!<Dual
		dual_primal		//!<\~french Combin dual et primal\~english Combined dual and primal.
	}EMILPSimplexType;
};
/**\~french Structure de dfinition de l'numr de spcialisation de l'algorithme du point intrieur\~english Definition structure of the enumeration for specializing the interior point algorithm.*/
struct TMILPIntPointType{
	/**\~french Enumr de spcialisation de l'algorithme du point intrieur (algorithme d'ordonnancement avant factorisation de Cholesky)\~english Enumeration for specializing the interior point algorithm (scheduling algorithm before Cholesky factorization).*/
	typedef enum EMILPIntPointType{
		none=0,				//!<\~french Algorithme standard\~english Standard algorithm.
		quot_min_degree,	//!<\~french quotient minimum degree (QMD)\~english Minimum Degree Quotient (QMD).
		appr_min_degree,	//!<approximate minimum degree (AMD)
		sym_appr_min_degree	//!<symmetric approximate minimum degree (SYMAMD)
	}EMILPIntPointType;
};
/**\~french Structure de dfinition de l'numr de dfinition de la mthode de branchement de l'algorithme MILP\~english Definition structure of the enumeration for defining the branching method of the MILP algorithm.*/
struct TMILPBranchingType{
	/**\~french Enumr de dfinition de la mthode de branchement de l'algorithme MILP\~english Enumeration for defining the branching method of the MILP algorithm.*/
	typedef enum EMILPBranchingType{
		first_frac_variable=0,	//!<\~french premire variable fractionnaire\~english First fractional variable.
		last_frac_variable,		//!<\~french dernire variable fractionnaire\~english Last fractional variable.
		most_frac_variable,		//!<\~french variable la plus fractionnaire\~english Most fractional variable.
		DrTom_heuristic,		//!<\~french Heuristique de Driebecket TomLin\~english Driebecket TomLin heuristic.
		pseudo_cost_heuristic	//!<\~french Heuristique hybride de pseudo-cot\~english Hybrid pseudo-cost heuristic.
	}EMILPBranchingType;
};
/**\~french Structure de dfinition de l'numr de dfinition de la mthode de recul de l'algorithme MILP\~english Definition structure of the enumeration for defining the rollback method of the MILP algorithm.*/
struct TMILPBacktrackingType{
	/**\~french Enumr de dfinition de la mthode de recul de l'algorithme MILP\~english Enumeration for defining the rollback method of the MILP algorithm.*/
	typedef enum EMILPBacktrackingType{
		depth_first=0,	//!<\~french En profondeur d'abord\~english Depth-first.
		breadth_first,	//!<\~french En largeur d'abord\~english Breadth-first.
		best_local_bound,//!<\~french En utilisant la meilleure borne locale\~english Using the best local bound.
		best_projection	//!<\~french En utilisant la meilleure projection\~english Using the best projection.
	}EMILPBacktrackingType;
};
/**\~french Structure de dfinition de l'numr de dfinition de la mthode de prtraitement de l'algorithme MILP\~english Definition structure of the enumeration for defining the preprocessing method of the MILP algorithm.*/
struct TMILPPreprocessingType{
	/**\~french Enumr de dfinition de la mthode de prtraitement de l'algorithme MILP\~english Enumeration for defining the preprocessing method of the MILP algorithm.*/
	typedef enum EMILPPreprocessingType{
		none=0,	//!<\~french Aucun prtraitement\~english No preprocessing.
		root,	//!<\~french Prtraitement de la racine\~english Root preprocessing.
		all		//!<\~french Prtraitement  tous les niveaux\~english Preprocessing at all levels.
	}EMILPPreprocessingType;
};

/**\~french Classe de dfinition d'un systme mixte entier linaire.
La classe constitue une interface avec la bilbiothque glpk dont elle reprend la plupart des options.
Un systme est constitu par :
- un ensemble de variables addVariable() et leurs bornes de variation setVariableLimits()
- un ensemble d'ingalits dfinies par des mthodes de remplssage d'IInequality addInequality()
- une fonction de calcul de la valeur globale  partir des valeurs des diffrentes variables dfinie par appel de addVariableObjCoef() sous la forme : var1 coef1 + var2 coef2 + ... 
- un sens d'optimisation (par dfaut une minimisation) setMaximize()

La rsolution est obtenue en appelant la mthode solve(). Les variables sont modifies en fonction du rsultat de l'optimisation.
Il est possible d'obtenir la valeur de la fonction objectif aprs optimisation objectiveValue()
Il est possible de dsigner une variable d'affectation automatique de la valeur de la fonction objectif aprs optimisation setObjectiveVar()

exemple:
~~~~{.cpp}
class TSTMILP :public CComponent
{
	IVariable*m_x1,*m_x2,*m_x3,*m_x4,*m_x5,*m_x6,*m_x7,*m_x8;
public:

	TSTMILP(char const*name,CSystem&sys):CComponent(name, sys)
	{
		//Cration des variables
		m_x1=addVariable("bi1",t_bool,true);
		m_x2=addVariable("bi2",t_bool,true);
		m_x3=addVariable("bi3",t_bool,true);
		m_x4=addVariable("bi4",t_bool,true);
		m_x5=addVariable("a1",t_double,0);
		m_x6=addVariable("a2",t_double,0);
		m_x7=addVariable("a3",t_double,0);
		m_x8=addVariable("scr",t_double,0);
		//Cration du systme
		IMILPManager&milp=*addMILPManager("MILP");
		//prix 350*5,330*3,310*4,280*6,500,450,400,100
		//Ajout des variables dans le systme avec leur coefficient de function obectif
		milp.addVariable(*m_x1,350*5);
		milp.addVariable(*m_x2,330*3);
		milp.addVariable(*m_x3,310*4);
		milp.addVariable(*m_x4,280*6);
		milp.addVariable(*m_x5,500);
		milp.setVariableMin(*m_x5,0); //Ajout d'une borne min  la variable
		milp.addVariable(*m_x6,450);
		milp.setVariableMin(*m_x6,0); //Ajout d'une borne min  la variable
		milp.addVariable(*m_x7,400);
		milp.setVariableMin(*m_x7,0); //Ajout d'une borne min  la variable
		milp.addVariable(*m_x8,100);
		milp.setVariableMin(*m_x8,0); //Ajout d'une borne min  la variable
		//Ajout des inquations dans le systme
		milp.addInequality("equa1",*this,&TSTMILP::equa1);
		milp.addInequality("equa2",*this,&TSTMILP::equa2);
		milp.addInequality("equa3",*this,&TSTMILP::equa3);
		//Ajout de la variable de reception de la valeur optimale
		milp.setObjectiveVar(addVariable("obj",t_double,0));
		//Dclaration de la methode d'initialisation
		addStartMethod("start",&TSTMILP::start);
	}

	//Mthode d'initialisation demandant la resolution du systme
	void start(){
		getMILPManager("MILP")->solve();
	}

	//Premire inquation
	void TSTMILP::equa1(IInequation&ineq){
		//5*x(1) + 3*x(2) + 4*x(3) + 6*x(4) + x(5) + x(6) + x(7) + x(8) = 25
		ineq.setCoefficient(m_x1,5);
		ineq.setCoefficient(m_x2,3);
		ineq.setCoefficient(m_x3,4);
		ineq.setCoefficient(m_x4,6);
		ineq.setCoefficient(m_x5,1);
		ineq.setCoefficient(m_x6,1);
		ineq.setCoefficient(m_x7,1);
		ineq.setCoefficient(m_x8,1);
		ineq.setLimits(25,25); //Les bornes sont gales : c'est une quation
	}

	//Deuxime inquation
	void equa2(IInequation&ineq){
		//5*0.05*x(1) + 3*0.04*x(2) + 4*0.05*x(3) + 6*0.03*x(4)+ 0.08*x(5) + 0.07*x(6) + 0.06*x(7) + 0.03*x(8) = 1.25.
		ineq.setCoefficient(m_x1,5*0.05);
		ineq.setCoefficient(m_x2,3*0.04);
		ineq.setCoefficient(m_x3,4*0.05);
		ineq.setCoefficient(m_x4,6*0.03);
		ineq.setCoefficient(m_x5,0.08);
		ineq.setCoefficient(m_x6,0.07);
		ineq.setCoefficient(m_x7,0.06);
		ineq.setCoefficient(m_x8,0.03);
		ineq.setLimits(1.25,1.25); //Les bornes sont gales : c'est une quation
	}

	//Troisime inquation
	void equa3(IInequation&ineq){
		//5*0.03*x(1) + 3*0.03*x(2) + 4*0.04*x(3) + 6*0.04*x(4)+ 0.06*x(5) + 0.07*x(6) + 0.08*x(7) + 0.09*x(8) = 1.25.
		ineq.setCoefficient(m_x1,5*0.03);
		ineq.setCoefficient(m_x2,3*0.03);
		ineq.setCoefficient(m_x3,4*0.04);
		ineq.setCoefficient(m_x4,6*0.04);
		ineq.setCoefficient(m_x5,0.06);
		ineq.setCoefficient(m_x6,0.07);
		ineq.setCoefficient(m_x7,0.08);
		ineq.setCoefficient(m_x8,0.09);
		ineq.setLimits(1.25,1.25); //Les bornes sont gales : c'est une quation
	}
};
~~~~
\~english
Class for defining a mixed integer linear system.
The class serves as an interface with the GLPK library, inheriting most of its options.
A system consists of:
- a set of variables defined by *addVariable()* and their variation bounds set by *setVariableLimits()*
- a set of inequalities defined by methods that populate *IInequality* using *addInequality()*
- a function to calculate the global value based on the values of the different variables, defined by calling *addVariableObjCoef()* in the form: var1 * coef1 + var2 * coef2 + ...
- an optimization direction (default is minimization) set by *setMaximize()*

The resolution is obtained by calling the *solve()* method. The variables are modified based on the optimization results.
It is possible to get the objective function value after optimization using *objectiveValue()*.
It is also possible to designate a variable for automatic assignment of the objective function value after optimization using *setObjectiveVar()*.

example:
~~~~{.cpp}
class TSTMILP : public CComponent {
	IVariable* m_x1, *m_x2, *m_x3, *m_x4, *m_x5, *m_x6, *m_x7, *m_x8;
public:

	TSTMILP(char const* name, CSystem& sys) : CComponent(name, sys) {
		// Create variables
		m_x1 = addVariable("bi1", t_bool, true);
		m_x2 = addVariable("bi2", t_bool, true);
		m_x3 = addVariable("bi3", t_bool, true);
		m_x4 = addVariable("bi4", t_bool, true);
		m_x5 = addVariable("a1", t_double, 0);
		m_x6 = addVariable("a2", t_double, 0);
		m_x7 = addVariable("a3", t_double, 0);
		m_x8 = addVariable("scr", t_double, 0);

		// Create the system
		IMILPManager& milp = *addMILPManager("MILP");

		// Add variables to the system with their objective function coefficients
		milp.addVariable(*m_x1, 350 * 5);
		milp.addVariable(*m_x2, 330 * 3);
		milp.addVariable(*m_x3, 310 * 4);
		milp.addVariable(*m_x4, 280 * 6);
		milp.addVariable(*m_x5, 500);
		milp.setVariableMin(*m_x5, 0); // Add a lower bound to the variable
		milp.addVariable(*m_x6, 450);
		milp.setVariableMin(*m_x6, 0); // Add a lower bound to the variable
		milp.addVariable(*m_x7, 400);
		milp.setVariableMin(*m_x7, 0); // Add a lower bound to the variable
		milp.addVariable(*m_x8, 100);
		milp.setVariableMin(*m_x8, 0); // Add a lower bound to the variable

		// Add inequalities to the system
		milp.addInequality("equa1", *this, &TSTMILP::equa1);
		milp.addInequality("equa2", *this, &TSTMILP::equa2);
		milp.addInequality("equa3", *this, &TSTMILP::equa3);

		// Add the variable for receiving the optimal value
		milp.setObjectiveVar(addVariable("obj", t_double, 0));

		// Declare the initialization method
		addStartMethod("start", &TSTMILP::start);
	}

	// Initialization method to request the resolution of the system
	void start() {
		getMILPManager("MILP")->solve();
	}

	// First inequality
	void TSTMILP::equa1(IInequation& ineq) {
		// 5*x(1) + 3*x(2) + 4*x(3) + 6*x(4) + x(5) + x(6) + x(7) + x(8) = 25
		ineq.setCoefficient(m_x1, 5);
		ineq.setCoefficient(m_x2, 3);
		ineq.setCoefficient(m_x3, 4);
		ineq.setCoefficient(m_x4, 6);
		ineq.setCoefficient(m_x5, 1);
		ineq.setCoefficient(m_x6, 1);
		ineq.setCoefficient(m_x7, 1);
		ineq.setCoefficient(m_x8, 1);
		ineq.setLimits(25, 25); // The bounds are equal: it's an equation
	}

	// Second inequality
	void equa2(IInequation& ineq) {
		// 5*0.05*x(1) + 3*0.04*x(2) + 4*0.05*x(3) + 6*0.03*x(4) + 0.08*x(5) + 0.07*x(6) + 0.06*x(7) + 0.03*x(8) = 1.25
		ineq.setCoefficient(m_x1, 5 * 0.05);
		ineq.setCoefficient(m_x2, 3 * 0.04);
		ineq.setCoefficient(m_x3, 4 * 0.05);
		ineq.setCoefficient(m_x4, 6 * 0.03);
		ineq.setCoefficient(m_x5, 0.08);
		ineq.setCoefficient(m_x6, 0.07);
		ineq.setCoefficient(m_x7, 0.06);
		ineq.setCoefficient(m_x8, 0.03);
		ineq.setLimits(1.25, 1.25); // The bounds are equal: it's an equation
	}

	// Third inequality
	void equa3(IInequation& ineq) {
		// 5*0.03*x(1) + 3*0.03*x(2) + 4*0.04*x(3) + 6*0.04*x(4) + 0.06*x(5) + 0.07*x(6) + 0.08*x(7) + 0.09*x(8) = 1.25
		ineq.setCoefficient(m_x1, 5 * 0.03);
		ineq.setCoefficient(m_x2, 3 * 0.03);
		ineq.setCoefficient(m_x3, 4 * 0.04);
		ineq.setCoefficient(m_x4, 6 * 0.04);
		ineq.setCoefficient(m_x5, 0.06);
		ineq.setCoefficient(m_x6, 0.07);
		ineq.setCoefficient(m_x7, 0.08);
		ineq.setCoefficient(m_x8, 0.09);
		ineq.setLimits(1.25, 1.25); // The bounds are equal: it's an equation
	}
};
~~~~
*/
class PYC_PUBLIC IMILPManager:public CNamedComp{
	friend class CComponent;
	friend void IMILPManager_addInequality(IMILPManager*, char const*, CSysNamedComp&, PyObject*);
protected:
	bool m_bMaximize;//Sens d'optimisation
	TMILPMsgLevel::EMILPMsgLevel m_MsgLevel;
	int m_MsgPeriod,m_MsgDelay,m_Scaling,m_TimeLimit,m_IterLimit;
	TMILPAlgorithmType::EMILPAlgorithmType m_Algo;
	int m_Options;
	TMILPSimplexType::EMILPSimplexType m_Simplex;
	TMILPIntPointType::EMILPIntPointType m_IntPoint;
	TMILPBranchingType::EMILPBranchingType m_Branching;
	TMILPBacktrackingType::EMILPBacktrackingType m_Backtracking;
	TMILPPreprocessingType::EMILPPreprocessingType m_Preprocessing;
	double m_ObjectiveVal;
	virtual void addIneq(char const*name,CSysNamedComp&comp,void(CSysNamedComp::*inequality)(IInequality&equ))=0;
	IMILPManager(CSysNamedComp&system,char const*name):CNamedComp(system,name),
		m_bMaximize(false),m_MsgLevel(TMILPMsgLevel::error),m_MsgPeriod(500),m_MsgDelay(0),m_Scaling(TMILPScalingType::none),m_TimeLimit(INT_MAX),m_IterLimit(INT_MAX),m_Algo(TMILPAlgorithmType::milp),m_Options(TMILPOptionsType::pr_steepest_edge|TMILPOptionsType::hr_ratio_test|TMILPOptionsType::presolve),m_Simplex(TMILPSimplexType::primal),m_IntPoint(TMILPIntPointType::appr_min_degree),m_Branching(TMILPBranchingType::DrTom_heuristic),m_Backtracking(TMILPBacktrackingType::best_local_bound),m_Preprocessing(TMILPPreprocessingType::all),m_ObjectiveVal(0){}
	virtual void addInequality(char const*name,CSysNamedComp&comp,PyObject*pVoid)=0;//Composant ncessaire lors de la destruction d'un composant
public:
	/**\~french Ajoute une inquation\~english Adds an inequality.*/
	template<typename Type>void addInequality(char const*name,Type&comp,void(Type::*inequality)(IInequality&inequality)){checkMethodAddrSize(sizeof(inequality));addIneq(name,comp,static_cast<void(CSysNamedComp::*)(IInequality&inequality)>(inequality));}
	virtual void removeInequality(CSysNamedComp&comp,char const*name)=0;
	/**\~french Ajoute une variable  calculer par le systme d'quations et son coefficient de fonction objectif.\~english Adds a variable to be calculated by the equation system and its objective function coefficient.*/
	virtual void addVariable(IVariable*var,double co)=0;
	void addVariable(IVariable&var,double co){addVariable(&var,co);}
	void addVariable(IVariable&var){addVariable(&var,0);}
	/**\~french Supprime la variable du systme\~english Removes the variable from the system.*/
	virtual void removeVariable(IVariable&var)=0;
	void removeVariable(IVariable*var){if(var)removeVariable(*var);}
	/**\~french Dfinit un coefficient de la fonction objectif\~english Sets a coefficient for the objective function.*/
	virtual void setVariableObjCoef(IVariable&var,double co)=0;
	void setVariableObjCoef(IVariable*var,double co){if(var)setVariableObjCoef(*var,co);}
	/**\~french Dfinit les bornes min et max de la variable\~english Sets the minimum and maximum bounds for the variable.*/
	void setVariableLimits(IVariable&var,double min,double max){setVariableMin(var,min);setVariableMax(var,max);}
	void setVariableLimits(IVariable*var,double min,double max){setVariableMin(var,min);setVariableMax(var,max);}
	/**\~french Dfinit la borne max de la variable\~english Sets the maximum bound for the variable.*/
	virtual void setVariableMax(IVariable&var,double max)=0;
	void setVariableMax(IVariable*var,double max){if(var)setVariableMax(*var,max);}
	/**\~french Dfinit la borne min de la variable\~english Sets the minimum bound for the variable.*/
	virtual void setVariableMin(IVariable&var,double min)=0;
	void setVariableMin(IVariable*var,double min){if(var)setVariableMin(*var,min);}
	/**\~french Dsigne la variable de stockage de la valeur de la fonction objectif\~english Designates the variable for storing the value of the objective function.*/
	virtual void setObjectiveVar(IVariable*var)=0;
	virtual void setObjectiveVar(IVariable&var){setObjectiveVar(&var);}
	/**\~french Dfinit le sens de l'optimisation (maximisation si bMax=vrai)\~english Sets the optimization direction (maximization if bMax = true).*/
	void setMaximize(bool bMax){m_bMaximize=bMax;};
	bool isMaximize()const{return m_bMaximize;}//!<\~french Retourne vrai si l'optimisation est une maximisation\~english Returns true if the optimization is a maximization.
	/**\~french Dfinit l'algorithme utilis parmi alg_simplex, alg_int_point, alg_milp\~english Sets the algorithm used from among *alg_simplex*, *alg_int_point*, *alg_milp*.*/
	void setAlgorithm(TMILPAlgorithmType::EMILPAlgorithmType algo){m_Algo=algo;}
	TMILPAlgorithmType::EMILPAlgorithmType algorithm()const{return m_Algo;}//!<\~french Retourne l'algorithme utilis\~english Returns the algorithm used.
	/**\~french Modifie la restitution de messages
	@param level niveau de restitution
	@param period priode de restitution des messages (en milli secondes)
	@param delay dlai avant restitution du premier message (en milli secondes)
	\~english Modifies the message output.  
	@param level: output level  
	@param period: message output period (in milliseconds)  
	@param delay: delay before the first message output (in milliseconds)*/
	void setMsgOptions(TMILPMsgLevel::EMILPMsgLevel level,int period,int delay){m_MsgLevel=level;m_MsgPeriod=period;m_MsgDelay=delay;}
	TMILPMsgLevel::EMILPMsgLevel msgLevel()const{return m_MsgLevel;}//!<\~french Reteourne le niveau de restitution de messages\~english Returns the message output level.
	int msgDelay()const{return m_MsgDelay;}//!<\~french Retourne le dlai avant mission du premier message\~english Returns the delay before the first message is output.
	int msgPeriod()const{return m_MsgPeriod;}//!<\~french Retourne la priode d'mission de messages\~english Returns the message emission period.
	void setScaling(int sc){m_Scaling=sc;}//!<\~french Modifie l'option de mise  l'chelle\~english Modifies the scaling option.
	int scaling()const{return m_Scaling;}//!<\~french Retourne l'option de mise  l'chelle\~english Returns the scaling option.
	void setOptions(int opts){m_Options=opts;}//!<\~french Modifie les options de calcul (cumul de valeurs de TMILPOptionsType::EMILPOptionsType)\~english Modifies the calculation options (cumulative values of TMILPOptionsType::EMILPOptionsType).
	int options()const{return m_Options;}//!<\~french Retourne les options de calcul\~english Returns the calculation options.
	/**\~french Modification d'une option de calcul
	@param opt numr numro d'option
	@param bSet vrai pour activer l'option
	\~english Modifies a calculation option.  
	@param opt: enumerated option number  
	@param bSet: true to activate the option.*/
	void setOption(TMILPOptionsType::EMILPOptionsType opt,bool bSet){if(bSet)m_Options=m_Options|opt;else m_Options=m_Options&~opt;}
	bool option(TMILPOptionsType::EMILPOptionsType opt)const{return (opt&m_Options)!=0;}//!<\~french Retourne vrai si l'option opt est active\~english Returns true if the option *opt* is activated.
	void addOptions(int opts){m_Options=m_Options|opts;}//!<\~french Active les options opts  (cumul de valeurs de TMILPOptionsType::EMILPOptionsType)\~english Activates the options *opts* (cumulative values of TMILPOptionsType::EMILPOptionsType).
	void delOptions(int opts){m_Options=m_Options&~opts;}//!<\~french Dsactive les options opts  (cumul de valeurs de TMILPOptionsType::EMILPOptionsType)\~english Deactivates the options *opts* (cumulative values of TMILPOptionsType::EMILPOptionsType).
	void setTimeLimit(int ms){m_TimeLimit=ms;}//!<\~french Fixe la limite de temps alloue  l'optimisation (en milli secondes)\~english Sets the time limit allocated for optimization (in milliseconds).
	int timeLimit()const{return m_TimeLimit;}//!<\~french Retourne la limite de temps alloue  l'optimisation\~english Returns the time limit allocated for optimization.
	void setIterLimit(int its){m_IterLimit=its;}//!<\~french Fixe le nombre maximum d'itrations\~english Sets the maximum number of iterations.
	int iterLimit()const{return m_IterLimit;}//!<\~french Retourne le nombre maximum d'itrations\~english Returns the maximum number of iterations.
	void setSimplexOpt(TMILPSimplexType::EMILPSimplexType smp){m_Simplex=smp;}//!<\~french Fixe l'option de l'algorithme du simplex\~english Sets the option for the Simplex algorithm.
	TMILPSimplexType::EMILPSimplexType simplexOpt()const{return m_Simplex;}//!<\~french Retourne l'option de l'algorithme du simplex\~english Returns the option for the Simplex algorithm.
	void setIntPointOpt(TMILPIntPointType::EMILPIntPointType ip){m_IntPoint=ip;}//!<\~french Fixe l'option de l'algorithme du point intrieur\~english Sets the option for the interior point algorithm.
	TMILPIntPointType::EMILPIntPointType intPointOpt()const{return m_IntPoint;}//!<\~french Retourne l'option de l'algorithme du point intrieur\~english Returns the option for the interior point algorithm.
	void setBacktracking(TMILPBacktrackingType::EMILPBacktrackingType bt){m_Backtracking=bt;}//!<\~french Fixe l'option de recul de l'algorithme MILP\~english Sets the rollback option for the MILP algorithm.
	TMILPBacktrackingType::EMILPBacktrackingType backtracking()const{return m_Backtracking;}//!<\~french Retourne l'option de recul de l'algorithme MILP\~english Returns the rollback option for the MILP algorithm.
	void setBranching(TMILPBranchingType::EMILPBranchingType br){m_Branching=br;}//!<\~french Fixe l'option de branchement de l'algorithme MILP\~english Sets the branching option for the MILP algorithm.
	TMILPBranchingType::EMILPBranchingType branching()const{return m_Branching;}//!<\~french Retourne l'option de branchement de l'algorithme MILP\~english Returns the branching option for the MILP algorithm.
	void setPreprocessing(TMILPPreprocessingType::EMILPPreprocessingType pre){m_Preprocessing=pre;}//!<\~french Fixe l'option de prtraitement\~english Sets the preprocessing option.
	TMILPPreprocessingType::EMILPPreprocessingType preprocessing()const{return m_Preprocessing;}//!<\~french Retourne l'option de prtraitement\~english Returns the preprocessing option.
	/**\~french Retourne la valeur de la fonction objectif obtenue\~english Returns the value of the objective function obtained.*/
	double objectiveValue()const{return m_ObjectiveVal;}
	/**\~french Demande la rsolution du systme (retourne VRAI si la rsolution s'est bien passe)\~english Requests the solution of the system (returns TRUE if the solution was successful).*/
	virtual bool solve()=0;
	/**\~french Demande l'impression du problme au format mps, glp, lp suivant l'extension du fichier\~english Requests printing of the problem in MPS, GLP, or LP format, depending on the file extension.*/
	virtual void print(char const*fileName)const=0;
};

/**\~french Classe de gestion de la lecture d'un ficher NetCDF
Pour l'instant, la classe ne permet que la lecture de variables simples sous la forme d'un vecteur de rels.
\~english Class for managing the reading of a NetCDF file.
Currently, the class only allows reading simple variables in the form of a vector of real numbers.*/
class PYC_PUBLIC INetCDFManager
{
public:
	/**\~french Construit un nouveau lecteur pour le fichier fileName.
	L'outil de lecture construit doit tre dtruit par l'utilisateur.
	@param fileName chemin du fichier  lire
	\~english Constructs a new reader for the file *fileName*.
	The constructed reader must be destroyed by the user.
	@param fileName: path to the file to be read.
	*/
	static INetCDFManager*newManager(char const*fileName);
	virtual ~INetCDFManager(){}//!<\~french Descructeur\~english Destructor.
	/**\~french Retourne un vecteur dcrivant les dimension de la matrice constitue par la variable varName.
	@param varName nom de la variable
	@return le vecteur des nombres d'lments dans chaque dimension
	\~english Returns a vector describing the dimensions of the matrix formed by the variable *varName*.  
	@param varName: name of the variable  
	@return: vector of the number of elements in each dimension.*/
	virtual std::vector<size_t>dimensions(char const*varName)const=0;
	/**\~french Retourne un vecteur de valeurs double de la variable varName
	@param varName nom de la variable
	@param startp vecteur des indices du premier point de l'hypercube  extraire
	@param countp vecteur du nombre de points  extraire dans chaque dimensions
	@return le vecteur des valeurs de la variable
	\~english Returns a vector of double values for the variable *varName*.  
	@param varName: name of the variable  
	@param startp: vector of indices for the first point of the hypercube to extract  
	@param countp: vector of the number of points to extract in each dimension  
	@return: vector of the values of the variable.*/
	virtual std::vector<double> dRead1D(char const*varName,std::vector<size_t>startp,std::vector<size_t>countp)const=0;
	/**\~french Retourne un vecteur de valeurs float de la variable varName
	@param varName nom de la variable
	@param startp vecteur des indices du premier point de l'hypercube  extraire
	@param countp vecteur du nombre de points  extraire dans chaque dimensions
	@return le vecteur des valeurs de la variable
	\~english Returns a vector of float values for the variable *varName*.  
	@param varName: name of the variable  
	@param startp: vector of indices for the first point of the hypercube to extract  
	@param countp: vector of the number of points to extract in each dimension  
	@return: vector of the values of the variable.*/
	virtual std::vector<float>fRead1D(char const*varName,std::vector<size_t>startp,std::vector<size_t>countp)const=0;
};

/**\~french Classe de stockage d'un tat du systme.
Un objet de cette classe peut tre obtenu en appelant la methode systemState de la classe CSystem.
La mthode reset() permet de replacer le systme dans l'tat stock
La mthode release() permet d'oublier cette mmorisation
La classe permet d'acceder aux valeurs des variables du systmes dans l'tat mmoris
\~english Class for storing a system state.
An object of this class can be obtained by calling the *systemState* method of the *CSystem* class.
The *reset()* method restores the system to the stored state.
The *release()* method forgets this memorization.
The class allows access to the values of the system's variables in the stored state.*/
class PYC_PUBLIC ISystemState{
	ISystemState(ISystemState const&);
	ISystemState const&operator=(ISystemState const&);
protected:
	ISystemState(){}
	virtual ~ISystemState(){}
public :
	virtual void reinit()=0;//!<\~french Reprend l'tat courant du systme\~english Restores the current state of the system.
	virtual void reset()const=0;//!<\~french Replace le systme dans l'tat mmoris\~english Restores the system to the stored state.
	virtual void release()const=0;//!<\~french Oublie la mmorisation de l'tat\~english Forgets the memorization of the state.
	/**\~french Retourne la valeur courante de var sous forme de complexe.\~english Returns the current value of *var* as a complex.*/
	virtual std::complex<double> cValue(IVarBase&var)const=0;
	/**\~french Retourne la valeur courante de var sous forme de double.\~english Returns the current value of *var* as a double.*/
	virtual double dValue(IVarBase&var)const=0;
	/**\~french Retourne la valeur courante de var sous forme de float.\~english Returns the current value of *var* as a float.*/
	virtual float fValue(IVarBase&var)const=0;
	/**\~french Retourne la valeur courante de var sous forme de int.\~english Returns the current value of *var* as an int.*/
	virtual int iValue(IVarBase&var)const=0;
	/**\~french Retourne la valeur courante de var sous forme de bool.\~english Returns the current value of *var* as a bool.*/
	virtual bool bValue(IVarBase&var)const=0;
	/**\~french Retourne la valeur courante de var sous forme de string.\~english Returns the current value of *var* as a string.*/
	virtual std::string sValue(IVarBase&var)const=0;
	/**\~french Retourne la liste des tats actifs.\~english Returns the list of active states.*/
	virtual std::vector<IState const*>const&activStates()const=0;
	/**\~french Oprateur de comparaison.\~english Comparison operator.*/
	virtual bool operator==(ISystemState const&)const=0;
};

/**\~french Structure de dfinition de l'numr de dfinition du type d'algorithme d'optimisation non linaire\~english Definition structure of the enumeration for defining the type of nonlinear optimization algorithm.*/
struct PYC_PUBLIC TOptimAlgorithmType{
	/**\~french Enumr de dfinition du type d'algorithme d'optimisation non linaire\~english Enumeration for defining the type of nonlinear optimization algorithm.*/
	typedef enum EOptimAlgorithmType{
		unknown=0,	//!<\~french Inconnu (non dfini)\~english Unknown (undefined).
		powell=1,	//!<Powell
		gradient	//!<\~french Gradient (plus grande pente)\~english Gradient (steepest descent).
	}EOptimAlgorithmType;
	static char const*st_Names[3];
	static EOptimAlgorithmType fromStr(char const*value);
};

/**\~french Classe de base des algorithmes ddi  l'valuation de la fiabilit
La classe permet de dfinir une cible en dfinissant les tats quivalents de perte de file.
L'entre dans un de ces tats signifie la perte d'une file
Les tats quivalents de perte d'une file sont dfinis par un masque textuel exprim sous la forme :
- d'un nom gnrique nomComposant.nomEtat o les noms peuvent tre luds pour signifier tous les composants ou tous les tats
- d'une expression rgulire de recherche sur le nom complet : l'expression rgulire doit tre prcde de #
\~english
Base class for algorithms dedicated to reliability evaluation.
The class allows defining a target by specifying the equivalent states of queue loss.
Entering one of these states signifies a queue loss.
The equivalent states of queue loss are defined by a textual mask expressed in the form of:
- a generic name `componentName.stateName`, where names can be omitted to signify all components or all states
- a regular expression search on the full name: the regular expression must be prefixed with #.*/
class PYC_PUBLIC IReliabilityEvaluator:public CNamedComp{
protected:
	unsigned int m_Trace;
	double m_Reliability,m_StdDevReliability;
	IReliabilityEvaluator(CSysNamedComp&sys,char const*name):CNamedComp(sys,name),m_Trace(0),m_Reliability(0),m_StdDevReliability(0){}
public:
	void setTrace(int level){m_Trace=level;}//!<\~french Modifie le niveau de trace\~english Modifies the trace level.
	int trace()const{return m_Trace;}//!<\~french Retourne le niveau de trace\~english Returns the trace level.
	virtual bool evalReliability()=0;//!<\~french Evalue la fiabilit et son cart type en utilisant les alphas courants ou aprs cross entropy si nbSimuOptim>0\~english Evaluates the reliability and its standard deviation using the current alphas or after cross entropy if *nbSimuOptim > 0*.
	double reliability()const{return m_Reliability;}//!<\~french Retourne la fiabilit calcule\~english Returns the calculated reliability.
	double stdDevReliability()const{return m_StdDevReliability;}//!<\~french Retourne l'cart type de la fiabilit\~english Returns the standard deviation of the reliability.
	virtual void addKOState(char const*name)=0;//!<\~french Dsigne les tats quivalents de perte d'une file\~english Designates the equivalent states of queue loss.
	virtual void removeKOState(char const*name)=0;//!<\~french Supprime un tat quivalent de perte de file\~english Removes an equivalent state of queue loss.
	virtual void setSequenceFile(char const*fileName)=0;//!<\~french Modifie le nom de base des fichiers de restitution des optimisations successives\~english Modifies the base name of the files for successive optimization results.
	virtual char const*sequenceFile()const=0;//!<\~french Retourne le nom de base des fichiers de restitution des optimisations successives\~english Returns the base name of the files for successive optimization results.
};

/**\~french Classe de calcul d'un tirage prfrentiel.
La classe permet de paramtrer et de lancer les calculs avec ou sans optimisation.
Le calcul est effectu en utilisant le nombre de squences donn pour effectuer une simulation standard.
Le tirage de nouvelles squences est stopp ds que le nombre d'vnements maximum est atteint (s'il est diffrent de 0).

Les fonctions de modification des paramtres de loi peuvent tre dfinies de 2 faons :
- en fournissant une mthode d'une classe fille de la classe CSystem qui ne doit se baser que sur les automates des composants.
- en donnant le numro du paramtre alpha utilis pour le calcul par dfaut : exp(-alpha*nbCompToBreak()*nbCompToBreak())

Les coeffients alpha doivent tre crs grce  addAlpha. Il peuvent tre limits grce  setAlphaMin et setAlphaMax.

La mthode de calcul de nbCompToBreak peut tre paramtre en dfinissant les tats dgrads grce  la mthode addKoState.
Les diffrents tats identifis par un appel de addKoState sont considrs comme un mme tat dgrad ce qui permet d'unifier la perte d'une file.
La mthode nbCompToBreak peut tre surcharge dans la class CSystem.

Les coefficients de correction appliqus aux lois instantanes ne peuvent pas dpasser le paramtre gammaModifierMax (pour viter que la probabilit dpasse 1)

Si nbSimuOptim = 0, un unique tirage prrentiel est excut sans optimisation.
Sinon, on effectue nbSimuOptim optimisation tant qu'on ne satisfait pas la limite de prcision requise sur les alphas.

Si un fichier sequenceFile est fourni, un fichier sequenceFileN est restitu aprs chaque optimisation et un fichier sequenceFile est restitu en fin de calcul.

Le rsultat est obtenu par appel de reliability() et stdDevReliability();
\~english
Class for calculating a preferential draw.
The class allows parameterizing and performing calculations with or without optimization.
The calculation is performed using the number of sequences provided to carry out a standard simulation.
New sequence generation is stopped once the maximum number of events is reached (if it is not 0).

The law parameter modification functions can be defined in two ways:
- by providing a method from a subclass of the *CSystem* class, which should only rely on the automata of the components.
- by giving the alpha parameter number used for the default calculation: *exp(-alpha*nbCompToBreak()*nbCompToBreak())*.

The alpha coefficients must be created using *addAlpha*. They can be limited using *setAlphaMin* and *setAlphaMax*.

The method for calculating *nbCompToBreak* can be parameterized by defining the degraded states using the *addKoState* method.
The different states identified by a call to *addKoState* are considered as the same degraded state, which unifies the loss of a queue.
The *nbCompToBreak* method can be overridden in the *CSystem* class.

The correction coefficients applied to the instantaneous laws cannot exceed the *gammaModifierMax* parameter (to prevent the probability from exceeding 1).

If *nbSimuOptim = 0*, a single preferential draw is executed without optimization.
Otherwise, *nbSimuOptim* optimizations are performed until the required precision limit on the alphas is satisfied.

If a *sequenceFile* is provided, a file named *sequenceFileN* is generated after each optimization, and a file *sequenceFile* is generated at the end of the calculation.

The result is obtained by calling *reliability()* and *stdDevReliability()*.*/
class PYC_PUBLIC IImportanceSampling:public IReliabilityEvaluator{
protected:
	size_t m_NbSimuOptim;//!<\~french Nombre de simulations d'optimisation\~english Number of optimization simulations.
	size_t m_NbEvtMax;//!<\~french Nombre maximal d'vnements pour un ensemble de traj constituant une simu\~english Maximum number of events for a set of trajectories forming a simulation.
	double m_CEPrecision;//!<\~french Prcision de maximisation de l'entropie croise\~english Precision for cross-entropy maximization.
	double m_Precision;//!<\~french Prcision de calcul des alphas\~english Precision for calculating alphas.
	TOptimAlgorithmType::EOptimAlgorithmType m_OptimAlgorithm;
	IImportanceSampling(CSysNamedComp&sys,char const*name):IReliabilityEvaluator(sys,name),m_NbSimuOptim(0),m_NbEvtMax(100),m_CEPrecision(0),m_Precision(0),m_OptimAlgorithm(TOptimAlgorithmType::powell){}
	virtual ~IImportanceSampling(){}
	virtual void addUFunction(char const*transName,char const*uFctName,double (CSystem::*)())=0;
public:
	void setNbSimuOptim(size_t nb){m_NbSimuOptim=nb;}//!<\~french Modifie le nombre d'optimisations  effectuer\~english Modifies the number of optimizations to perform.
	size_t nbSimuOptim()const{return m_NbSimuOptim;}//!<\~french Retourne le nombre d'optimisations  effectuer\~english Returns the number of optimizations to perform.
	void setNbEvtMax(size_t nb){m_NbEvtMax=nb;}//!<\~french Modifie le nombre maximum d'vnements  obtenir dans les squences d'optimisation\~english Modifies the maximum number of events to obtain in the optimization sequences.
	size_t nbEvtMax()const{return m_NbEvtMax;}//!<\~french Retourne le nombre maximum d'vnements  obtenir dans les squences d'optimisation\~english Returns the maximum number of events to obtain in the optimization sequences.
	void setCEPrecision(double prec){m_CEPrecision=prec;}//!<\~french Modifie la prcision de maximisation de l'entropie croise\~english Modifies the precision for cross-entropy maximization.
	double CEPrecision()const{return m_CEPrecision;}//!<\~french Retourne la prcision de maximisation de l'entropie croise\~english Returns the precision for cross-entropy maximization.
	void setAlphaPrecision(double prec){m_Precision=prec;}//!<\~french Modifie la prcision de calcul des Alphas\~english Modifies the precision for calculating alphas.
	double alphaPrecision()const{return m_Precision;}//!<\~french Retourne la prcision de calcul des Alphas\~english Returns the precision for calculating alphas.
	TOptimAlgorithmType::EOptimAlgorithmType optimAlgorithm()const{return m_OptimAlgorithm;}//!<\~french Retourne l'algorithme utilis pour l'optimisation\~english Returns the algorithm used for optimization.
	void setOptimAlgorithm(TOptimAlgorithmType::EOptimAlgorithmType type){m_OptimAlgorithm=type;}
	virtual size_t dimension()const=0;//!<\~french Retourne la dimension du vecteur des alphas\~english Returns the dimension of the alpha vector.
	virtual double alphaInit(size_t i)const=0;//!<\~french Retourne le ime alpha valeur initiale\~english Returns the i-th initial alpha value.
	virtual double alpha(size_t i)const=0;//!<\~french Retourne le ime alpha\~english Returns the i-th alpha.
	virtual double alphaMax(size_t i)const=0;//!<\~french Retourne le max du ime alpha\~english Returns the maximum of the i-th alpha.
	virtual double alphaMin(size_t i)const=0;//!<\~french Retourne le min du ime alpha\~english Returns the minimum of the i-th alpha.
	virtual void setAlpha(size_t i,double alpha)=0;//!<\~french Modifie le ime alpha\~english Modifies the i-th alpha.
	virtual void setAlphaMax(size_t i,double value)=0;//!<\~french Modifie la limite suprieure du ime alpha\~english Modifies the upper limit of the i-th alpha.
	virtual void setAlphaMin(size_t i,double value)=0;//!<\~french Modifie la limite infrieure du ime alpha\~english Modifies the lower limit of the i-th alpha.
	virtual void addAlpha(double value)=0;//!<\~french Ajoute un alpha au systme (augmente la dimension de 1)\~english Adds an alpha to the system (increases the dimension by 1).
	virtual void removeAlpha(size_t i)=0;//!<\~french Supprime le ime alpha du systme (diminue la dimension de 1)\~english Removes the i-th alpha from the system (decreases the dimension by 1).
	/**\~french Ajoute une fonction de calcul des paramtres en fonction des alphas\~english Adds a function to calculate the parameters based on the alphas.*/
	template<typename TYPE>void addUFunction(char const*transName,char const*uFctName,TYPE uFct){checkMethodAddrSize(sizeof(uFct));addUFunction(transName,uFctName,static_cast<double (CSystem::*)()>(uFct));}
	virtual void addUFunction(char const*transName,char const*uFctName,PyObject*uFct)=0;//!<\~french Ajoute une fonction de calcul des paramtres (python)\~english Adds a function to calculate the parameters (Python).
	virtual void addUFunction(char const*transName,char const*uFctName){addUFunction(transName,uFctName,(PyObject*)NULL);}//!<\~french Ajoute une fonction de calcul des paramtres (python)\~english Adds a function to calculate the parameters (Python).
	/**\~french Ajoute une utilisation de USimple pour le iAlpha me alpha\~english Adds a use of USimple for the i-th alpha.*/
	virtual void addUFunction(char const*transName,size_t iAlpha)=0;
};

/**\~french Classe de calcul suivant la mthode Monte Carlo squentiel
La mthode consiste  simuler un ensemble de squences en parallle sur des intervalles de temps successifs.
Aprs chaque intervalle, on effectue un calcul de rpartition de la fonction G qui value la distance de la squence  la cible
Si la fonction G est suffisamment inhomogne dans l'espace des squences (seuillage C), on effectue un tri des squences se rapprochant le plus de la cible pour continuer la simulation
Les squences pousuivies sont dmultiplies pour conserver un nombre des squences constant avec un algorithme de rpartition simple ou avec rechantillonnage rsiduel.
L'algorithme peut utiliser la mmorisation setMemorization(), mais il faut dans ce cas calculer la vraisemblance des squences CSystem::evalLikelyhood()
L'cart type de la fiabilit peut tre valu en demandant de rpter la simulation plusieurs fois setNbSimu()
\~english Class for calculation using the Sequential Monte Carlo method.
The method consists of simulating a set of sequences in parallel over successive time intervals.
After each interval, a distribution calculation of the function G is performed, which evaluates the distance of the sequence to the target.
If the function G is sufficiently inhomogeneous in the sequence space (threshold C), the sequences closest to the target are sorted to continue the simulation.
The pursued sequences are split to maintain a constant number of sequences using a simple distribution algorithm or residual resampling.
The algorithm can use memorization with *setMemorization()*, but in that case, the likelihood of the sequences must be calculated with *CSystem::evalLikelihood()*.
The standard deviation of the reliability can be evaluated by requesting to repeat the simulation multiple times using *setNbSimu()*.*/
class PYC_PUBLIC ISeqMonteCarlo:public IReliabilityEvaluator{
protected:
	bool m_Memorization;
	bool m_ResidualResampling;
	PyTime m_Dt;
	double(CSystem::*m_GFct)(ISystemState const&,ISystemState const&);
	PyObject*m_PyGFct;
	char*m_GFctName;
	double m_C;//!<\~french Coefficient de seuil de slection\~english Threshold coefficient for selection.
	unsigned m_NbSimu;//!<\~french Nombre de simulations  effectuer pour calculer la variance (<=1: pas de variance)\~english Number of simulations to perform to calculate the variance (<=1: no variance).
	ISeqMonteCarlo(CSysNamedComp&sys,char const*name):IReliabilityEvaluator(sys,name),m_Memorization(false),m_ResidualResampling(false),m_Dt(0),m_GFct(NULL),m_PyGFct(NULL),m_GFctName(NULL),m_C(0),m_NbSimu(0){}
	virtual ~ISeqMonteCarlo();
	void setGFunction(double (CSystem::*gFct)(ISystemState const&,ISystemState const&));
public:
	void setMemorization(bool value){m_Memorization=value;}//!<\~french Modifie l'option de mmorisation\~english Modifies the memorization option.
	bool memorization()const{return m_Memorization;}//!<\~french Retourne l'option de mmorisation\~english Returns the memorization option.
	void setResidualResampling(bool value){m_ResidualResampling=value;}//!<\~french Modifie l'option de rchantillonnage rsiduel\~english Modifies the residual resampling option.
	bool residualResampling()const{return m_ResidualResampling;}//!<\~french Retourne l'option de rchantillonnage rsiduel\~english Returns the residual resampling option.
	void setDt(PyTime value){m_Dt=value;}//!<\~french Fixe le pas temporel de simulation\~english Sets the time step for the simulation.
	double dt()const{return m_Dt;}//!<\~french Retourne le pas temporel de simulation\~english Returns the time step for the simulation.
	void setC(double C){m_C=C;}//!<\~french Fixe le seuil de dclanchement d'un filtrage des squences\~english Sets the threshold for triggering sequence filtering.
	double c()const{return m_C;}//!<\~french Retourne le seuil de dclanchement d'un filtrage des squences\~english Returns the threshold for triggering sequence filtering.
	void setNbSimu(unsigned value){m_NbSimu=value;}//!<\~french Fixe le nombre de simulations utilises pour calculer la fiabilit et son cart type\~english Sets the number of simulations used to calculate the reliability and its standard deviation.
	unsigned nbSimu()const{return m_NbSimu;}//!<\~french Retourne le nombre de simulations utilises pour calculer la fiabilit et son cart type\~english Returns the number of simulations used to calculate the reliability and its standard deviation.
	/**\~french Dsigne la fonction de calcul de G\~english Designates the function for calculating G.*/
	template<typename TYPE>void setGFunction(TYPE gFct){checkMethodAddrSize(sizeof(gFct));setGFunction(static_cast<double (CSystem::*)(ISystemState const&,ISystemState const&)>(gFct));}
	virtual void setGFunction(char const*gFctName,PyObject*gFct);//!<\~french Dsigne une fonction de calcul de G (python)\~english Designates a function for calculating G (Python).
};

/**\~french Structure de dfinition de l'numr de dfinition du type d'algorithme d'exploration de l'arbre des squences\~english Structure for defining the enumeration of the type of sequence tree exploration algorithm.*/
struct PYC_PUBLIC TSeqTreeExploAlgorithmType{
	/**\~french Enumr de dfinition du type d'algorithme d'optimisation non linaire\~english Enumeration for defining the type of nonlinear optimization algorithm.*/
	typedef enum ESeqTreeExploAlgorithmType{
		unknown=0,	//!<\~french Inconnu (non dfini)\~english Unknown (undefined).
		sampling=1,	//!<\~french Echantillonnage\~english Sampling.
		harrison,	//!<Harrison (INS-EXP)
		sri			//!<SRI(INS-EXP)
	}ESeqTreeExploAlgorithmType;
	static char const*st_Names[4];
	static ESeqTreeExploAlgorithmType fromStr(char const*value);
};

/**\~french Classe de calcul par exploration systmatique du graphe de squences.

L'exploration peut tre faite avec 2 algorithmes :
- Echantillonnage des lois de tirage tales dans le temps
	Il faut dfinir un nombre d'chantillons pour chacune des lois non ponctuelles utilises (IDistLaw::setNbSamples)
	L'algorithme est compatible avec le monitoring complet
- algorithme de Harisson
	Il faut utiliser un modle markovien (lois INS et AXP uniquement, pas d'ODE)
	L'algorithme ne permet que le monitoring de transitions et ne fournit pas de date de tirage autre que 0

Les 2 algorithmes fournissent une estimation de la probabilit de chaque squence.

L'exploration est compatible avec le monitoring et l'utilisation d'indicateurs.
Comme l'exploration est trs lourde, il est ncessaire de dfinir des limites  la poursuite des squences :
- limite de nombre de branches (nombre de tirages de transitions)
- limite de nombre de pannes (arrt aprs tirage de plus de N pannes)
- limite minimale de la probabilit d'une squence
\~english
Class for calculation using systematic exploration of the sequence graph.

The exploration can be done using two algorithms:
- **Sampling of the drawing laws spread over time**
  A number of samples must be defined for each of the non-point laws used (*IDistLaw::setNbSamples*).
  The algorithm is compatible with full monitoring.
- **Harrison's algorithm**
  A Markovian model must be used (only INS and AXP laws, no ODE).
  This algorithm only supports monitoring of transitions and does not provide any draw date other than 0.

Both algorithms provide an estimation of the probability for each sequence.

The exploration is compatible with monitoring and the use of indicators.
Since exploration is computationally intensive, limits must be defined to continue the sequences:
- Limit on the number of branches (number of transition draws)
- Limit on the number of failures (stop after more than N failures are drawn)
- Minimum probability limit for a sequence.*/
class PYC_PUBLIC ISeqTreeExplorer:public CNamedComp{
protected:
	unsigned int m_Trace;
	TSeqTreeExploAlgorithmType::ESeqTreeExploAlgorithmType m_Algorithme;//!<\~french Algorithme utilis pour le calcul de la probabilit\~english Algorithm used for calculating the probability.
	double m_CurProbability,m_MinProbability;
	unsigned m_MaxNbBranches, m_MaxNbFailures;
	double m_Epsilon1,m_Epsilon2,m_Epsilon3,m_Precision;//\~french Paramtres de l'algorithme de Harrison\~english Parameters of Harrison's algorithm.
	double m_ResidanceTimeTolerance;//\~french Paramtre de controle de l'algorithme SRI\~english Control parameter for the SRI algorithm.
	ISeqTreeExplorer(CSysNamedComp&sys,char const*name):CNamedComp(sys,name),m_Trace(0),m_Algorithme(TSeqTreeExploAlgorithmType::sampling),m_CurProbability(0)
		,m_MinProbability(0),m_MaxNbBranches(-1),m_MaxNbFailures(-1),m_Epsilon1(1.e-3),m_Epsilon2(1.e-3),m_Epsilon3(1.e-3),m_Precision(1.e-8),m_ResidanceTimeTolerance(0.01){}
public:
	void setTrace(int level){m_Trace=level;}//!<\~french Modifie le niveau de trace\~english Modifies the trace level.
	int trace()const{return m_Trace;}//!<\~french Retourne le niveau de trace\~english Returns the trace level.
	void setAlgoHarrison(bool value){m_Algorithme=value?TSeqTreeExploAlgorithmType::harrison:TSeqTreeExploAlgorithmType::sampling;}
	void setAlgoSRI(bool value){m_Algorithme=value?TSeqTreeExploAlgorithmType::sri:TSeqTreeExploAlgorithmType::sampling;}
	bool isAlgoHarrison()const{return m_Algorithme==TSeqTreeExploAlgorithmType::harrison;}//!<\~french Retourne vrai si l'algorithme de Harrison doit tre utilis\~english Returns true if the Harrison algorithm should be used.
	bool isAlgoSampling()const{return m_Algorithme==TSeqTreeExploAlgorithmType::sampling;}//!<\~french Retourne vrai si l'algorithme d'chantillonnage doit tre utilis\~english Returns true if the sampling algorithm should be used.
	bool isAlgoSRI()const{return m_Algorithme==TSeqTreeExploAlgorithmType::sri;}//!<\~french Retourne vrai si l'algorithme SRI doit tre utilis\~english Returns true if the SRI algorithm should be used.
	void setAlgorithme(TSeqTreeExploAlgorithmType::ESeqTreeExploAlgorithmType algorithme){m_Algorithme=algorithme;}
	TSeqTreeExploAlgorithmType::ESeqTreeExploAlgorithmType algorithme()const{return m_Algorithme;}
	void setMinProbability(double minP){m_MinProbability=minP;}//!<\~french Modifie le seuil de probabilit min en dea duquel la squence est stoppe\~english Modifies the minimum probability threshold below which the sequence is stopped.
	double minProbability()const{return m_MinProbability;}//!<\~french Retourne le seuil de probabilit min en dea duquel la squence est stoppe\~english Returns the minimum probability threshold below which the sequence is stopped.
	double curProbability()const{return m_CurProbability;}//!<\~french Retourne la probabilit des squences parcourues\~english Returns the probability of the traversed sequences.
	void setMaxNbBranches(unsigned nb){m_MaxNbBranches=nb;}//!<\~french Modifie le nombre max de branches au del duquel la squence est stoppe\~english Modifies the maximum number of branches beyond which the sequence is stopped.
	unsigned maxNbBranches()const{return m_MaxNbBranches;}//!<\~french Retourne le nombre max de branches au del duquel la squence est stoppe\~english Returns the maximum number of branches beyond which the sequence is stopped.
	void setMaxNbFailures(unsigned nb){m_MaxNbFailures=nb;}//!<\~french Modifie le nombre max de dfaillances au del duquel la squence est stoppe\~english Modifies the maximum number of failures beyond which the sequence is stopped.
	unsigned maxNbFailures()const{return m_MaxNbFailures;}//!<\~french Retourne le nombre max de dfaillances au del duquel la squence est stoppe\~english Returns the maximum number of failures beyond which the sequence is stopped.
	/**\~french Modifie les paramtres de calcul de l'algorithme de Harrison
	@param epsilon1 : limite des tats rapides (somme 1/lambda*t < epsilon1)
	@param epsilon2 : limite des tats lents (lambda*t < epsilon)
	@param epsilon3 : limite de fusion des ples ((l1-l2)/l1 < epsilon3)
	@param precision : contrle de prcision des additions/soustractions ( abs(1 - somme termes positifs/somme termes ngatif)< precision => erreur)
	\~english Modifies the calculation parameters of the Harrison algorithm  
	@param epsilon1: limit for fast states (sum 1/lambda*t < epsilon1)  
	@param epsilon2: limit for slow states (lambda*t < epsilon2)  
	@param epsilon3: limit for pole merging ((l1-l2)/l1 < epsilon3)  
	@param precision: controls the precision of additions/subtractions (abs(1 - sum of positive terms/sum of negative terms) < precision => error)*/
	void setHarrisonParameters(double epsilon1,double epsilon2,double epsilon3,double precision);
	double epsilon1()const{return m_Epsilon1;}//!<\~french Retourne la limite des tats rapides (par dfaut 1.e-3)\~english Returns the limit for fast states (default is 1.e-3).
	double epsilon2()const{return m_Epsilon2;}//!<\~french Retourne la limite des tats lents (par dfaut 1.e-3)\~english Returns the limit for slow states (default is 1.e-3).
	double epsilon3()const{return m_Epsilon3;}//!<\~french Retourne la limite de fusion d'tats (par dfaut 1.e-3)\~english Returns the state merging limit (default is 1.e-3).
	double precision()const{return m_Precision;}//!<\~french Retourne la limite de contrle de prcision (par dfaut 1e-8)\~english Returns the precision control limit (default is 1e-8).
	/**\~french Modifie la tolrance sur le temps de sjour pour l'algorithme SRI
	@param tolerance : tolrance sur le rapport entre les temps de sjour des tats intermdiaires et celui de l'tat initial (par dfaut 0.01)
	\~english Modifies the tolerance on the residence time for the SRI algorithm.  
	@param tolerance: tolerance on the ratio between the residence times of intermediate states and that of the initial state (default is 0.01).*/
	void setResidenceTimeTolerance(double tolerance){m_ResidanceTimeTolerance = tolerance;}
	double residenceTimeTolerance()const{return m_ResidanceTimeTolerance;}
	virtual bool exploreTree()=0;
	virtual bool test()=0;//\~french Procdure de tests unitaires des calculs de Harrison\~english Unit testing procedure for Harrison's calculations.
};

/**\~french Structure de dfinition de la nature des noeuds d'arbres\~english Structure for defining the nature of tree nodes.*/
struct PYC_PUBLIC TTreeNodeType{
	/**\~french Enumr de dfinition de la nature des noeuds d'arbres\~english Enumeration for defining the nature of tree nodes.*/
	enum ETreeNodeType{
		false_gt,//!<\~french Feuille faux\~english Leaf false.
		true_gt,//!<\~french Feuille vrai\~english Leaf true.
		basic_evt,//!<\~french vnement de base\~english Basic event.
		house_evt,//!<\~french drapeau\~english Flag.
		or_gt,//!<\~french Porte ou\~english OR gate.
		and_gt,//!<\~french Porte et\~english AND gate.
		xor_gt,//!<\~french Porte ou exclusif\~english XOR gate.
		atleast_gt,//!<\~french Porte au moins K/N\~english At least K/N gate.
		atmost_gt//!<\~french Porte au plus K/N\~english At most K/N gate.
	};
};

/**\~french Classe des noeuds d'arbres de dfaillances.

Chaque noeud possde une nature (kind) dtermine par l'numr TTreeNodeType.
Les noeuds de nature atleast_gt et atmost_gt ont en plus une valeur de k donne par k().
Les noeuds portes (or_gt, and_gt et atleast_gt, atmost_gt) peuvent avoir des enfants.
Chaque enfant peut tre appel avec ou sans ngation
\~english Class for failure tree nodes.

Each node has a nature (kind) determined by the enumeration `TTreeNodeType`.
Nodes with the nature `atleast_gt` and `atmost_gt` also have a value of `k` defined by `k()`.
Gate nodes (`or_gt`, `and_gt`, `atleast_gt`, `atmost_gt`) can have children.
Each child can be called with or without negation.*/
class ITreeNode{
public:
	/**\~french Retourne le nom du noeud\~english Returns the name of the node.*/
	virtual char const*name()const=0;
	/**\~french Retourne le libell du noeud\~english Returns the label of the node.*/
	virtual char const*label()const=0;
	/**\~french Retourne ne nombre de parents du noeud\~english Returns the number of parents of the node.*/
	virtual size_t parentCount()const=0;
	/**\~french Retourne le ime parent du noeud\~english Returns the i-th parent of the node.*/
	virtual ITreeNode const*parent(size_t i)const=0;
	/**\~french Retourne le nombre d'enfants de la porte\~english Returns the number of children of the gate.*/
	virtual size_t sonCount()const=0;
	/**\~french Retourne le ime enfant de la porte\~english Returns the i-th child of the gate.*/
	virtual ITreeNode const*son(size_t i)const=0;
	/**\~french Retourne vrai si le ime enfant de la porte est appel avec une ngation\~english Returns true if the i-th child of the gate is called with negation.*/
	virtual bool negated(size_t i)const=0;
	/**\~french Retourne la nature du noeud\~english Returns the nature of the node.*/
	virtual TTreeNodeType::ETreeNodeType kind()const=0;
	/**\~french Retourne la valeur de k pour une porte K/N\~english Returns the value of k for a K/N gate.*/
	virtual size_t k()const=0;
	/**\~french Retourne la transition  partir de laquelle a t construit l'vnement de base\~english Returns the transition from which the basic event was constructed.*/
	virtual ITransition const*transition()const=0;
	/**\~french Retourne l'indice de sortie de la transition utilis par l'vnement de base\~english Returns the exit index of the transition used by the basic event.*/
	virtual size_t trIndex()const=0;
	int compare(ITreeNode const&other)const;//!<\~french Comparaison en fonction du nom\~english Comparison based on the name.
	bool operator<(ITreeNode const&other)const;//!<\~french Oprateur de comparaison en fonction du nom\~english Comparison operator based on the name.
	virtual std::vector<ITreeNode const*>firstLoop()const=0;//!<\~french Dtection de la premire boucle de l'arbre\~english Detection of the first loop in the tree.
};

/**\~french Classe de dfinition d'un ensemble de noeuds d'arbre.

Classe utilise pour dfinir des coupes minimales ou des chemins minimaux.
Attention ! les chemins minimaux utilisent la ngation des vnements de base.
\~english Class for defining a set of tree nodes.

This class is used to define minimal cuts or minimal paths.
Caution! Minimal paths use the negation of basic events.*/
struct PYC_PUBLIC IMinimalPathCut{
public:
	virtual size_t size()const=0;//!<\~french Retourne le nombre d'lments de l'ensemble\~english Returns the number of elements in the set.
	virtual double probability()const=0;//!<\~french Retourne la probabilit si elle a t calcule\~english Returns the probability if it has been calculated.
	virtual double contribution()const=0;//!<\~french Retourne la contribution si elle a t calcule\~english Returns the contribution if it has been calculated.
	/**\~french Retourne le ime vnement de base de l'ensemble\~english Returns the i-th basic event of the set.*/
	virtual ITreeNode const*basic_event(size_t i)const=0;
	/**\~french Retourne la transition  partir de laquelle a t  le ime vnement de base de l'ensemble\~english Returns the transition from which the i-th basic event of the set was derived.*/
	ITransition const*transition(size_t i)const{return basic_event(i)->transition();}
	/**\~french Retourne l'indice de sortie de la transition utilis par le ime vnement de base de l'ensemble\~english Returns the exit index of the transition used by the i-th basic event of the set.*/
	size_t trIndex(size_t i)const{return basic_event(i)->trIndex();}
};

/**\~french Structure de dfinition des options de gnration des arbres\~english Structure for defining tree generation options.*/
struct PYC_PUBLIC TFaultTreeOption{
	/**\~french Enumr de dfinition de la simplification des arbres (les valeurs peuvent tre cumules avec l'oprateur | ou +)\~english Enumeration for defining tree simplification options (values can be combined using the | or + operator).*/
	enum EFaultTreeOption{
		none=0,//!<\~french Pas d'options\~english No options.
		build_label=1,//!<\~french Construction de libells avec les expressions manipules\~english Construction of labels with the manipulated expressions.
		reduce_one_son_gt=2,//!<\~french Simplification des portes  simple entre\~english Simplification of gates with a single input.
		reduce_constant_gt=4,//!<\~french Simplification des portes Vrai et Faux\~english Simplification of True and False gates.
		reduce_same_op_gt=8,//!<\~french Regroupement des portes de mme oprateur\~english Grouping gates with the same operator.
		build_eb_name=16,//!<\~french Construction du nom des EB avec la transition franchie\~english Construction of the name for basic events with the crossed transition.
		remove_xor_gt=32,//!<\~french Remplace les portes A Xor B par (A ou B) et non (A et B)\~english Replace the XOR gates A XOR B with (A OR B) AND NOT (A AND B).
		unloop=64,//!<\~french Dboucle l'arbre\~english Unroll the tree.
		all=127//!<\~french Toutes les options\~english All options.
	};
	static char const*st_Names[];
	static unsigned fromStr(char const*value);//!<\~french Lit les options  partir d'une chane au format du fichier XML de paramtres\~english Reads the options from a string in the format of the parameter XML file.
	static std::string toString(int value);//!<\~french Ecrit les options sous forme d'une chane au format du fichier XML de paramtres\~english Writes the options as a string in the format of the parameter XML file.
};

/**\~french Classe de dfinition d'un profil de gnration d'arbre de dfaillance.

Le profil contient un ensemble d'affectations de valeurs  des variables.
\~english Class for defining a failure tree generation profile.

The profile contains a set of value assignments to variables.*/
class PYC_PUBLIC IProfile:public CNamedComp{
protected:
	IProfile(CNamedComp&parent,char const*name):CNamedComp(parent,name){}
public:
	virtual void setValue(char const*name, char const*value)=0;
	virtual char const*value(char const*name)const=0;
	virtual void removeValue(char const*name)=0;
	virtual void clearValues()=0;
};

/**\~french Classe de construction d'un arbre de dfaillances.

L'arbre de dfaillances est construit de manire  expliquer pourquoi l'expression initiale peut prendre des valeurs non nulles.
La construction considre que toutes les variables Pycatshoo ont une valeur fixe gale  leur valeur initiale.
Seuls les tats et les automates sont considrs comme des valeurs modifiables par le biais de franchissement de transitions.
La construction utilise dans le chanage arrire les conditions textuelles places dans les transitions.

L'arbre gnr est restitu, soit sous forme mmoire par la donne de sa porte sommet (mthode faultTree()), soit
sous forme d'un fichier XML au format OpenPSA (mthode printFaultTree()).
\~english
Class for constructing a fault tree.

The fault tree is constructed to explain why the initial expression can take non-zero values.
The construction assumes that all Pycatshoo variables have a fixed value equal to their initial value.
Only states and automata are considered as modifiable values through transition crossings.
The construction uses the textual conditions placed in the transitions during backward chaining.

The generated tree is output either as a memory structure by providing its root gate (method faultTree()), or
as an XML file in OpenPSA format (method printFaultTree()).*/
class PYC_PUBLIC IFaultTreeGenerator:public CNamedComp{//Hrite de CNamedComp car doit tre parent des CProfile
protected:
	unsigned int m_Trace;
	unsigned int m_Options;
	IFaultTreeGenerator(CSysNamedComp&sys,char const*name):CNamedComp(sys,name),m_Trace(0),m_Options(TFaultTreeOption::all){}
public:
	virtual void addEndMethod(CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(char const*),PyObject*pyMth)=0;//!<\~french Ajoute une fonction  appeler aprs la gnration\~english Add a function to be called after generation.
	/**\~french Modifie l'expression constituant la condition dfinissant le sommet de l'arbre de dfaillances\~english Modify the expression constituting the condition defining the root of the fault tree.*/
	virtual void setExpression(char const*expr)=0;
	/**\~french Retourne l'expression constituant la condition dfinissant le sommet de l'arbre de dfaillances\~english Returns the expression constituting the condition defining the root of the fault tree.*/
	virtual char const*expression()const=0;
	void setTrace(int level){m_Trace=level;}//!<\~french Modifie le niveau de trace\~english Modify the trace level.
	int trace()const{return m_Trace;}//!<\~french Retourne le niveau de trace\~english Returns the trace level.
	void setOptions(unsigned int value){m_Options=value;}
	unsigned int options()const{return m_Options;}
	virtual IProfile*addProfile(char const*name)=0;
	virtual IProfile*insertProfile(char const*name,PRF_ID place)=0;
	virtual void moveProfile(char const*name,PRF_ID place)=0;
	virtual IProfile*profile(char const*name)const=0;
	virtual void removeProfile(char const*name)=0;
	virtual void clearProfiles()=0;
	virtual bool isBuildingFT()const=0;
	/**\~french Construit l'arbre de dfaillances et retourne la porte sommet de l'arbre\~english Constructs the fault tree and returns the root gate of the tree.*/
	virtual ITreeNode const*buildFaultTree()=0;
	/**\~french Retourne la porte sommet de l'arbre ou NULL si l'arbre n'a pas t construit\~english Returns the root gate of the tree or NULL if the tree has not been constructed.*/
	virtual ITreeNode const*faultTree()const=0;
	/**\~french Enregistre l'arbre construit dans un fichier XML au format Open-PSA\~english Saves the constructed tree in an XML file in Open-PSA format.*/
	virtual bool printFaultTree(char const*fileName)const=0;
	/**\~french Calcule les coupes minimales et les enregistre dans le fichier fileName\~english Calculates the minimal cuts and saves them in the file fileName.*/
	virtual bool buildMCS(char const*fileName, double missionTime, int maxOrder, double minProba, char const*xftar=NULL)const=0;
	/**\~french Calcule les chemins minimaux et les enregistre dans le fichier fileName\~english Calculates the minimal paths and saves them in the file fileName.*/
	bool buildMPS(char const*fileName, int maxOrder, char const*xftar=NULL)const{return buildMCS(fileName,-1,maxOrder,0, xftar);}
	/**\~french Retourne un vecteur des coupes ou chemins minimaux lus dans le fichier\~english Returns a vector of minimal cuts or paths read from the file.*/
	virtual std::vector<IMinimalPathCut*>readPathsCuts(char const*fileName, size_t nbMax=-1)const=0;
};

/**\~french Structure de dfinition de l'numr de choix du traitement  effectuer\~english Structure definition of the enumeration for the choice of processing to be performed.*/
struct PYC_PUBLIC TProcessing{
	/**\~french Enumr de dfinition du traitement envisag (les valeurs peuvent tre cumules avec l'oprateur | ou +)\~english Enumeration for defining the intended processing (the values can be combined with the operator | or +).*/
	enum EProcessing{
		none = 0,//!<\~french Pas de traitement envisag\~english No processing intended.
		simulation = 1,//!<Simulation
		sequence_tree = 2,//!<\~french Exploration de l'arbre des squences\~english Exploration of the sequence tree.
		fault_tree = 4,//!<\~french Construction d'un arbre de dfaillances\~english Construction of a fault tree.
		seq_monte_carlo = 8,//!<\~french Monte Carlo squentiel\~english Sequential Monte Carlo.
		importance_sampling = 16,//!<\~french Tirage prfrentiel\~english Preferential sampling.
		monte_carlo_decision_process = 32,//!<\~french Processus de dcision markovien\~english Markov decision process.
		all = 63//!<\~french Tous les traitements\~english All processing.
	};
	static char const*st_Names[];
	static unsigned fromStr(char const*value);//!<\~french Lit les options  partir d'une chane au format du fichier XML de paramtres\~english Reads the options from a string in the parameter XML file format.
	static std::string toString(int value);//!<\~french Ecrit les options sous forme d'une chane au format du fichier XML de paramtres\~english Writes the options as a string in the parameter XML file format.
};

/**\~french Interface d'enregistrement d'une BdC
La surchage de cette classe lors de la dfinition d'une base de connaissances permet de dfinir entirement un systme grce  un fichier de paramtrage
La classe surcharge doit surcharger la mthode newComponent() pour retourner un objet de la classe souhaite
Il est possible de surcharger la mthode classes qui retourne la liste des classes disponibles dans la BdC ainsi que la mthode version

exemple:
\~english Interface for recording a Knowledge Base
Overriding this class when defining a knowledge base allows for fully defining a system through a configuration file.
The overriding class must override the `newComponent()` method to return an object of the desired class.
It is also possible to override the `classes` method, which returns the list of available classes in the Knowledge Base, as well as the `version` method.

example:
\~
~~~~{.cpp}
//Dfinition of a knowledge base in a .cpp file
class BdC:public IBdC
{
public:
	//Constructeur de la BdC
	BdC():IBdC("BdC"){}
	std::vector<std::string>classes()const override{
		static char const*cls[]={"Classe1","Classe2","Classe3"};
		return std::vector<std::string>(cls,cls+sizeof(cls)/sizeof(char*));
	}
	char const*version()const{return "1.2.1";}
	std::vector<std::string>classes(){
		static std::vector<std::string>cls={"Classe1","Classe2","Classe3"};
		return cls;
	}
	std::vector<std::string>parentClass(char const*name){
		static std::vector<std::string>cls;
		if(!strcmp(name,"Classe3"))
			cls.push_back("Classe1");
		return cls;
	}
	bool verifyModel(CSystem&system, TProcessing::EProcessing processing)const override{
		if(processing==TProcessing::fault_tree)
			return false;//Knowledge base dont support fault tree generation
		return true;
	}
	//Surcharge de la mthode newComponent
	CComponent*newComponent(char const*clName,char const*name,CSysNamedComp&parent)const override{
		//Cration d'un composant pour les classes connues
		if(0==strcmp(clName,"Classe1"))
			return new Classe1(name,parent) ;
		if(0==strcmp(clName,"Classe2"))
			return new Classe2(name,parent) ;
		if(0==strcmp(clName,"Classe3"))
			return new Classe3(name,parent) ;
		return NULL ;//Classe inconnue, on retourne NULL
	}
};

BdC st_BdC;//Singleton
~~~~
*/
class PYC_PUBLIC IBdC
{
	PyObject*m_Self;//Rfrence  l'objet Python associ
	char const*m_Name;
	IBdC(IBdC const&):m_Self(NULL),m_Name(NULL){}
	IBdC&operator=(IBdC const&){delete[]const_cast<char*>(m_Name);m_Self=NULL;m_Name=NULL;return*this;}
	//Mthodes permettant la surcharge en Python (renseignes lors de l'exposition de IBdC en Python)
	static CComponent*(*st_newComponent)(PyObject*,char const*,char const*,CSysNamedComp&);
	static std::vector<std::string>(*st_classes)(PyObject*,char const*fmt,...);
	static std::vector<std::string>(*st_parentClasses)(PyObject*,char const*clName);
	static char const*(*st_version)(PyObject*,char const*fmt,...);
	static CSystem*(*st_buildSystem)(PyObject*,char const*fmt,...);
	static bool(*st_verifyModel)(PyObject*,CSystem&,int);
	static PyObject*(*st_getOverride)(PyObject*,char const*fctName);
	friend void export_system();
	void registerKB(char const*pycVersion);
protected:
	explicit IBdC(char const* name, char const* pyVersion = VER_FILEVERSION_STR);//!<\~french Construit une BdC de nom name\~english Construct a Knowledge Base named `name`.
public:
	IBdC(PyObject* self, char const* name);//!<\~french Construction pour Python\~english Construction for Python
	virtual~IBdC();
	PyObject*self(){return m_Self;}
	char const*name()const{return m_Name;}//!<\~french Retourne le nom de la BdC\~english Returns the name of the Knowledge Base
	char const*basename()const{return m_Name;}//!<\~french Retourne le nom de la BdC\~english Returns the name of the Knowledge Base
	/**\~french Ajoute un composant de classe clName et de nom name dans le systme CSystem\~english Adds a component of class `clName` and name `name` to the `CSystem` system.*/
	virtual CComponent* newComponent(char const* clName, char const* name, CSystem& system)const;
	/**\~french Ajoute un composant de classe clName et de nom name dans le systme ou le composant parent\~english Adds a component of class `clName` and name `name` to the system or the parent component.*/
	virtual CComponent* newComponent(char const* clName, char const* name, CSysNamedComp& parent)const;
	/**\~french Construit le systme  partir de donnes contenues dans le rpertoire fourni (par dfaut non implment)\~english Constructs the system from data contained in the provided directory (default not implemented)*/
	virtual CSystem*buildSystem(char const*resourcesDir)const;
	/**\~french Vrifie le systme pour le traitement demand\~english Check the system for the requested processing*/
	virtual bool verifyModel(CSystem&system, TProcessing::EProcessing)const;
	/**\~french Retourne la liste des classes disponibles\~english Returns the list of available classes*/
	virtual std::vector<std::string>classes()const;
	/**\~french Retourne le nom de la classe mre ou NULL\~english Returns the name of the parent class or NULL*/
	virtual std::vector<std::string>parentClasses(char const*name)const;
	/**\~french Retourne la version de la BdC\~english Returns the version of the Knowledge Base.*/
	virtual char const*version()const;
};

/**\~french Classe de dfinition d'une stratgie pour un ensemble de transitions de dcision.
Pour construire un lment de cette classe, il faut appeler la mthode IMDPManager::newStrategy
\~english Class for defining a strategy for a set of decision transitions.  
To construct an element of this class, the `IMDPManager::newStrategy` method must be called.*/
class PYC_PUBLIC IStrategy{
protected:
	IStrategy(){}
	virtual~IStrategy(){}
public:
	virtual void setDecision(ITransition*transition, unsigned decIndex, int outIndex)=0; //!<\~french Modifie la dcision d'indice decIndex pour la sortie de la transition transition 'outIndex donne l'indice de sortie de la transition; si -a, il n'y a pas de dcision)\~english Modifies the decision at index `decIndex` for the output of the transition. `outIndex` gives the output index of the transition; if `-1`, there is no decision.
	virtual int decision(ITransition*transition, unsigned decIndex)const=0; //!<\~french Retourne la dcision d'indice decINdex pour la sortie de la transition transition\~english Returns the decision at index `decIndex` for the output of the transition.
	virtual void setReward(double reward)=0; //!<\~french Donne le rsultat de la simulation\~english Provides the result of the simulation.
	virtual double reward()const=0; //!<\~french Retourne la valeur du gain stocke\~english Returns the stored gain value.
	virtual std::vector<ITransition*>transitions()const=0;//!<\~french Retourne la liste des transitions de dcision gres\~english Returns the list of managed decision transitions.
	virtual std::vector<int> decisions(ITransition* transition)const=0;//!<\~french Retourne la liste des dcisions pour la transition\~english Returns the list of decisions for the transition.
	virtual bool operator==(IStrategy const& other)const=0;
};

/**\~french Structure de dfinition des types de Solver\~english Structure for defining Solver types.*/
struct PYC_PUBLIC TMDPSolverType{
	/**\~french Enumr de dfinition de la simplification des arbres\~english Enumeration for defining tree simplification.*/
	enum ESolverType{
		custom=0, // Alogrithme fourni par l'utilisateur
		pei, // Algorithme pei
		mras // Algorithme mras
	};
	static char const*st_Names[];
	static ESolverType fromStr(char const*value);
};

/**\~french Classe de base des algorithme d'optimisation de processus markoviens.
Cette classe doit tre surcharge pour implmenter un solveur.
2 solveurs sont primplments : mras et pei.
\~english Base class for optimization algorithms of Markovian processes.
This class must be overridden to implement a solver.
Two solvers are pre-implemented: `mras` and `pei`.*/
class PYC_PUBLIC IMDPSolver{
	friend class IMDPManager;
	IMDPManager*m_Manager;
	void setManager(IMDPManager*manager){m_Manager=manager;}
protected:
	IMDPSolver():m_Manager(NULL){}
public:
	virtual~IMDPSolver(){}
	IMDPManager*manager(){return m_Manager;}//!<\~french Retourne le MDPManager associ\~english Returns the associated `MDPManager`.
	virtual void setParameter(char const*nomParametre, double valeurParametre)=0;//!<\~french Renseigne ou modifie la valeur du paramtre de nom nomParametre\~english Sets or modifies the value of the parameter named `nomParametre`.
	virtual double parameter(char const* nomParametre) = 0; //!<\~french Retourne la valeur du parametre de nom nomParametre ou NAN s'il n'est pas dfini\~english Returns the value of the parameter named `nomParametre` or NAN if not defined.
	double parameter(char const* nomParametre, double defaut); //!<\~french Retourne la valeur du parametre de nom nomParametre ou default s'il n'est pas dfini\~english Returns the value of the parameter named `nomParametre` or default if not defined.
	virtual std::vector<IStrategy*>nextStrategies(std::vector<IStrategy*>const& oldStrategies, unsigned int iterIndex)=0; //!<\~french Retourne la liste des nouvelles stratgies  valuer compte tenu des rsultats obtenus  l'itration prcdente (rsultats vides lors de la premire itration)\~english Returns the list of new strategies to evaluate based on the results obtained in the previous iteration (empty results during the first iteration).
	virtual bool verify()=0; //!<\~french Vrifie que les parametres sont dans un tat qui permet de lancer l'algorithme de gnration des stratgies\~english Verifies that the parameters are in a state that allows the strategy generation algorithm to be executed.
	virtual TMDPSolverType::ESolverType type() = 0;//!<\~french Retourne le type du solveur\~english Returns the type of the solver.
	virtual std::vector<std::string>parameterNames()const=0;//!<\~french Retourne la liste des noms des paramtres du solveur\~english Returns the list of solver parameter names.
	virtual std::vector<std::string>definedParameterNames()const = 0;//!<\~french Retourne la liste des noms des paramtres explicitement dfinis\~english Returns the list of explicitly defined parameter names.
};

/**\~french Classe de creation d'un gestionnaire des processus markoviens.

exemple:
\~english Class for creating a Markov process manager.

example:
\~
~~~~{.cpp}
class MDPOperationController:
	public CComponent
{
	IVariable&InspectionPeriod,&InspectionDuration,&RefurbishProba,&ReplaceProba,&decision,&globalBalance;
	IReference&deviceBalance;
	IAutomaton&automaton;
	IState&operation,&inspection,&choice;
	ITransition*trDecision;
	IMessageBox&device;
public:
	enum EDecision{
		none=0,
		repare,
		refurbish,
		replace};
	MDPOperationController(char const*name,CSystem&sys)
		:CComponent(name,sys)
		,RefurbishProba(*addVariable("RefurbishProba", TVarType::t_double, 0))
		,ReplaceProba(*addVariable("ReplaceProba", TVarType::t_double, 0))
		,InspectionPeriod(*addVariable("InspectionPeriod", TVarType::t_double, 9))
		,InspectionDuration(*addVariable("InspectionDuration", TVarType::t_double, 3))
		,decision(*addVariable("decision", TVarType::t_int, 0))
		,globalBalance(*addVariable("globalBalance", TVarType::t_double, 0))
		,deviceBalance(*addReference("deviceBalance"))
		,automaton(*addAutomaton("automaton"))
		,operation(*automaton.addState("operation",1))
		,choice(*automaton.addState("choice",2))
		,inspection(*automaton.addState("inspection",3))
		,device(*addMessageBox("device"))
	{
		deviceBalance.addCallback("onBalanceChange",&MDPOperationController::onBalanceChange);

		automaton.setInitState(operation);

		ITransition*tr=operation.addTransition("operation->choice");
		tr->setDistLaw(TLawType::defer,InspectionPeriod);
		tr->addTarget(choice);

		trDecision=choice.addTransition("decision");
		trDecision->setDistLaw(TLawType::inst,1);
		trDecision->distLaw()->setParameter(RefurbishProba,1);
		trDecision->distLaw()->setParameter(ReplaceProba,2);
		trDecision->addTarget(inspection);//Rparation, standard type
		trDecision->addTarget(inspection,4);//Refurbish
		trDecision->addTarget(inspection,5);//Replace
		trDecision->addCallback("onDecision",&MDPOperationController::onDecision);

		tr=inspection.addTransition("inspection->operation");
		tr->setDistLaw(TLawType::defer,InspectionDuration);
		tr->addTarget(operation);
		tr->addCallback("onDecision",&MDPOperationController::onDecision);

		device.addExport(automaton,"state");
		device.addExport(decision,"decision");
		device.addImport(deviceBalance,"balance");
	}
	void onDecision(){
		decision.setIValue(trDecision->firedState()+1);//Pour correspondre  l'numr de dcision
	}
	void onBalanceChange(){
		globalBalance.setDValue(deviceBalance.sumValue());
	}
};

class MDPDevice :
	public CComponent
{
	IVariable&Lambda,&lambda,&Mu,&BetaOperation,&BetaAge,&IncomeRate,&RefurbishCost,&ReplaceCost,&RepairCost,&balance,&cumulativeOnTime,&lastReplaceTime,&lastOnEnter;
	IReference&controllerState,&controllerDecision;
	IAutomaton&automaton;
	IState&off,&on,&ko;
	IMessageBox&controller;
public:
	MDPDevice(char const*name,CSystem&sys)
		:CComponent(name,sys)
		,Lambda(*addVariable("Lambda" , TVarType::t_double, 0.01))
		,lambda(*addVariable("lambda", TVarType::t_double, 0.0))
		,Mu(*addVariable("Mu", TVarType::t_double, 0.1))
		,BetaOperation(*addVariable("BetaOperation", TVarType::t_double, 0.1))
		,BetaAge(*addVariable("BetaAge", TVarType::t_double, 0.01))
		,IncomeRate(*addVariable("IncomeRate"   , TVarType::t_double, 0.))
		,RefurbishCost(*addVariable("RefurbishCost", TVarType::t_double, 0.))
		,ReplaceCost(*addVariable("ReplaceCost", TVarType::t_double, 0.))
		,RepairCost(*addVariable("RepairCost", TVarType::t_double, 0.))
		,balance(*addVariable("balance", TVarType::t_double, 0))
		,cumulativeOnTime(*addVariable("cumulativeOnTime", TVarType::t_double, 0))
		,lastReplaceTime(*addVariable("lastReplaceTime", TVarType::t_double, 0))
		,lastOnEnter(*addVariable("lastOnEnter", TVarType::t_double, 0))
		,controllerState(*addReference("controllerState"))
		,controllerDecision(*addReference("controllerDecision"))
		,automaton(*addAutomaton("automaton"))
		,on(*automaton.addState("on",1))
		,off(*automaton.addState("off",2))
		,ko(*automaton.addState("ko",3))
		,controller(*addMessageBox("controller"))
	{
		on.addCallback("onOnEnter", &MDPDevice::onOnEnter,  1);
		on.addCallback("onOnLeave", &MDPDevice::onOnLeave, -1);
		controllerDecision.addCallback("onDecisionChange",&MDPDevice::onDecisionChange);

		automaton.setInitState(off);

		ITransition*trans = off.addTransition("off->on");
		trans->setCondition(&MDPDevice::condOffOn);
		trans->addTarget(on, TTransType::trans);

		trans = on.addTransition("on->ko");
		trans->setDistLaw(TLawType::expo, lambda);
		trans->addTarget(ko, TTransType::fault);
		trans->setModifiable(TModificationMode::discrete_modification);

		trans = ko.addTransition("ko->on");
		trans->setDistLaw(TLawType::expo, Mu);
		trans->addTarget(on, TTransType::rep);
		trans->addCallback("onKoLeave", &MDPDevice::onKoLeave);

		trans = ko.addTransition("ko->off");
		trans->addTarget(off, TTransType::rep);
		trans->setCondition(&MDPDevice::condOffOn,true);
		trans->addCallback("onKoLeave", &MDPDevice::onKoLeave);

		trans = on.addTransition("on->off");
		trans->setCondition(&MDPDevice::condOffOn,true);
		trans->addTarget(off, TTransType::trans);

		controller.addExport(balance,"balance");
		controller.addImport(controllerState,"state");
		controller.addImport(controllerDecision,"decision");

		addCheckMethod("check",&MDPDevice::check);
	}

	void check(){
		updateLambda();
	}
	void updateLambda(){
		lambda.setDValue(Lambda.dValue()*std::exp(BetaAge.dValue() * (currentTime()-lastReplaceTime.dValue())
			+BetaOperation.dValue() * cumulativeOnTime.dValue()));
	}
	bool condOffOn(){
		return controllerState.iValue(0)!=3;
	}
	void onOnEnter(){
		lastOnEnter.setDValue(currentTime());
	}
	void onOnLeave(){
		cumulativeOnTime.setDValue(cumulativeOnTime.dValue() + currentTime() - lastOnEnter.dValue());
		balance.setDValue(balance.dValue() + (currentTime() - lastOnEnter.dValue()) * IncomeRate.dValue());
	}
	void onKoLeave(){//Calcul du cot de rparation
		if(controllerDecision.iValue(0)<=MDPOperationController::repare)
			balance.setDValue(balance.dValue() - RepairCost.dValue());
	}
	void onDecisionChange(){
		int dec=controllerDecision.iValue(0);
		if(dec==MDPOperationController::none)
			return;//C'est la reprise de cycle
		if(dec==MDPOperationController::refurbish){
			balance.setDValue(balance.dValue() - RefurbishCost.dValue());
			cumulativeOnTime.setDValue(0);
		}else if(dec==MDPOperationController::replace){
			balance.setDValue(balance.dValue() - ReplaceCost.dValue());
			lastReplaceTime.setDValue(currentTime());
			cumulativeOnTime.setDValue(0);
		}
		updateLambda();
	}
};
int main(int argc, char* argv) {
	try {
		CSystem& system = *new CSystem("MDP");
		system.setNbSeqToSim(100);
		system.setTMax(180);
		system.setRNG("yarn5");
		system.setRNGBlockSize(100);
		system.setRNGSeed(1);
		auto& device = *new MDPDevice("dev", system);
		auto& controller = *new MDPOperationController("cont", system);
		device.messageBox("controller")->connectTo(*controller.messageBox("device"));
		device.variable("IncomeRate")->setDValue(10.0);
		device.variable("RepairCost")->setDValue(10.);
		device.variable("RefurbishCost")->setDValue(100.);
		device.variable("ReplaceCost")->setDValue(150.);
		system.setUseMDPManager(true);
		auto& MDPManager = *system.MDPManager();
		MDPManager.setTrace(1);
		MDPManager.setSolver(TMDPSolverType::mras);
		MDPManager.setNbIterations(10);
		MDPManager.addDecisionTransition(controller.transition("Decision"), 15);
		MDPManager.setRewardValue(controller.variable("globalBalance"));
		auto& solver = *MDPManager.solver();
		solver.setParameter("strategiesIncrement", 1.1);
		solver.setParameter("simulationsIncrement", 1.01);
		solver.setParameter("mixPkP0", 0.3);
		solver.setParameter("mixPkPkm1", 0.7);
		solver.setParameter("epsilon", 5);
		solver.setParameter("ROk", 0.5);
		solver.setParameter("N", 30);
		if(MDPManager.optimizeStrategy());
		MDPManager.printBestStrategy("strategy.xml");
		std::string msg = ILogManager::glLogManager().messages();
		if(!msg.empty());
		std::cout << msg << "\n";
	}
	catch(std::exception& e) {
		std::cerr << x.what() << std::endl;
	}
}
~~~~
*/
class PYC_PUBLIC IMDPManager :public CSysNamedComp{
	friend void IMDPManager_setRewardValue(IMDPManager&, CSysNamedComp*, char const*, PyObject*);
public:
	enum TMethod{
		met_start_iter,
		met_end_iter,
		met_start_smp,
		met_end_smp};
protected:
	friend void IMDPManager_setMethod(IMDPManager&, IMDPManager::TMethod, CSysNamedComp*, char const*, PyObject*);
	unsigned int m_Trace;
	IMDPManager(CSystem&sys,char const*name):CSysNamedComp(sys,name),m_Trace(0),m_Solver(NULL){}
	virtual ~IMDPManager();
	IMDPSolver* m_Solver;
	virtual void setMethod(TMethod,CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(),PyObject*pyMth=NULL)=0;
	virtual void setRewardValue(CSysNamedComp*comp,char const*name,double(CSysNamedComp::*method)(),PyObject*pyMth)=0; //!<\~french Designe la mtode permettant d'obtenir la valeur de gain en fin de squence\~english Specifies the method for obtaining the gain value at the end of the sequence.
public:
	void setTrace(int level){m_Trace=level;}//!<\~french Modifie le niveau de trace\~english Modifies the trace level.
	int trace()const{return m_Trace;}//!<\~french Retourne le niveau de trace\~english Returns the trace level.
	virtual void setRewardValue(IVariable* var)=0; //!<\~french Designe la variable permettant d'obtenir la valeur de gain en fin de squence\~english Specifies the variable for obtaining the gain value at the end of the sequence.
	void setRewardValue(IVariable&var){setRewardValue(&var);}
	void setRewardValue(CSysNamedComp*comp,char const*name,double(CSysNamedComp::*method)()){setRewardValue(comp,name,method,NULL);}
	virtual void setSolver(TMDPSolverType::ESolverType type)=0; //!<\~french Choix de l'algorithme utilis pour optimiserles dcisions (uniquement parmi mras et pei)\~english Choice of the algorithm used to optimize decisions (only between `mras` and `pei`).
	virtual IMDPSolver* solver(){return m_Solver;}; //!<\~french Retourne le solveur courant utilis\~english Returns the current solver being used.
	virtual void setSolver(IMDPSolver* solver){delete m_Solver;m_Solver = solver;solver->setManager(this);}; //!<\~french Renseigne le solveur dfini par l'utilisateur\~english Sets the solver defined by the user.
	template<typename TYPE> void setStartIterationMethod(TYPE*comp,char const*name,void(TYPE::*method)()){setMethod(met_start_iter,comp,name,static_cast<void(CSysNamedComp::*)()>(method));}//!<\~french Dsigne la mthode appele avant chaque lancement des simulations de MonteCarlo de toutes les stratgies\~english Specifies the method called before each Monte Carlo simulation run for all strategies.
	template<typename TYPE> void setEndIterationMethod(TYPE*comp,char const*name,void(TYPE::*method)()){setMethod(met_end_iter,comp,name,static_cast<void(CSysNamedComp::*)()>(method));}//!<\~french Dsigne la mthode appele aprs chaque lancement des simulations de MonteCarlo de toutes les stratgies\~english Specifies the method called after each Monte Carlo simulation run for all strategies.
	template<typename TYPE> void setStartSampleMethod(TYPE*comp,char const*name,void(TYPE::*method)()){setMethod(met_start_smp,comp,name,static_cast<void(CSysNamedComp::*)()>(method));}//!<\~french Dsigne la mthode appele avant chaque lancement d'une simulations de Monte Carlo d'une stratgie\~english Specifies the method called before each Monte Carlo simulation run for a single strategy.
	template<typename TYPE> void setEndSampleMethod(TYPE*comp,char const*name,void(TYPE::*method)()){setMethod(met_end_smp,comp,name,static_cast<void(CSysNamedComp::*)()>(method));}//!<\~french Dsigne la mthode appele aprs chaque lancement d'une simulations de Monte Carlo d'une stratgie\~english Specifies the method called after each Monte Carlo simulation run for a single strategy.
	/**\~french Ajoute une transition de dcision au gestionnaire de MDP et le nombre de dcision  prendr epour cette transition (on vrifiera que l'tat de dpart ne ocmporte que cette transition de sortie et la transition se verra attribuer une distribution spcifique de type inst_dec permettant de grer le choix de l'indice de sortieen fonction d'une stratgie communique pralablement.
	Attention! il est du ressort de l'utilisateur d'attribuer un nombre de dcisions correpsondant au nombre maximum de franchissements de la transition effectus pendant une simulation
	\~english Adds a decision transition to the MDP manager and specifies the number of decisions to be made for this transition. (It will ensure that the starting state includes only this outgoing transition, and the transition will be assigned a specific distribution of type `inst_dec`, which manages the choice of the output index based on a previously communicated strategy. 
	Note! It is the user's responsibility to assign a number of decisions corresponding to the maximum number of crossings of the transition during a simulation.*/
	virtual void addDecisionTransition(ITransition* transition, unsigned int nbDecision)=0;
	virtual void removeDecisionTransition(char const*name)=0;//!<\~french Supprime une transition de dcision du gestionnaire de MDP\~english Removes a decision transition from the MDP manager.
	void removeDecisionTransition(ITransition* transition){removeDecisionTransition(transition->name());} //!<\~french Supprime une transition de dcision du gestionnaire de MDP\~english Deletes a decision transition from the MDP manager.
	virtual unsigned int nbDecisions(ITransition*transition)const=0;//!<\~french Retourne le nombre de dcisions de la transition\~english Returns the number of decisions for the transition.
	virtual void setTransitionDistribution(char const*name, unsigned int iDecision, std::vector<double>const&dist)=0;//!<\~french Modifie la iDecision me distribution de choix pour la transition de nom name\~english Modifies the `iDecision`-th choice distribution for the transition named `name`.
	void setTransitionDistribution(ITransition* transition, unsigned int iDecision, std::vector<double>const&dist){setTransitionDistribution(transition->name(), iDecision, dist);}//!<\~french Modifie la iDecision me distribution de choix pour la transition\~english Modifies the `iDecision`-th choice distribution for the transition.
	virtual std::vector<double>transitionDistribution(char const*name, unsigned int iDecision)const=0;//!<\~french Retourne la iDecision me distribution de choix pour la transition de nom name\~english Returns the `iDecision`-th choice distribution for the transition named `name`.
	std::vector<double>transitionDistribution(ITransition* transition, unsigned int iDecision)const{return transitionDistribution(transition->name(),iDecision);}//!<\~french Retourne la iDecision me distribution de choix pour la transition\~english Returns the `iDecision`-th choice distribution for the transition.
	virtual void clearTransitionDistribution(char const*name)=0;//!<\~french Supprime la distribution de choix pour la transition de nom name\~english Deletes the choice distribution for the transition named `name`.
	void clearTransitionDistribution(ITransition* transition){clearTransitionDistribution(transition->name());}//!<\~french Supprime la distribution de choix pour la transition\~english Deletes the choice distribution for the transition.
	virtual std::vector<ITransition*>decisionTransitions()const=0;
	virtual std::vector<IStrategy*>strategies()=0; //!<\~french Retourne la liste des stratgies dfinies pour l"itration courante\~english Returns the list of strategies defined for the current iteration.
	virtual int nbIterations()const=0;
	virtual void setNbIterations(int nb)=0;
	virtual int currentIteration()const = 0;
	virtual IStrategy*newStrategy(IStrategy const*strategy=NULL)=0;//!<\~french Fournit une nouvelle stratgie de la bonne forme soit comme copie de la stratgie fournie, soit initialise avec des choix 0\~english Provides a new strategy in the correct format, either as a copy of the provided strategy or initialized with choices set to 0.
	virtual IStrategy*newStrategy(bool bDefault)=0;//!<\~french Fournit une nouvelle stratgie tire alatoirement de manire uniforme ou utilisant la stratgie par dfaut\~english Provides a new strategy drawn randomly in a uniform manner or using the default strategy.
	virtual void deleteStrategy(IStrategy*strategy) = 0;//!<\~french Supprime la stratgie de la liste et la dtruit\~english Remove the strategy from the list and destroy it.
	virtual IStrategy*currentStrategy()=0;
	virtual IStrategy*bestStrategy()const=0; //!<\~french Retourne la meilleure stratgie courante\~english Returns the current best strategy.
	virtual bool printBestStrategy(char const*fileName)const=0; //!<\~french Enregistre la meilleure stratgie dans le fichier fileName\~english Save the current best strategy in the file fileName.
	virtual bool printStrategy(IStrategy& strategy, char const* fileName)const = 0;//!<\~french Enregistre la stratgie dans le fichier fileName\~english Save the strategy in the file fileName.
	virtual IStrategy* loadStrategy(char const* fileName) = 0;//!<\~french Construit la stratgie  partir du fichier fileName\~english Build the strategy from the file fileName.
	virtual bool optimizeStrategy()=0;//!<\~french Lance la recherche de la meilleure stratgie\~english Initiates the search for the best strategy.
};
#endif
