/*************************************
*         Copyright 2015 EDF         *
*************************************/
#ifndef SYSTEME_H
#define SYSTEME_H
/**\file
\~french Fichier de dfinition de la classe CSystem qui contient la dfinition du systme  tudier et permet de grer les diffrents traitements proposs par PyCATSHOO.
\~english Definition file of the CSystem class which contains the definition of the system to be studied and allows the management of the different treatments offered by PyCATSHOO.
*/

#include "Interfaces.h"

class CXMLNode;
class CSequence;
class IRNG;
class CSystemP;
class CVariableDbl;
class CReference;
class CIndicator;
class IFunctionalMockupUnit;

/**\~french Structure de dfinition de l'numr de dfinition des mthodes systme.\~english Structure for defining the enumeration of system method definitions.*/
struct TSysMethod{
	/**\~french Enumr de dfinition des mthodes systme\~english Enumeration for defining system methods.*/
	typedef enum ESysMethod{
		check,//!<\~french Mthode appele une seule fois avant le lancement de la simulation (vrification, initialisation)\~english Method called once before the simulation starts (verification, initialization).
		start,//!<\~french Mthode appele en dbut de simulation de chaque squence (initialisation de squence)\~english Method called at the beginning of the simulation of each sequence (sequence initialization).
		pre_sensib,//!<\~french Mthode appele avant l'exploitation des mthodes sensibles actives\~english Method called before the execution of the activated sensitive methods.
		in_sensib,//!<\~french Mthode appele  la fin de chaque exploitation des mthodes sensibles (peut demander une rexploitation si une mthode sensible y est active)\~english Method called at the end of each execution of the sensitive methods (may request re-execution if a sensitive method is activated).
		post_sensib//!<\~french Mthode appele  la fin de l'exploitation des mthodes sensibles (pas de rexploitation mme si une mthode sensible y est active)\~english Method called at the end of the execution of the sensitive methods (no re-execution even if a sensitive method is activated).
	}ESysMethod;
};
/**\~french Structure de dfinition de l'numr de dfinition des lments du systme  tracer.\~english Structure for defining the enumeration of system elements to trace.*/
struct TSysTrace{
	/**\~french Enumr de dfinition des lments du systme  tracer.
	Les lments peuvent tre cumuls
	\~english Enumeration for defining the elements of the system to trace.
	The elements can be accumulated.
	*/
	typedef enum ESysTrace{
		none=0,			//!<\~french Aucun\~english None.
		seq_number=1,	//!<\~french Numro de squence\~english Sequence number.
		planning=2,		//!<\~french Planning des transitions\~english Transition schedule.
		sys_methods=4,	//!<\~french Appels des mthodes systme\~english System method calls.
		sens_methods=8,	//!<\~french Appels des mthodes sensibles\~english Calls to sensitive methods.
		other=1<<30,	//!<\~french Autres\~english Other.
		traces=other-1
	}ESysTrace;
};
/**\~french Structure de dfinition de l'numr de type de simulation en cours\~english Structure for defining the enumeration of the current simulation type.*/
struct TSimuMode{
	/**\~french Enumr de dfinition du type de simulation en cours\~english Enumeration for defining the current simulation type.*/
	typedef enum ESimuMode{
		sm_stopped=0,		//!<\~french Aucune simulation en cours\~english No simulation in progress.
		sm_imp_smp,			//!<\~french Simulation pour le tirage prfrentiel\~english Simulation for preferential sampling.
		sm_smc,				//!<\~french Simulation pour Monte Carlo squentiel\~english Simulation for sequential Monte Carlo.
		sm_ste,				//!<\~french Simulation pour explorateur d'arbres de squences\~english Simulation for sequence tree explorer.
		sm_standard,		//!<\~french Simulation standard\~english Standard simulation.
		sm_interactive		//!<\~french Simulation interactive\~english Interactive simulation.
	}ESimuMode;
};
/**\~french Structure de dfinition des options (cumulables) de gestion du systme en mode FMU\~english Structure for defining the options (accumulative) for managing the system in FMU mode.*/
struct TFMUOptions{
	/**\~french Enumr de dfinition des options (cumulables) de gestion du systme en mode FMU\~english Enumeration for defining the options (accumulative) for managing the system in FMU mode.*/
	typedef enum EFMUOptions{
		none=0,				//!<\~french Aucune option\~english No option.
		simpleStateStorage=1,//!<\~french Stockage simplifi de l'tat du systme\~english Simplified storage of the system state.
		noFMUControl=2,		//!<\~french Suppression des contrles des mthodes fmu2\~english Removal of fmu2 method checks.
		noFMULog=4			//<!\~french Suppression des log des mthodes fmu2\~english Removal of fmu2 method logs.
	}EFMUOptions;
};
/**\~french Classe de gestion d'un systme  simuler.
	Cette classe doit comprendre tous les composants de haut niveau du systme  simuler.

	Cette class peut tre utilise sous 2 formes :
	- telle quelle : le systme peut tre cr soit par le fichier de paramtres soit par des mthodes externes
	- en sous classant CSystem et en surchargeant le constructeur pour y construire les composants du systme et les liens entre eux.

	L'intrt de la 2me mthode consiste  permettre l'ajout de mthodes systme pour dfinir les indicateurs ou les cibles.

	La classe propose des outils de monitoring et de trace.

	La classe permet de dfinir des cibles  travers des fonctions boolennes. Les cibles servent de condition d'arrt des squences.

	La classe permet enfin de grer le droulement de la simulation (initialisation, lancement, redmarrage, gestion des rsultats).

	La classe permet de grer l'utilisation d'algorithmes non standard :
	- Tirage prfrentiel : setUseImportanceSampling(), l'outil de destion de l'algorithme est obtenu ensuite par importanceSampling()
	- Monte Carlo squentiel : setUseSeqMonteCarlo(), l'outil de gestion de l'algorithme est obtenu ensuite par seqMonteCarlo()

exemple1 (cration d'un systme avec une classe ddie):
~~~~{.cpp}
//Inclusion de la dfinition de la BdC utilise
#include "BdCDF.h"

class DFSystem:public CSystem
{
	//Reference to the end component of the system
	CComponent*c;
	//Definition of an indicator function
	float indic(){
		return (float)(c->getVariable("feeding")->bValue());
	}
	//Definition of a target function
	bool target(){
		return !c->getVariable("feeding")->bValue()
 && c->getVariable("NFED")->bValue();
	}
	//   c1--cb1
	//  / |   |  \
	// cS  \ /   cF
	//  \   |    /
	//   --c2---
public:
	DFSystem():CSystem("2files"){
		//Build the components thanks to newComponent
		newComponent("cS","DFSource");
		newComponent("c1","DFComposant");
		//Direct build dof composants if classes are availables
		new DFComposant("cb1",*this);
		c = new DFFin("cF",*this);
		new DFCompSecours("c2",*this);

		//Connection of composants
		connect("cS","aval","c1","amont");
		ICnctInfo&info=connect("c1","aval","cb1","amont");
		//Add information in the connection
		info.setValue("NomInfo",3.5);
		...

		//Modify initial value
		getComponent("c1")->getVariable("fed")->setValue(true);

		//Modify initial state
		getComponent("cb1")->setInitState("NOK");

		//Add the previously defined indicator for calcul of mean, standard deviation and distribution
		IIndicator*ind=addIndicator("F.feeding",&DFSystem::indic);
		ind->setRestitutions(IIndicator::mean|IIndicator::std_val|IIndicator::distrib);
		//Define distribution limits (only one for a boolean variable)
		std::vector<float>vlims
		vlims.push_back(0.5);
		Ind->setDistribLimits(vlims);
		//Add instants of indicator evaluations
		for(int i=0 ;i<1000 ;i++)
			addInstant(i) ;
		//Add the previously defined target
		addTarget("F.feeding",&DFSystem::target);

		//Choice of the random generator and its seed
		setRNG("yarn5");
		setRNGSeed(10);
	}
};

int main( int argc, char* argv[] )
{
	//Build the system
	DFSystem sys("");
	sys.loadParameters(path);//Load parameters
}
~~~~
exemple2 (excutable gnrique de lancement d'une simulation simple):
~~~~{.cpp}
int main( int argc, char* argv[] )
{
	bool bdcLoaded=false;
	char const*inFile=NULL,*swapFile=NULL;
	char const*resFile=NULL;
	CSystem*sys=new CSystem();
	try{
		int i;
		for(i=1;i<argc;i++)
			if(!strcmp(argv[i],"-bdc")){//Parameter of the C++ knowledge base library to load
				if(i+1==argc)
					break;
				i++;
				if(!CSystem::loadKBLibrary(argv[i]))
					ILogManager::glLogManager().throwError();
			}else if(!strcmp(argv[i],"-pbdc")){//Parameter of the Python knowledge base file to load
				if(i+1==argc)
					break;
				i++;
				if(!CSystem::loadKBPython(argv[i]))
					ILogManager::glLogManager().throwError();
			}else if(!strcmp(argv[i],"-rf")){//Parameter of the result file
				if(i+1==argc)
					break;
				resFile=argv[++i];
			}else if(!strcmp(argv[i],"-n")){//Parameter of the number of sequences
				if(i+1==argc)
					break;
				seqN=atoi(argv[++i]);
			}else if(!strcmp(argv[i],"-t")){//Parameter of the date of end of simulation
				if(i+1==argc)
					break;
				else
					tmax=atof(argv[++i]);
			}else if(argv[i][0]!='-'){//Parameter of a parameter file
				inFile=argv[i];
			}else
				ILogManager::glLogManager().throwError(formatMsg("Unknown argument %s",argv[i]).c_str());
		if(i+1==argc)
			ILogManager::glLogManager().throwError(formatMsg("Waiting for a parameter value after %s",argv[argc-1]).c_str());
		if(inFile){
			//We loed the parameter file which must include the definition of the system (components and connections)
			sys->load(inFile);
			if(resFile)//Set result file if defined
				sys->setResFileName(resFile,true);
			if(seqN)//Set sequence number if defined
				sys->setNbSeqToSim(seqN);
			if(tmax)//Set end date if defined
				sys->setTMax(tmax);
			//Launch simulation
			sys->simulate();
		}else
			ILogManager::glLogManager().msgError("No parameter file");
		//Verify waiting messages
		ILogManager::glLogManager().throwError();
	}
	catch( const std::exception& x )
	{
		if(!ILogManager::glLogManager().isLogManagerException(x)){//Display messages from ouside PyCATSHOO
			std::cerr<<x.what()<<std::endl;
			expt_thrown=true;
		}
	}
	catch(...)
	{
		std::cerr<<"Uknown error"<<std::endl;
		expt_thrown=true;
	}
	//Destroy system
	delete sys;
	return 0;
}
~~~~
\~english Class for managing a system to simulate.
This class must include all the high-level components of the system to be simulated.

This class can be used in two ways:
- As is: the system can be created either by the parameter file or through external methods.
- By subclassing `CSystem` and overriding the constructor to build the system components and the links between them.

The benefit of the second method is that it allows the addition of system methods to define indicators or targets.

The class provides monitoring and tracing tools.

The class allows the definition of targets through boolean functions. Targets serve as stopping conditions for sequences.

Finally, the class manages the simulation process (initialization, start, restart, result handling).

The class also manages the use of non-standard algorithms:
- Preferential sampling: `setUseImportanceSampling()`, the algorithm management tool is then obtained by `importanceSampling()`.
- Sequential Monte Carlo: `setUseSeqMonteCarlo()`, the algorithm management tool is then obtained by `seqMonteCarlo()`.

example1 (creating a system with a dedicated class):**
~~~~{.cpp}
// Including the definition of the Knowledge Base used
#include "BdCDF.h"

class DFSystem : public CSystem
{
	// Reference to the end component of the system
	CComponent* c;

	// Definition of an indicator function
	float indic() {
		return (float)(c->getVariable("feeding")->bValue());
	}

	// Definition of a target function
	bool target() {
		return !c->getVariable("feeding")->bValue() && c->getVariable("NFED")->bValue();
	}

	//   c1--cb1
	//  / |   |  \
	// cS  \ /   cF
	//  \   |    /
	//   --c2---
public:
	DFSystem() : CSystem("2files") {
		// Build the components using newComponent
		newComponent("cS", "DFSource");
		newComponent("c1", "DFComposant");

		// Direct build of components if classes are available
		new DFComposant("cb1", *this);
		c = new DFFin("cF", *this);
		new DFCompSecours("c2", *this);

		// Connect the components
		connect("cS", "aval", "c1", "amont");
		ICnctInfo& info = connect("c1", "aval", "cb1", "amont");
		// Add information to the connection
		info.setValue("NomInfo", 3.5);
		...

		// Modify initial value
		getComponent("c1")->getVariable("fed")->setValue(true);

		// Modify initial state
		getComponent("cb1")->setInitState("NOK");

		// Add the previously defined indicator for calculating mean, standard deviation, and distribution
		IIndicator* ind = addIndicator("F.feeding", &DFSystem::indic);
		ind->setRestitutions(IIndicator::mean | IIndicator::std_val | IIndicator::distrib);
		// Define distribution limits (only one for a boolean variable)
		std::vector<float> vlims;
		vlims.push_back(0.5);
		ind->setDistribLimits(vlims);

		// Add instants of indicator evaluations
		for (int i = 0; i < 1000; i++)
			addInstant(i);

		// Add the previously defined target
		addTarget("F.feeding", &DFSystem::target);

		// Choice of the random generator and its seed
		setRNG("yarn5");
		setRNGSeed(10);
	}
};

int main(int argc, char* argv[]) {
	// Build the system
	DFSystem sys("");
	sys.loadParameters(path); // Load parameters
}
~~~~
example2 (generic executable for launching a simple simulation):**
~~~~{.cpp}
int main(int argc, char* argv[]) {
	bool bdcLoaded = false;
	char const* inFile = NULL, * swapFile = NULL;
	char const* resFile = NULL;
	CSystem* sys = new CSystem();
	try {
		int i;
		for (i = 1; i < argc; i++)
			if (!strcmp(argv[i], "-bdc")) {  // Parameter of the C++ knowledge base library to load
				if (i + 1 == argc)
					break;
				i++;
				if (!CSystem::loadKBLibrary(argv[i]))
					ILogManager::glLogManager().throwError();
			} else if (!strcmp(argv[i], "-pbdc")) {  // Parameter of the Python knowledge base file to load
				if (i + 1 == argc)
					break;
				i++;
				if (!CSystem::loadKBPython(argv[i]))
					ILogManager::glLogManager().throwError();
			} else if (!strcmp(argv[i], "-rf")) {  // Parameter of the result file
				if (i + 1 == argc)
					break;
				resFile = argv[++i];
			} else if (!strcmp(argv[i], "-n")) {  // Parameter of the number of sequences
				if (i + 1 == argc)
					break;
				seqN = atoi(argv[++i]);
			} else if (!strcmp(argv[i], "-t")) {  // Parameter of the date of end of simulation
				if (i + 1 == argc)
					break;
				else
					tmax = atof(argv[++i]);
			} else if (argv[i][0] != '-') {  // Parameter of a parameter file
				inFile = argv[i];
			} else
				ILogManager::glLogManager().throwError(formatMsg("Unknown argument %s", argv[i]).c_str());

		if (i + 1 == argc)
			ILogManager::glLogManager().throwError(formatMsg("Waiting for a parameter value after %s", argv[argc - 1]).c_str());

		if (inFile) {
			// We load the parameter file which must include the definition of the system (components and connections)
			sys->load(inFile);
			if (resFile)  // Set result file if defined
				sys->setResFileName(resFile, true);
			if (seqN)  // Set sequence number if defined
				sys->setNbSeqToSim(seqN);
			if (tmax)  // Set end date if defined
				sys->setTMax(tmax);
			// Launch simulation
			sys->simulate();
		} else
			ILogManager::glLogManager().msgError("No parameter file");

		// Verify waiting messages
		ILogManager::glLogManager().throwError();
	}
	catch (const std::exception& x) {
		if (!ILogManager::glLogManager().isLogManagerException(x)) {  // Display messages from outside PyCATSHOO
			std::cerr << x.what() << std::endl;
			expt_thrown = true;
		}
	}
	catch (...) {
		std::cerr << "Unknown error" << std::endl;
		expt_thrown = true;
	}
	// Destroy system
	delete sys;
	return 0;
}
~~~~
*/
class PYC_PUBLIC CSystem : public IMonitor,public CSysNamedComp
{
	friend class CSystemP;
	friend class IExpression;
	friend class CComponent;
	friend struct SPythonWrapperFcts;
public :
private:
	CSystemP&m_SystemP;
	float m_PctSimu;
	TSimuMode::ESimuMode m_SimuMode;//Mode de simulation souhait (ne signifie pas que la simulation a commenc => CSystemP::m_SimuMode)
	int m_FMUOptions;
	static void*m_PythonWrapperLib;
	void doMonitor(IMonitored const&elt)override;
	void monitor(IMonitored&elt,bool bMonit)override;
	static bool loadPythonWrapperLib();
	/**\~french Excute un script Python si le systme a t construit en Python
	@param str script  excuter
	@param name nom de l'objet global  rcuprer
	@param ctxName nom donn  un objet dfini en Python (facultatif)
	@param ctx obj dont le nom en python doit tre ctxName
	@return L'objet de nom name si name est non Null
	\~english Executes a Python script if the system has been built in Python.
	@param str The script to execute.
	@param name The name of the global object to retrieve.
	@param ctxName The name given to an object defined in Python (optional).
	@param ctx The object whose name in Python should be `ctxName`.
	@return The object named `name` if `name` is not null.
	*/
	static PyObject*(*runPythonString)(char const*str, char const*name, char const*ctxName,CSysNamedComp*ctx);
	static CNamedComp* (*toCNamedCompW)(PyObject* obj);
	static std::string(*toString)(PyObject* obj);
protected:
	PyObject*m_Self;
	ITarget*addTarget(char const*name,bool (CSystem::*)(),PyObject*pyObj);
	IIndicator*addIndicator(char const*name,float (CSystem::*)(),PyObject*pyObj);
	void addSystemMethod(TSysMethod::ESysMethod type,CSysNamedComp*comp,char const*name,void(CSysNamedComp::*method)(),PyObject*pyMth);
	void deleteData();
	void simulate(bool assynchrone);
public:
	/**\~french Construit un systme de nom name.\~english Builds a system named `name`.*/
	explicit CSystem(char const*name);
	CSystem(PyObject*pyObj,char const*name);
	virtual~CSystem(void);
	static void setNoMPI();//!<\~french Demande  ne pas utiliser MPI (doit tre appele avant la construction du systme)\~english Requests not to use MPI (must be called before constructing the system).
	static void terminate();//!<\~french Termine la session en librant compltement la mmoire\~english Ends the session by fully freeing the memory.
	static std::vector<std::string>loadedFMUs();//!<\~french Retourne la liste des FMU chargs\~english Return the liste of loaded FMUs.
	PyObject*self(){return m_Self;}
	void deleteSys();//!<\~french Dtruit le systme sauf si m_Self est non nul (alors, ne dtruit que l'intrieur)\~english Destroys the system unless `m_Self` is not null (in which case, only the interior is destroyed).
	/**\~french Retourne le systme en cours de simulation\~english Returns the system currently being simulated.*/
	static CSystem*glSystem();
	CSystemP&systemP(){return m_SystemP;}//!<\~french Retourne le stockage priv (uniquement pour utilisation interne)\~english Returns the private storage (for internal use only).
	CSystemP const&systemP()const{return m_SystemP;}//!<\~french Retourne le stockage priv (uniquement pour utilisation interne)\~english Returns the private storage (for internal use only).
	/**\~french Enregistre une BdC pour les systmes  construire\~english Saves a Knowledge Base for the systems to be constructed.*/
	static void registerBdC(IBdC&BdC);
	/**\~french Desenregistre une BdC pour les systmes  construire\~english Unregisters a Knowledge Base for the systems to be constructed.*/
	static void unregisterBdC(char const*name);
	/**\~french Retourne la liste des noms des BdC\~english Returns the list of Knowledge Base names.*/
	static std::vector<std::string>knowledgeBases();
	DEPRECATED static std::vector<std::string>BdCs(){return knowledgeBases();}
	/**\~french Demande le stockage des mthodes avec ou sans tenir compte du nom du composant\~english Requests the storage of methods with or without considering the component name.*/
	static void setMethodsUseCompName(bool use=true);
	/**\~french Retourne la BdC de nom name ou NULL\~english Returns the Knowledge Base named `name` or NULL.*/
	static IBdC const*knowledgeBase(char const*name);
	DEPRECATED static IBdC const*BdC(char const*name){return knowledgeBase(name);}
	/**\~french Evalue une valeur exprime en python dans le contexte de ctx\~english Evaluates a value expressed in Python within the context `ctx`.*/
	std::string evalPythonScript(char const* script, CSysNamedComp* ctx);
	bool verify()override;//!<\~french Vrifie le systme avant le lancement d'une simulation\~english Verifies the system before launching a simulation.
	virtual bool verifyFor(int mode);//!<\~french Vrifie le systme en slectionnant les lments  vrifier\~english Verifies the system by selecting the elements to check.
	bool verifyForProcessing(int proc);//!<\~french Vrifie le systme pour les traitements envisags\~english Verifies the system for the intended processes.
	std::vector<IVariable*>buildFMUMap()const;//!<\~french Construit la map des rfrences vers les variables\~english Builds the map of references to variables.
	static bool dumpFMUXML(char const* FMULib, char const* resourceLocation, char const* filePath, int version = 2);//!<\~french Construit un fichier de description du modle gr par FMULib associe  resourceLocation\~english Builds a description file of the model managed by FMULib associated with `resourceLocation`.
	static bool buildFMU(char const* FMULib, char const* resourceLocation, char const*modelDescription, char const* filePath, int version = 2);//!<\~french Construit un fichier FMU contenant FMULib associe  resourceLocation (le fichier modelDescription est construit s'il est NULL\~english Builds a FMU file including FMULib associated with `resourceLocation`; the modelDescription file is build if NULL.
	bool dumpFMUXML(char const*filePath, int version=2)const;//!<\~french Construit un fichier XML de description du modle FMU\~english Builds an XML description file of the FMU model.
	std::string guid()const;//!<\~french Gnre un GUID  partir du contenu du systme\~english Generates a GUID from the content of the system.
	/**\~french Construit un composant de nom name et de classe clName dans le systme.
	@param name nom du composant
	@param clName nom de la classe du composant
	@param bdcName nom de la BdC (facultatif)
	\~english Builds a component named `name` and of class `clName` in the system.
	@param name The name of the component.
	@param clName The name of the component's class.
	@param bdcName The name of the Knowledge Base (optional).
	*/
	CComponent* newComponent(char const* name, char const* clName, char const* bdcName = NULL);
	/**\~french Retourne vrai si le composant vient d'tre construit en cours de simulation\~english Returns true if the component has just been built during the simulation.*/
	bool isNewComponent(CNamedComp const* comp);
	DEPRECATED std::vector<CComponent*>getComponents(char const*name=NULL, char const*className=NULL)const;//!<\~french Retourne tous les composants de nom name (# en tte si regex, tous rcursivement si NULL) et de classe className (# en tte si regex) (obsolte)\~english Returns all components named `name` (# at the beginning if regex, all recursively if NULL) and of class `className` (# at the beginning if regex) (deprecated).
	std::vector<CComponent*>components(char const*name=NULL, char const*className=NULL)const;//!<\~french Retourne tous les composants de nom name (# en tte si regex, tous rcursivement si NULL) et de classe className (# en tte si regex)\~english Returns all components named `name` (# at the beginning if regex, all recursively if NULL) and of class `className` (# at the beginning if regex).
	std::vector<IFunctionalMockupUnit*>FMUs(char const*name=NULL, char const*modelName=NULL)const;//!<\~french Retourne tous les FMUs de nom name (# en tte si regex, tous rcursivement si NULL) et de nom de modle modelName (# en tte si regex)\~english Returns all FMUs named `name` (# at the beginning if regex, all recursively if NULL) and with the model name `modelName` (# at the beginning if regex).
	DEPRECATED std::vector<IPDMPManager*>getPDMPManagers(char const*name=NULL)const;//!<\~french Retourne tous les PDMPManagers correspondant  l'expression rgulire name (tous si NULL) (obsolte)\~english Returns all `PDMPManagers` matching the regular expression `name` (all if NULL) (deprecated).
	std::vector<IPDMPManager*>PDMPManagers(char const*name=NULL)const;//!<\~french Retourne tous les PDMPManagers correspondant  l'expression rgulire name (tous si NULL)\~english Returns all `PDMPManagers` matching the regular expression `name` (all if NULL).
	DEPRECATED std::vector<ISLEManager*>getSLEManagers(char const*name=NULL)const;//!<\~french Retourne tous les SLEManagers correspondant  l'expression rgulire name (tous si NULL) (obsolte)\~english Returns all `SLEManagers` matching the regular expression `name` (all if NULL) (deprecated).
	std::vector<ISLEManager*>SLEManagers(char const*name=NULL)const;//!<\~french Retourne tous les SLEManagers correspondant  l'expression rgulire name (tous si NULL)\~english Returns all `SLEManagers` matching the regular expression `name` (all if NULL).
	DEPRECATED std::vector<IMILPManager*>getMILPManagers(char const*name=NULL)const;//!<\~french Retourne tous les MILPManagers correspondant  l'expression rgulire name (tous si NULL) (obsolte)\~english Returns all `MILPManagers` matching the regular expression `name` (all if NULL) (deprecated).
	std::vector<IMILPManager*>MILPManagers(char const* name = NULL)const;//!<\~french Retourne tous les MILPManagers correspondant  l'expression rgulire name (tous si NULL)\~english Returns all `MILPManagers` matching the regular expression `name` (all if NULL).
	DEPRECATED std::vector<ITransition*>getTransitions(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne toutes les transitions de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex) (obsolte)\~english Returns all transitions named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex) (deprecated).
	std::vector<ITransition*>transitions(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne toutes les transitions de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex)\~english Returns all transitions named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex).
	DEPRECATED std::vector<IVariable*>getVariables(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne toutes les variables de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex) (obsolte)\~english Returns all variables named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex) (deprecated).
	std::vector<IVariable*>variables(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne toutes les variables de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex)\~english Returns all variables named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex).
	std::vector<IVariable*>variablesFMU(char const*name=NULL,char const*modelName=NULL)const;//!<\~french Retourne toutes les variables de nom name (# en tte si regex) et dont le modle FMU est de nom modelName (# en tte si regex)\~english Returns all variables named `name` (# at the beginning if regex) and whose FMU model is named `modelName` (# at the beginning if regex).
	DEPRECATED std::vector<IReference*>getReferences(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne toutes les rfrences de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex) (obsolte)\~english Returns all references named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex) (deprecated).
	std::vector<IReference*>references(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne toutes les rfrences de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex)\~english Returns all references named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex).
	DEPRECATED std::vector<IState*>getStates(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne tous les tats de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex) (obsolte)\~english Returns all states named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex) (deprecated).
	std::vector<IState*>states(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne tous les tats de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex)\~english Returns all states named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex).
	DEPRECATED std::vector<IAutomaton*>getAutomatons(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne tous les automates de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex) (obsolte)\~english Returns all automatons named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex) (deprecated).
	std::vector<IAutomaton*>automatons(char const*name=NULL,char const*className=NULL)const;//!<\~french Retourne tous les automates de nom name (# en tte si regex) et dont le composant est de classe className (# en tte si regex)\~english Returns all automatons named `name` (# at the beginning if regex) and whose component is of class `className` (# at the beginning if regex).
	DEPRECATED std::vector<IIndicator*>getIndicators(char const*name=NULL)const;//!<\~french Retourne tous les indicateurs de nom name correspondant  l'expression rgulire name (obsolte)\~english Returns all indicators named `name` matching the regular expression `name` (deprecated).
	std::vector<IIndicator*>indicators(char const*name=NULL)const;//!<\~french Retourne tous les indicateurs de nom name correspondant  l'expression rgulire name\~english Returns all indicators named `name` matching the regular expression `name`.
	DEPRECATED std::vector<ITarget*>getTargets(char const*name)const;//!<\~french Retourne toutes les cibles de nom correspondant  l'expression rgulire name (obsolte)\~english Returns all targets named `name` matching the regular expression `name` (deprecated).
	std::vector<ITarget*>targets(char const*name)const;//!<\~french Retourne toutes les cibles de nom correspondant  l'expression rgulire name\~english Returns all targets named `name` matching the regular expression `name`.
	DEPRECATED std::vector<IStep*>getSteps(char const*name=NULL)const;//!<\~french Retourne toutes les tapes de nom correspondant  l'expression rgulire name (obsolte)\~english Returns all steps named `name` matching the regular expression `name` (deprecated).
	std::vector<IStep*>steps(char const*name=NULL)const;//!<\~french Retourne toutes les tapes de nom correspondant  l'expression rgulire name\~english Returns all steps named `name` matching the regular expression `name`.
	DEPRECATED std::vector<IMonitored const*>getMonitoredElts(char const*name=NULL)const;//!<\~french Retourne tous les lments monitors de nom correspondant  l'expression rgulire name (obsolte)\~english Returns all monitored elements named `name` matching the regular expression `name` (deprecated).
	std::vector<IMonitored const*>monitoredElts(char const*name=NULL)const;//!<\~french Retourne tous les lments monitors de nom correspondant  l'expression rgulire name\~english Returns all monitored elements named `name` matching the regular expression `name`.
	DEPRECATED std::vector<ITransition*>getPlannedTransitions()const;//!<\~french Retourne toutes les transitions actives (obsolte)\~english Returns all active transitions (deprecated).
	std::vector<ITransition*>plannedTransitions()const;//!<\~french Retourne toutes les transitions actives\~english Returns all active transitions.
	DEPRECATED std::vector<ITransition*>getActiveTransitions()const;//!<\~french Retourne toutes les transitions actives (obsolte)\~english Returns all active transitions (deprecated).
	std::vector<ITransition*>activeTransitions()const;//!<\~french Retourne toutes les transitions actives\~english Returns all active transitions.
	DEPRECATED std::vector<ITransition*>getActiveTransitionsDyn()const;//!<\~french Retourne toutes les transitions dont l'tat de dpart est actif et la condition actuelle est vrai (obsolte)\~english Returns all transitions whose starting state is active and the current condition is true (deprecated).
	std::vector<ITransition*>activeTransitionsDyn()const;//!<\~french Retourne toutes les transitions dont l'tat de dpart est actif et la condition actuelle est vrai\~english Returns all transitions whose starting state is active and the current condition is true.
	DEPRECATED std::vector<ITransition*>getFiredTransitions()const;//!<\~french Retourne les transitions tires au pas prcdent (obsolte)\~english Returns the transitions drawn at the previous step (deprecated).
	std::vector<ITransition*>firedTransitions()const;//!<\~french Retourne les transitions tires au pas prcdent\~english Returns the transitions drawn at the previous step.
	DEPRECATED IMessageBox*getMessageBox(char const*name)const;//!<\~french Retourne la boite de message de nom complet name (obsolte)\~english Returns the message box with the full name `name` (deprecated).
	IMessageBox*messageBox(char const*name)const;//!<\~french Retourne la boite de message de nom complet name\~english Returns the message box with the full name `name`.
	ISystemState const*systemState()const;//!<\~french Retourne un stockage de l'tat du systme\~english Returns a storage of the system's state.
	void setFMUOptions(int options){m_FMUOptions = options;}//!<\~french Modifie le mode de stockage de l'tat du systme\~english Modifies the storage mode of the system's state.
	int FMUOptions()const{return m_FMUOptions;}
	/**\~french Ajoute une mthode systme de type type et de nom name\~english Adds a system method of type `type` and name `name`.*/
	template<typename Type>void addSystemMethod(TSysMethod::ESysMethod type,Type&nc,char const*name,void(Type::*method)()){checkMethodAddrSize(sizeof(method));addSystemMethod(type,&nc,name,static_cast<void(CSysNamedComp::*)()>(method),NULL);}
	/**\~french Supprime une mthode systme de type type et de nom name\~english Removes a system method of type `type` and name `name`.*/
	void removeSystemMethod(TSysMethod::ESysMethod type,CSysNamedComp&comp,char const*name);
	/**\~french Ajoute un indicateur et dfinit sa fonction de calcul.
	@param name nom de l'indicateur
	@param fct fonction indicatrice
	\~english Adds an indicator and defines its calculation function.
	@param name The name of the indicator.
	@param fct The indicator function.
	*/
	template<typename TYPE>IIndicator*addIndicator(char const*name,float (TYPE::*fct)()){
		checkMethodAddrSize(sizeof(fct));
		return addIndicator(name,static_cast<float (CSystem::*)()>(fct),NULL);
	}
	/**\~french Ajoute un indicateur dfini par une variable et ventuellement une comparaison avec une valeur.
	@param name nom de l'indicateur
	@param elementName nom de la variable
	@param elementType type de l'lment (ST VAR AUT)
	@param op oprateur de comparaison
	@param value valeur de comparaison
	\~english Adds an indicator defined by a variable and optionally a comparison with a value.  
	@param name The name of the indicator.  
	@param elementName The name of the variable.  
	@param elementType The type of the element (ST, VAR, AUT).  
	@param op The comparison operator.  
	@param value The comparison value.
	*/
	IIndicator*addIndicator(char const*name, char const * elementName, char const * elementType, char const * op=NULL, float value=0);
	IIndicator*addIndicator(char const*name,IVariable&var, char const * op=NULL, float value=0);
	/**\~french Retourne l'indicateur de nom name.\~english Returns the indicator named `name`.*/
	IIndicator*indicator(char const*name)const;
	/**\~french Supprime l'indicateur de nom name.\~english Removes the indicator named `name`.*/
	void removeIndicator(char const*name);
	/**\~french Ajoute un instant de mmorisation des indicateurs.\~english Adds a moment for storing the indicators.*/
	void addInstant(double date);
	/**\~french Ajoute des instants de mmorisation des indicateurs dfinis par un intervalle dcoup.\~english Adds moments for storing the indicators defined by a split interval.*/
	void addInstants(double minDate, double maxDate, int nTotalPoints);
	/**\~french Dfinit les instants de mmorisation des indicateurs.\~english Defines the moments for storing the indicators.*/
	void setInstants(std::vector<PyTime>insts);
	/**\~french Retourne les instants de mmorisation des indicateurs.\~english Returns the moments for storing the indicators.*/
	std::vector<double>instants()const;
	virtual size_t nbBrokenComp();//!<\~french Retourne le nombre composants casss parmi les KOStates ( surcharger pour utiliser USimple en tirage prfrentiel)\~english Returns the number of broken components among the `KOStates` (to be overridden for use with `USimple` in preferential sampling).
	virtual size_t nbBrokenComp(ISystemState const*s)const;//!<\~french Retourne le nombre composants casss parmi les KOStates ( surcharger pour utiliser G en SMC)\~english Returns the number of broken components among the `KOStates` (to be overridden for use with `G` in Sequential Monte Carlo).
	/**\~french Ajoute une cible et dfinit sa fonction condition.
	@param name nom de la cible
	@param fct fonction condition (retourne vrai si la cible est atteinte)
	\~english Adds a target and defines its condition function.  
	@param name The name of the target.  
	@param fct The condition function (returns true if the target is reached).
	*/
	template<typename TYPE>ITarget*addTarget(char const*name,bool(TYPE::*fct)()){
		checkMethodAddrSize(sizeof(fct));
		return addTarget(name,static_cast<bool(CSystem::*)()>(fct),NULL);
	}
	/**\~french Ajoute une cible dfinie par la comparaison d'une variable avec une valeur\~english Adds a target defined by comparing a variable with a value.*/
	ITarget*addTarget(char const*name,IVariable&var,char const*op="!=",float value=0);
	/**\~french Ajoute une cible dfinie par la comparaison d'une variable avec une valeur
	@param name nom de la cible
	@param elementName nom de la variable
	@param elementType type de la variable parmi VAR, ST, AUT ou NULL
	@param op oprateur de comparaison
	@param value valeur de comparaison
	\~english Adds a target defined by comparing a variable with a value.  
	@param name The name of the target.  
	@param elementName The name of the variable.  
	@param elementType The type of the variable, one of VAR, ST, AUT, or NULL.  
	@param op The comparison operator.  
	@param value The comparison value.
	*/
	ITarget*addTarget (char const * name, char const * elementName, char const * elementType, char const*op="!=", float value=0);
	/**\~french Suprime la cible de nom name.\~english Removes the target named `name`.*/
	void removeTarget(char const*name);
	/**\~french Retourne la cible de nom name.\~english Returns the target named `name`.*/
	ITarget*target(char const*name);
	/**\~french Retourne l'option de bouclage global dans les tapes\~english Returns the global looping option in the steps.*/
	bool stepLoop()const;
	/**\~french Modifie l'option de bouclage global dans les tapes (par dfaut faux)\~english Modifies the global looping option in the steps (default is false).*/
	void setStepLoop(bool stepLoop);
	/**\~french Ajoute une tape de nom name.
	La mthode retourne l'tape ajoute ou retrouve dans la liste des tapes.
	\~english Adds a step named `name`.  
	The method returns the added or found step in the list of steps.*/
	IStep*addStep(char const*name);
	/**\~french Insert une tape de nom name en position ind.
	La mthode retourne l'tape ajoute ou retrouve dans la liste des tapes.
	\~english Inserts a step named `name` at position `ind`.  
	The method returns the added or found step in the list of steps.*/
	IStep*insertStep(char const*name,unsigned int index);
	/**\~french Retourne l'tape de nom name\~english Returns the step named `name`.*/
	IStep*step(char const*name)const;
	/**\~french Modifie la condition de l'tape de nom name\~english Modifies the condition of the step named `name`.*/
	void setStepCondition(char const*name,bool val);
	/**\~french Modifie la condition de l'tape de nom name\~english Modifies the condition of the step named `name`.*/
	void setStepCondition(char const*name,IVariable&var,bool negate=false);
	/**\~french Modifie la condition de l'tape de nom name\~english Modifies the condition of the step named name*/
	template<typename Type> void setStepCondition(char const*name,Type&nc,bool(Type::*fct)(),bool negate=false){step(name)->setCondition(nc, static_cast<bool(CNamedComp::*)()>(fct),negate);}
	/**\~french Supprime l'tape de nom name\~english Deletes the step named name*/
	void removeStep(char const*name);
	/**\~french Ajoute une mthode  l'tape de nom name (si comp=NULL, toutes les mthodes de nom methodName font partie de l'tape)\~english Adds a method to the step named name (if comp=NULL, all methods named methodName are part of the step)*/
	void addStepMethod(char const*name,CComponent const*comp, char const*methodName);
	/**\~french Abandonne la construction de la squence\~english Abandons the construction of the sequence*/
	void abortSeq();
	/**\~french Abandonne la simulation\~english Abandons the simulation*/
	void abortSim();
	/**\~french Retourne vrai si le systme est en cours d'abandon de la squence\~english Returns true if the system is in the process of abandoning the sequence*/
	bool isAborting()const;
	/**\~french Retourne vrai si le systme est en cours d'abandon de la simulation\~english Returns true if the system is in the process of abandoning the simulation*/
	bool isAbortingSim()const;
	/**\~french Retourne la variable de nom name d'un type donn parmi (NULL, "VAR", "ST", "AUT") (obsolte)\~english Returns the variable named name of a given type among (NULL, "VAR", "ST", "AUT") (deprecated)*/
	DEPRECATED IVariable*getElement(char const*name,char const*type)const;
	/**\~french Retourne la variable de nom name d'un type donn parmi (NULL, "VAR", "ST", "AUT", "FMUV")\~english Returns the variable named name of a given type among (NULL, "VAR", "ST", "AUT", "FMUV")*/
	IVariable*element(char const*name,char const*type)const;
	/**\~french Retourne les variables de nom name d'un type donn parmi ("VAR", "ST", "AUT", "FMUV")\~english Returns the variables named name of a given type among ("VAR", "ST", "AUT", "FMUV")*/
	std::vector<IVariable*>elements(char const*name,char const*type,char const*className)const;
	/**\~french Retourne un lment nomm dfini par une expression d'accs de la forme 'type:nom' ou 'type:#nom'
	
	Si on utilise nom, la variable est recherche dans le composant courant
	Le nom doit dsigner un lment unique et ne peut pas utiliser d'expressions rgulires
	\~english Returns a named element defined by an access expression of the form 'type:name' or 'type:#name'

	If name is used, the variable is searched in the current component.  
	The name must refer to a unique element and cannot use regular expressions.*/
	CNamedComp*access(char const*accessExpr,CComponent*current,bool compOk=true)const;
	/**\~french Retourne la refrence de nom name (obsolte)\~english Returns the reference named name (deprecated)*/
	DEPRECATED IReference*getReference(char const*name)const;
	/**\~french Retourne la refrence de nom name\~english Returns the reference named name*/
	IReference*reference(char const*name)const;
	/**\~french Retourne le composant de nom name (nom ventuellement compos) (obsolte)
	@param name nom complet du composant recherch
	@param check indicateur de contrle d'erreur : si vrai, une exception est lance si le composant n'existe pas
	\~english Returns the component named name (possibly a composite name) (deprecated)  
	@param name full name of the component being searched  
	@param check error control flag: if true, an exception is thrown if the component does not exist
	*/
	DEPRECATED CComponent*getComponent(char const*name,bool check=true)const;
	/**\~french Retourne le nombre de composants de 1er niveau\~english Returns the number of top-level components*/
	size_t componentCount()const;
	/**\~french Retourne le composant de nom name (nom ventuellement compos)
	@param name nom complet du composant recherch
	@param check indicateur de contrle d'erreur : si vrai, une exception est lance si le composant n'existe pas
	\~english Returns the component named name (possibly a composite name)  
	@param name full name of the component being searched  
	@param check error control flag: if true, an exception is thrown if the component does not exist
	*/
	CComponent*component(char const*name,bool check=true)const;
	/**\~french Retourne le gestionnaire d'ODE de nom name (obsolte)\~english Returns the ODE manager named name (deprecated)*/
	DEPRECATED IPDMPManager*getPDMPManager(char const*name)const;
	/**\~french Retourne le gestionnaire d'ODE de nom name\~english Returns the ODE manager named name*/
	IPDMPManager*PDMPManager(char const*name)const;
	/**\~french Retourne le gestionnaire d'ODE courant\~english Returns the current ODE manager*/
	IPDMPManager*currentPDMPManager()const;
	/**\~french Dsigne le gestionnaire d'ODE courant (peut tre null si pas d'ODE)\~english Sets the current ODE manager (can be null if no ODE)*/
	IPDMPManager*setCurrentPDMPManager(char const*name);
	void setCurrentPDMPManager(IPDMPManager*pdmp);
	/**\~french Ajoute un nouveau gestionnaire d'ODE (s'il existe dj, retourne celui l)\~english Adds a new ODE manager (if it already exists, returns that one)*/
	IPDMPManager*addPDMPManager(char const*name);
	/**\~french Retourne le systme d'quations linaires de nom name (obsolte)\~english Returns the linear equation system named name (deprecated)*/
	DEPRECATED ISLEManager*getSLEManager(char const*name)const;
	/**\~french Retourne le systme d'quations linaires de nom name\~english Returns the linear equation system named name*/
	ISLEManager*SLEManager(char const*name)const;
	/**\~french Ajoute un nouveau systme d'quations linaires (s'il existe dj, retourne celui l)\~english Adds a new linear equation system (if it already exists, returns that one)*/
	ISLEManager*addSLEManager(char const*name);
	//*\~french Supprime lle systme d'quations linaries de nom name\~english Remove the linear equation system named name*/
	void removeSLEManager(char const* name);
	/**\~french Retourne l'optimiseur linaire sous contraintes de nom name (obsolte)\~english Returns the constrained linear optimizer named name (deprecated)*/
	DEPRECATED IMILPManager*getMILPManager(char const*name)const;
	/**\~french Retourne l'optimiseur linaire sous contraintes de nom name\~english Returns the constrained linear optimizer named name*/
	IMILPManager* MILPManager(char const* name)const;
	/**\~french Ajoute un nouvel optimiseur linaire sous contraintes (s'il existe dj, retourne celui l)\~english Adds a new constrained linear optimizer (if it already exists, returns that one)*/
	IMILPManager* addMILPManager(char const* name);
	//*\~french Supprime l'optimiseur sous contraintes de nom name\~english Remove the constrained linear optimizer named name*/
	void removeMILPManager(char const* name);
	/**\~french Modifie le nom du fichier de rsultats.\~english Modifies the name of the results file.*/
	void setResultFileName(char const*filename,bool bBin);
	/**\~french Retourne le nom du fichier de rsultats.\~english Returns the name of the results file.*/
	char const*resFileName()const;
	/**\~french Modifie le caractre binaire du stockage des squences\~english Modifies the binary character of sequence storage*/
	void setBinSeqFile(bool bBin);
	/**\~french Retourne vrai si le stockage des squences est binaire\~english Returns true if sequence storage is binary*/
	bool binSeqFile()const;
	/**\~french Retourne l'instant courant.\~english Returns the current instant.*/
	PyTime currentTime()const;
	/**\~french Retourne l'indice de branche courant.\~english Returns the current branch index.*/
	unsigned currentIndBranch()const;
	/**\~french Lance les simulations des squences\~english Runs sequence simulations*/
	void simulate(){simulate(true);}
	/**\~french Lance les simulations des squences sans MPI\~english Runs sequence simulations without MPI*/
	void simulateLocal(){simulate(false);}
	//*\~french Rinitialise la simulation (cas d'une exception en cours de simulation)\~english Reset the simulation (in case of exception while simulating)*/
	void reset();
	/**\~french Rejoue la squence et s'arrte en fin de squence.\~english Replays the sequence and stops at the end of the sequence.*/
	void replaySequence(ISequence const*seq);
	/**\~french Dmarre une session interactive\~english Starts an interactive session*/
	void startInteractive();
	/**\~french Arrte une session interactive\~english Stops an interactive session*/
	void stopInteractive();
	/**\~french Avance jusqu' la date date en interactif FMU (si date<0, arrt avant la premire transition)\~english Advances to the date date in FMU interactive mode (if date<0, stops before the first transition)*/
	void stepInteractive(PyTime date);
	/**\~french Modifie la date de tirage et le numro de sortie de la transition\~english Modifies the firing date and the output number of the transition*/
	void setTransPlanning(ITransition*trans,PyTime date,unsigned out);
	/**\~french Avance la squence courante jusqu' la prochaine transition ou  la prochaine interruption d'intgration\~english Advances the current sequence to the next transition or the next integration interruption*/
	void stepForward();
	/**\~french Recule  l'tat prcdent\~english Reverts to the previous state*/
	void stepBackward();
	/**\~french Recule  l'tat initial\~english Reverts to the initial state*/
	void restart();
	/**\~french Dfinit un remplacement dans les expressions d'accs\~english Defines a replacement in access expressions*/
	void setAccessReplace(char const*kw,char const*text);
	/**\~french Ajoute une expression  valuer en simulation interactive (l'expression expr est value dans le contexte du composant cName)\~english Adds an expression to evaluate in interactive simulation (the expression expr is evaluated in the context of the component cName)*/
	int addSimulationExpr(char const*cName,char const*expr);
	/**\~french Retourne la valeur de l'expression de simulation d'indice ind\~english Returns the value of the simulation expression at index ind*/
	double evalSimulationExpr(int ind)const;
	/**\~french Retourne l'expression de simulation instancie sous forme textuelle\~english Returns the instantiated simulation expression in textual form*/
	std::string simulationExprStr(int ind)const;
	/**\~french Retourne la squence courante\~english Returns the current sequence*/
	ISequence const*currentSequence()const;
	/**\~french Met  jour le planning en mode interactif aprs avoir fix les dlais et les indices de sortie\~english Updates the schedule in interactive mode after setting the delays and output indices*/
	void updatePlanningInt();
	/**\~french Met  jour le planning pour tenir compte d'une modification des paramtres des lois\~english Updates the schedule to account for a modification of the law parameters*/
	void updatePlanningForModifiers();
	/**\~french Connecte deux botes de messages de composants. Retourne l'information associe  la nouvelle connexion.
	@param c1 composant de dpart
	@param msgB1 bote de messages de c1
	@param c2 composant d'arrive
	@param msgB2 bote de messages de c2
	@param weight poids associ  la connexion (par dfaut 0)
	\~english Connects two component message boxes. Returns the information associated with the new connection.
	@param c1 starting component
	@param msgB1 message box of c1
	@param c2 destination component
	@param msgB2 message box of c2
	@param weight weight associated with the connection (default is 0)
	*/
	ICnctInfo&connect(CComponent&c1,char const*msgB1,CComponent&c2,char const*msgB2,double weight=0);
	/**\~french Connecte deux botes de messages de composants. Retourne l'information associe  la nouvelle connexion.
	@param c1Name nom du composant de dpart
	@param msgB1 bote de messages de c1
	@param c2Name nom du composant d'arrive
	@param msgB2 bote de messages de c2
	@param weight poids associ  la connexion (par dfaut 0)
	\~english Connects two component message boxes. Returns the information associated with the new connection.  
	@param c1Name name of the starting component  
	@param msgB1 message box of c1  
	@param c2Name name of the destination component  
	@param msgB2 message box of c2  
	@param weight weight associated with the connection (default is 0)
	*/
	ICnctInfo&connect(char const*c1Name,char const*msgB1,char const*c2Name,char const*msgB2,double weight=0);
	/**\~french Dconnecte deux botes de messages de composants.
	@param c1 composant de dpart
	@param msgB1 bote de messages de c1
	@param c2 composant d'arrive
	@param msgB2 bote de messages de c2
	\~english Disconnects two component message boxes.  
	@param c1 starting component  
	@param msgB1 message box of c1  
	@param c2 destination component  
	@param msgB2 message box of c2
	*/
	void disconnect(CComponent&c1,char const*msgB1,CComponent&c2,char const*msgB2);
	/**\~french Dconnecte deux botes de messages de composants.
	@param c1Name nom du composant de dpart
	@param msgB1 bote de messages de c1
	@param c2Name nom du composant d'arrive
	@param msgB2 bote de messages de c2
	\~english Disconnects two component message boxes.  
	@param c1Name name of the starting component  
	@param msgB1 message box of c1  
	@param c2Name name of the destination component  
	@param msgB2 message box of c2
	*/
	void disconnect(char const*c1Name,char const*msgB1,char const*c2Name,char const*msgB2);
	/**\~french Retourne vrai si la simulation d'une squence est en cours\~english Returns true if the simulation of a sequence is in progress*/
	bool isRunning()const;
	TSimuMode::ESimuMode simuMode()const;
	/**\~french Retourne le pourcentage de la simulation dj ralis\~english Returns the percentage of the simulation that has already been completed*/
	float pctSimulation()const{return m_PctSimu;}
	/**\~french Retourne vrai si la rsolution d'un systme d'ODE est en cours\~english Returns true if the resolution of an ODE system is in progress*/
	bool isODESolving()const;
	/**\~french Retourne vrai si le systme est reconstruit uniquement  partir du fichier de rsultats\~english Returns true if the system is reconstructed solely from the results file*/
	bool isAutoBuilt()const;
	/**\~french Supprime les rsultats courants.\~english Deletes the current results.*/
	void clearRes();
	/**\~french Modifie le nombre de squences  tirer\~english Modifies the number of sequences to be drawn.*/
	void setNbSeqToSim(long long nbSeq);
	/**\~french Retourne le nombre de squences  tirer.\~english Returns the number of sequences to be drawn.*/
	SEQ_ID nbSeqToSim()const;
	/**\~french Modifie le pourcentage de rduction affect au nombre de squences gres par le processus matre.\~english Modifies the percentage reduction applied to the number of sequences handled by the master process.*/
	void setPctMinNbSeqMaster(double pct);
	/**\~french Retourne le pourcentage de rduction affect au nombre de squences gres par le processus matre.\~english Returns the percentage reduction applied to the number of sequences handled by the master process.*/
	double pctMinNbSeqMaster()const;
	/**\~french Retourne le nombre de squences tires en fin de simulation.\~english Returns the number of sequences drawn at the end of the simulation.*/
	SEQ_ID nbSeqSim()const;
	/**\~french Retourne le nombre de squences tires depuis le dbut de la simulation.\~english Returns the number of sequences drawn since the beginning of the simulation.*/
	SEQ_ID nbSeqSimCurrent()const;
	/**\~french Retourne le nombre de squences conserves.\~english Returns the number of sequences retained.*/
	SEQ_ID nbSequences()const;
	/**\~french Retourne la liste des squences conserves\~english Returns the list of retained sequences.*/
	std::vector<ISequence*>sequences()const;
	/**\~french Retourne la numSeqme squence parmi les nbSequences conserves\~english Returns the numSeq-th sequence among the nbSequences retained.*/
	ISequence*sequence(SEQ_ID numSeq)const;
	/**\~french Retourne la squence suivante ou la premire si seq est NULL\~english Returns the next sequence or the first one if seq is NULL.*/
	ISequence*nextSequence(ISequence const*seq)const;
	/**\~french Retourne le filtre de squence s'il existe\~english Returns the sequence filter if it exists.*/
	IFilterFct const*seqFilter()const;
	/**\~french Modifie le filtre de squences (dtruit le prcdent)\~english Modifies the sequence filter (destroys the previous one).*/
	void setSeqFilter(IFilterFct*seqFilter);
	void chgSeqFilter(IFilterFct*seqFilter);//!<\~french Modifie le filtre sans dtruire le prcdent\~english Modifies the filter without destroying the previous one.
	/**\~french Renseigne la fonction de filtrage des squences sous la forme d'une fonction boolenne d'une squence (la fonction retourne vrai si la squence doit tre conserve).\~english Sets the sequence filtering function in the form of a boolean function for a sequence (the function returns true if the sequence should be retained).*/
	void setSeqFilter(bool (*seqFilter)(ISequence const&));
	/**\~french Renseigne la fonction de filtrage des squences sous la forme d'un pattern de dfinition d'un filtre\~english Sets the sequence filtering function in the form of a pattern for defining a filter.*/
	void setSeqFilter(char const*filterPattern);
	/**\~french Retourne l'lment monitor de nom name et de type donn parmi (NULL, "VAR", "ST", "AUT", "TR", "IND") (obsolte)\~english Returns the monitored element named name and of a given type among (NULL, "VAR", "ST", "AUT", "TR", "IND") (deprecated)*/
	DEPRECATED IMonitored const*getMonitoredElt(char const*name,char const*type=NULL)const;
	/**\~french Retourne l'lment monitor de nom name et de type donn parmi (NULL, "VAR", "ST", "AUT", "TR", "IND")\~english Returns the monitored element named name and of a given type among (NULL, "VAR", "ST", "AUT", "TR", "IND")*/
	IMonitored const*monitoredElt(char const*name,char const*type=NULL)const;
	/**\~french Demande l'inhibition des transitions de nom name (le nom peut tre compos ou une expression rgulire prcde par #).\~english Requests the inhibition of transitions named name (the name can be composite or a regular expression prefixed by #).*/
	void inhibateTrans(char const*name,int type,bool inhibate);
	/**\~french Demande l'inhibition de la cible de nom correspondant  l'expression rgulire name (toutes si NULL)\~english Requests the inhibition of the target corresponding to the regular expression name (all if NULL).*/
	void inhibateTarget(char const*name,bool inhibate);
	/**\~french Demande le monitoring d'un lment\~english Requests the monitoring of an element.*/
	void monitorElt(IMonitored&element,int level=1);
	/**\~french Demande le monitoring du ou des variables de nom name (le nom peut tre compos ou une expression rgulire prcde par #).\~english Requests the monitoring of the variable(s) named name (the name can be composite or a regular expression prefixed by #).*/
	void monitorVariable(char const*name,int level=1);
	/**\~french Demande le monitoring de la ou des transitions de nom name (le nom peut tre compos ou une expression rgulire prcde par #).\~english Requests the monitoring of the transition(s) named name (the name can be composite or a regular expression prefixed by #).*/
	void monitorTransition(char const*name,int level=1,char const*stMask=NULL);
	/**\~french Demande le monitoring du ou des tats de nom name (le nom peut tre compos ou une expression rgulire prcde par #).\~english Requests the monitoring of the state(s) named name (the name can be composite or a regular expression prefixed by #).*/
	void monitorState(char const*name,int level=1);
	/**\~french Demande le monitoring du ou des automates de nom name (le nom peut tre compos ou une expression rgulire prcde par #).\~english Requests the monitoring of the automaton(s) named name (the name can be composite or a regular expression prefixed by #).*/
	void monitorAutomaton(char const*name,int level=1);
	/**\~french Fixe le niveau de trace du ou des variables de nom name (le nom peut tre compos ou une expression rgulire prcde par #).
	@param name nom de la variable
	@param level cf. IVariable::trace
	\~english Sets the trace level for the variable(s) named name (the name can be composite or a regular expression prefixed by #).  
	@param name name of the variable  
	@param level see IVariable::trace
	*/
	void traceVariable(char const*name,int level);
	/**\~french Fixe le niveau de trace du ou des transition de nom name (le nom peut tre compos ou une expression rgulire prcde par #).
	@param name nom de la transition
	@param level cf. ITransition::trace
	\~english Sets the trace level for the transition(s) named name (the name can be composite or a regular expression prefixed by #).
	@param name name of the transition
	@param level see ITransition::trace
	*/
	void traceTrans(char const*name,int level);
	/**\~french Fixe le niveau de trace du ou des tats de nom name (le nom peut tre compos).
	@param name nom de l'tat
	@param level cf. IVariable::trace
	\~english Sets the trace level for the state(s) named name (the name can be composite).
	@param name name of the state
	@param level see IVariable::trace
	*/
	void traceState(char const*name,int level);
	/**\~french Fixe le niveau de trace du ou des automates de nom name (le nom peut tre compos).
	@param name nom de l'automate
	@param level cf. IVariable::trace
	\~english Sets the trace level for the automaton(s) named name (the name can be composite).
	@param name name of the automaton
	@param level see IVariable::trace
	*/
	void traceAutomaton(char const*name,int level);
	/**\~french Fixe le niveau de trace du systme.
	@param level cf. TSysTrace::ESysTrace
	\~english Sets the trace level for the system.
	@param level see TSysTrace::ESysTrace
	*/
	void setTrace(int level);
	/**\~french Retourne le niveau de trace du systme.\~english Returns the trace level of the system.*/
	int trace()const;
	/**\~french Demande l'valuation de la vraisemblance (par dfaut faux)\~english Requests the evaluation of the likelihood (default is false).*/
	void setEvalLikelyhood(bool bEval);
	/**\~french Retourne vrai si la vraisemblance est value\~english Returns true if the likelihood is being evaluated.*/
	bool evalLikelyhood()const;
	/**\~french Enregistre les paramtres dans le fichier filename.\~english Saves the parameters to the file filename.*/
	void dumpParameters(char const*filename,bool bBin,TEncoding::EEncoding encoding=TEncoding::unknown)const;
	std::string dumpParameters(bool bBin,TEncoding::EEncoding encoding=TEncoding::unknown)const;
	std::string dumpPythonSystem()const;
	/**\~french Charge les paramtres du fichier (si permissive est vrai, on tente de charger le maximum malgr les erreurs) (si append est vrai, on n'efface rien avant de charger)\~english Loads the parameters from the file (if permissive is true, it attempts to load as much as possible despite errors) (if append is true, nothing is erased before loading).*/
	void loadParameters(char const*filename,bool permissive=false,bool append=false);
	/**\~french Enregistre les rsultats dans le fichier filename.
	@param filename chemin du fichier de rsultats
	@param bBin si vrai les squences sont enregistres dans un fichier binaire
	@param encoding encodage du fichier de rsultat (par dfaut, l'encodage est obtenu en dtectant la prsence de caractres UTF8 ou Latin1 dans les textes du fichier)
	\~english Saves the results to the file filename.  
	@param filename path of the results file  
	@param bBin if true, the sequences are saved in a binary file  
	@param encoding encoding of the results file (by default, the encoding is determined by detecting the presence of UTF8 or Latin1 characters in the text of the file)
	*/
	void dumpResults(char const*filename,bool bBin=true,TEncoding::EEncoding encoding=TEncoding::unknown)const;
	std::string dumpResults(bool bBin=true,TEncoding::EEncoding encoding=TEncoding::unknown)const;
	/**\~french Charge les rsultats  partir du fichier filename.\~english Loads the results from the file filename.*/
	void loadResults(char const*filename);
	/**\~french Charge un fichier de paramtres ou de rsultats.\~english Loads a parameters or results file.*/
	void load(char const*filename);
	/**\~french Retourne l'instant de fin des squences\~english Returns the end instant of the sequences.*/
	PyTime seqEndTime()const;
	PyTime tMax()const;//!<\~french Retourne la date de fin des squences courantes ou souhaites\~english Returns the end date of the current or desired sequences.
	/**\~french Modifie la date de fin de simulation. (si tMax>0 on prolonge les squences du dernier niveau, si tMax=0 on ajout des squences au dernier niveau)\~english Modifies the simulation end date. (If tMax > 0, the sequences at the last level are extended; if tMax = 0, sequences are added at the last level.)*/
	void setTMax(PyTime tMax);
	/**\~french Modifie le nombre max de transitions sans changement de date\~english Modifies the maximum number of transitions without changing the date.*/
	void setMaxTransitionsDt0(unsigned max);
	unsigned maxTransitionsDt0()const;//!<\~french Retourne le nombre max de transitions sans changement de date\~english Returns the maximum number of transitions without changing the date.
	/**\~french Modifie la graine du gnrateur de nombres alatoires.\~english Modifies the seed of the random number generator.*/
	void setRNGSeed(unsigned seed);
	/**\~french Retourne la graine du gnrateur alatoire\~english Returns the seed of the random number generator.*/
	unsigned RNGSeed()const;
	/**\~french Retourne l'tat du gnrateur alatoire lors de l'initialisation de la dernire squence simule
	Cet tat peut tre redonn au systme par setRNGStatus() de manire  rejouer exactement la dernire squence simule.
	\~english Returns the state of the random number generator at the initialization of the last simulated sequence.  
	This state can be restored to the system using setRNGStatus() to exactly replay the last simulated sequence.
	*/
	std::string RNGLastStatus()const;
	/**\~french Dfini l'tat initial du gnrateur alatoire pour la premire squence
	Cette dfinition prend le pas sur la graine du gnrateur, mais cette dernire doit aussi tre fournie si le gnrateur est utilis dans les mthodes check.
	\~english Defines the initial state of the random number generator for the first sequence.
	This definition overrides the seed of the generator, but the seed must also be provided if the generator is used in the check methods.
	*/
	void setRNGStatus(char const*st);
	/**\~french Modifie le gnrateur alatoire utilis (par dfaut KISS).
	Choix du nom parmi :
	- KISS (gnrateur par dfaut)
	- yarn5 (unique gnrateur utilisable avec MPI)
	- mt19937.
	\~english Modifies the random number generator used (default is KISS).  
	Choose the name from:  
	- KISS (default generator)  
	- yarn5 (the only generator usable with MPI)  
	- mt19937.
	*/
	IRNG*setRNG(char const*name);
	/**\~french Retourne le gnrateur alatoire courant (obsolte).\~english Returns the current random number generator (deprecated).*/
	DEPRECATED IRNG*getRNG()const;
	/**\~french Retourne le gnrateur alatoire courant.\~english Returns the current random number generator.*/
	IRNG*RNG()const;
	/**\~french Modifie la taille d'un bloc de gnration de nombre alatoire affect  une squence (par dfaut 0).
	Pour un bloc de taille nulle, on n'effectue pas de saut d'une squence  l'autre.
	\~english Modifies the size of a random number generation block assigned to a sequence (default is 0).  
	For a block of zero size, no jump is made from one sequence to another.*/
	void setRNGBlockSize(unsigned int RNGBloksize);
	DEPRECATED void setRNGBlocSize(unsigned int RNGBlockSize) { ILogManager::glLogManager().msgObsolete("CSystem::setRNGBlocSize", "setRNGBlockSize"); setRNGBlockSize(RNGBlockSize); }
	/**\~french Retourne la taille du bloc de gnration de nombre alatoire utilise.\~english Returns the size of the random number generation block used.*/
	unsigned int RNGBlocSize()const;
	/**\~french Retourne vrai si l'change MPI des squences est effectu par fichiers\~english Returns true if the MPI exchange of sequences is done through files.*/
	bool MPIFileExchange();
	/**\~french Modifie la forme des changes de squences sous MPI
	@param value si vrai, l'change utilise des fichiers, sinon, l'change est effectu par MPI
	\~english Modifies the form of sequence exchanges under MPI.  
	@param value if true, the exchange uses files; otherwise, the exchange is done via MPI.*/
	void setMPIFileExchange(bool value);
	/**\~french Retourne le niveau de trace du gestionnaire d'changes MPI\~english Returns the trace level of the MPI exchange manager.*/
	unsigned int MPITrace()const;
	/**\~french Modifie le niveau de trace du gestionnaire d'changes MPI\~english Modifies the trace level of the MPI exchange manager.*/
	void setMPITrace(unsigned int trace);
	/**\~french Retourne le rang de la tche parmi les tches parallles de MPI (par dfaut 0 hors MPI).\~english Returns the rank of the task among the parallel MPI tasks (default is 0 outside of MPI).*/
	unsigned int MPIRank()const;
	/**\~french Retourne le nombre de tches parallles MPI.\~english Returns the number of parallel MPI tasks.*/
	unsigned int MPINbTasks()const;
	/**\~french Demande le chargement d'une BdC python\~english Requests the loading of a Python Knowledge Base.*/
	static bool loadKBPython(char const*moduleName);
	/**\~french Demande le chargement d'une bibliothque de BdC\~english Requests the loading of a Knowledge Base library.*/
	static bool loadKBLibrary(char const*fileName);
	DEPRECATED static bool loadBdCLibrary(char const*fileName){return loadKBLibrary(fileName);}
	/**\~french Demande le chargement d'un profil par dfaut\~english Requests the loading of a default profile.*/
	static bool loadDefaultProfile(char const *fileName);
	/**\~french Demande l'export d'un profil par dfaut\~english Requests the export of a default profile.*/
	static bool dumpDefaultProfile(char const * fileName, char const *BdC);
	/**\~french Retourne les messages (y compris les messages python si la BdC est charge avec le chargeur python)\~english Returns the messages (including Python messages if the Knowledge Base is loaded with the Python loader).*/
	static char const*messages(bool bClear=true);
	/**\~french Imprime les message (y compris les messages python)\~english Prints the messages (including Python messages).*/
	static void printMessages(bool bClear=true);
	/**\~french Dclare un prparamtrage des variables de nom name\~english Declares a preset for the variables named name.*/
	void setPreAssignment(char const*name,char const*value);
	/**\~french Dclare un prrenommage de la variable FMU de nom name et de FMU FMUName (facultatif)\~english Declares a pre-rename for the FMU variable named name and FMU FMUName (optional).*/
	void setPreRenaming(char const*name,char const*FMUName,char const*newName);
	/**\~french Retourne la proprit d'utilisation des squences filtres pour les indicateurs\~english Returns the usage property of filtered sequences for the indicators.*/
	bool keepFilteredSeqForInd()const;
	/**\~french Modifie la conservation des squences filtres pour le clacul des indicateurs (par dfaut faux)\~english Modifies the retention of filtered sequences for the calculation of indicators (default is false).*/
	void setKeepFilteredSeqForInd(bool keep);
	/**\~french Incrmente le compteur de rfrence de l'objet Python (uniquement pour un systme cr en Python)\~english Increments the reference counter of the Python object (only for a system created in Python).*/
	static PyObject*(*incRef)(PyObject*obj);
	/**\~french Dcrmente le compteur de rfrence de l'objet Python (uniquement pour un systme cr en Python)\~english Decrements the reference counter of the Python object (only for a system created in Python).*/
	static void(*decRef)(PyObject*obj);
	/**\~french Retourne vrai si l'objet est une fonction (uniquement pour un systme cr en Python), lance une exception si name est non null\~english Returns true if the object is a function (only for a system created in Python), throws an exception if name is not null.*/
	static bool(*isCallable)(PyObject*obj,char const*name);
	/**\~french Retourne le message d'erreur de la dernire exception python\~english Returns the error message of the last Python exception.*/
	static char const*(*pythonErrorMsg)();
	/**\~french Retourne l'objet python comme un CNamedComp*\~english Returns the Python object as a CNamedComp*. */
	static CNamedComp* toCNamedComp(PyObject* obj);
	/**\~french Retourne vrai si le nom peut tre utilis dans les expressions des conditions FT\~english Returns true if the name can be used in the expressions of FT conditions.*/
	static bool isFTCompatibleName(char const*name,bool except=true);
	/**\~french Modifie l'option de prise en compte des variables alatoires dans la 1re squence\~english Modifies the option for considering random variables in the first sequence.*/
	void setFirstSequenceRandomInitValues(bool random);
	/**\~french Retourne l'option de prise en compte des variables alatoires dans la 1re squence\~english Returns the option for considering random variables in the first sequence.*/
	bool firstSequenceRandomInitValues()const;
	/**\~french Modifie l'option de stockage de la squence dterministe (par dfaut vrai)\~english Modifies the option for storing the deterministic sequence (default is true).*/
	void setStoreDeterministicSeq(bool store);
	/**\~french Retourne l'option de stockage de la squence dterministe\~english Returns the option for storing the deterministic sequence.*/
	bool storeDeterministicSeq()const;
	/**\~french Retourne l'outil de paramtrage du tirage prfrentiel\~english Returns the tool for configuring preferential sampling.*/
	IImportanceSampling*importanceSampling()const;
	/**\~french Dfinit l'emploi du tirage prfrentiel (par dfaut faux)\~english Defines the use of preferential sampling (default is false).*/
	void setUseImportanceSampling(bool bUse);
	/**\~french Retourne l'outil de paramtrage du Monte Carlo squentiel\~english Returns the tool for configuring sequential Monte Carlo.*/
	ISeqMonteCarlo*seqMonteCarlo()const;
	/**\~french Dfinit l'emploi du Monte Carlo squentiel (par dfaut faux)\~english Defines the use of sequential Monte Carlo (default is false).*/
	void setUseSeqMonteCarlo(bool bUse);
	/**\~french Retourne l'outil de paramtrage de l'explorateur d'arbres de squences\~english Returns the tool for configuring the sequence tree explorer.*/
	ISeqTreeExplorer*seqTreeExplorer()const;
	/**\~french Dfinit l'emploi de l'explorateur d'arbres de squences (par dfaut faux)\~english Defines the use of the sequence tree explorer (default is false).*/
	void setUseSeqTreeExplorer(bool bUse);
	/**\~french Retourne l'outil de paramtrage du gnrateur d'arbre de dfaillances\~english Returns the tool for configuring the fault tree generator.*/
	IFaultTreeGenerator*faultTreeGenerator()const;
	/**\~french Dfinit l'emploi du gnrateur d'arbres de dfaillances (par dfaut faux)\~english Defines the use of the fault tree generator (default is false).*/
	void setUseFaultTreeGenerator(bool bUse);
	/**\~french Retourne l'outil de paramtrage de l'optimisation de processus de dcisions markoviens\~english Returns the tool for configuring the optimization of Markov decision processes.*/
	IMDPManager*MDPManager()const;
	/**\~french Dfinit l'emploi de l'optimisation de processus de dcisions markoviens (par dfaut faux)\~english Defines the use of Markov decision process optimization (default is false).*/
	void setUseMDPManager(bool bUse);
	/**\~french Charge un FMU  partir du fichier fmuPath et l'instancie avec le nom name (s'il existe dj, on retourne celui l ; attention ! pas de vrification qu'il s'agit du mme fmu)\~english Loads an FMU from the file fmuPath and instantiates it with the name name (if it already exists, returns that one; caution! No verification that it is the same FMU).*/
	IFunctionalMockupUnit*addFMU(char const*fmuPath,char const*name);
	/**\~french Retourne le FMU de nom\~english Returns the FMU named name.*/
	IFunctionalMockupUnit*FMU(char const*name)const;
	/**\~french Applique la tratgie strategy (crase les stratgies prcdentes sans les supprimer)\~english Applies the strategy strategy (overwrites previous strategies without deleting them).*/
	void applyStrategy(IStrategy&strategy);
	/**\~french Supprime toutes les stratgies en cours\~english Deletes all current strategies.*/
	void clearStrategies();
};

#endif
