/*************************************
*         Copyright 2015 EDF         *
*************************************/
#pragma once
/**\file
* \~french Fichier de dfinition de l'accessibilit Python aux classes de base de PyCATSHOO.
*
* Ce fichier est principalement utilis pour construire un wrapper Python pour les bases de connaissances crites en C++.
* 
* Le module Boost Python doit commencer par l'appel de la fonction export_component().
* \~english File for defining Python accessibility to PyCATSHOO base classes.
*
* This file is mainly used to build a Python wrapper for knowledge bases written in C++.
* 
* The Boost Python module must begin by a call to the function export_component().
*/
#define BOOST_PYTHON_STATIC_LIB
#include <boost/python/class.hpp>
#include <boost/python/module.hpp>
#include <boost/python/implicit.hpp>
#include <boost/python/def.hpp>
#include <boost/python/call_method.hpp>
#include <boost/python/overloads.hpp>
#include <boost/python/reference_existing_object.hpp>
#include <boost/python/manage_new_object.hpp>
#include <boost/python/copy_const_reference.hpp>
#include <boost/python/return_value_policy.hpp>
#include <boost/python/pure_virtual.hpp>
#include <boost/python/enum.hpp>
#include <boost/python/list.hpp>
#include <boost/python/suite/indexing/vector_indexing_suite.hpp>
#include <boost/python/suite/indexing/map_indexing_suite.hpp>
#include <boost/python/exception_translator.hpp>
#include <boost/python/wrapper.hpp>
#include <boost/python/handle.hpp>

typedef boost::python::override TPythonOverride;

#include <iostream>
#include <string>
#include <vector>
#include "Component.h"
#include "PyFctFiltre.h"
#include "IFunctionalMockupUnit.h"
#include "Version.h"

boost::python::object IVariable_value(IVariable* var) {
	std::type_info const& ti = var->typeInfo();
	if(ti == typeid(bool))
		return boost::python::object(var->bValue());
	if(ti == typeid(int))
		return boost::python::object(var->iValue());
	if(ti == typeid(float))
		return boost::python::object(var->fValue());
	if(ti == typeid(double))
		return boost::python::object(var->dValue());
	if(ti == typeid(std::complex<double>))
		return boost::python::object(var->cValue());
	if(ti == typeid(std::string))
		return boost::python::object(var->sValue());
	ILogManager::glLogManager().throwError(formatMsg("Type inconnu pour la variable %s", var->name()).c_str());
	return boost::python::object();
}

boost::python::object IVariable_value(IVariable* var, std::vector<int>const&pt) {
	std::type_info const& ti = var->typeInfo();
	if(ti == typeid(bool))
		return boost::python::object(var->bValue(pt));
	if(ti == typeid(int))
		return boost::python::object(var->iValue(pt));
	if(ti == typeid(float))
		return boost::python::object(var->fValue(pt));
	if(ti == typeid(double))
		return boost::python::object(var->dValue(pt));
	if(ti == typeid(std::complex<double>))
		return boost::python::object(var->cValue(pt));
	if(ti == typeid(std::string))
		return boost::python::object(var->sValue(pt));
	ILogManager::glLogManager().throwError(formatMsg("Type inconnu pour la variable %s", var->name()).c_str());
	return boost::python::object();
}

boost::python::object IVariable_initValue(IVariable* var) {
	std::type_info const& ti = var->typeInfo();
	if(ti == typeid(bool))
		return boost::python::object(var->bInitValue());
	if(ti == typeid(int))
		return boost::python::object(var->iInitValue());
	if(ti == typeid(float))
		return boost::python::object(var->fInitValue());
	if(ti == typeid(double))
		return boost::python::object(var->dInitValue());
	if(ti == typeid(std::complex<double>))
		return boost::python::object(var->cInitValue());
	if(ti == typeid(std::string))
		return boost::python::object(var->sInitValue());
	ILogManager::glLogManager().throwError(formatMsg("Type inconnu pour la variable %s", var->name()).c_str());
	return boost::python::object();
}

boost::python::object IVariable_initValue(IVariable* var, std::vector<int>const&pt) {
	std::type_info const& ti = var->typeInfo();
	if(ti == typeid(bool))
		return boost::python::object(var->bInitValue(pt));
	if(ti == typeid(int))
		return boost::python::object(var->iInitValue(pt));
	if(ti == typeid(float))
		return boost::python::object(var->fInitValue(pt));
	if(ti == typeid(double))
		return boost::python::object(var->dInitValue(pt));
	if(ti == typeid(std::complex<double>))
		return boost::python::object(var->cInitValue(pt));
	if(ti == typeid(std::string))
		return boost::python::object(var->sInitValue(pt));
	ILogManager::glLogManager().throwError(formatMsg("Type inconnu pour la variable %s", var->name()).c_str());
	return boost::python::object();
}

boost::python::object IVarRef_value(IReference* ref, CNX_ID i) {
	std::type_info const& ti = ref->typeInfo(i);
	if(ti == typeid(bool))
		return boost::python::object(ref->bValue(i));
	if(ti == typeid(int))
		return boost::python::object(ref->iValue(i));
	if(ti == typeid(float))
		return boost::python::object(ref->fValue(i));
	if(ti == typeid(double))
		return boost::python::object(ref->dValue(i));
	if(ti == typeid(std::complex<double>))
		return boost::python::object(ref->cValue(i));
	if(ti == typeid(std::string))
		return boost::python::object(ref->sValue(i));
	ILogManager::glLogManager().throwError(formatMsg("Type inconnu pour la %dme connexion de la rfrence %s", (int)i, ref->name()).c_str());
	return boost::python::object();
}

boost::python::object IVarRef_value(IReference* ref, CNX_ID i, std::vector<int>const&pt) {
	std::type_info const& ti = ref->typeInfo(i);
	if(ti == typeid(bool))
		return boost::python::object(ref->bValue(i, pt));
	if(ti == typeid(int))
		return boost::python::object(ref->iValue(i, pt));
	if(ti == typeid(float))
		return boost::python::object(ref->fValue(i, pt));
	if(ti == typeid(double))
		return boost::python::object(ref->dValue(i, pt));
	if(ti == typeid(std::complex<double>))
		return boost::python::object(ref->cValue(i, pt));
	if(ti == typeid(std::string))
		return boost::python::object(ref->sValue(i, pt));
	ILogManager::glLogManager().throwError(formatMsg("Type inconnu pour la %dme connexion de la rfrence %s", (int)i, ref->name()).c_str());
	return boost::python::object();
}

/*Dtermination du contenu d'un objet python en thermes de fct et/ou de nom de fct*/
bool PyFunctionCheck(PyObject*&pyObj,std::string&name,int iP=0){
	CPyGIL l_GIL;
	if(PyCallable_Check(pyObj)) {
		PyObject* name_attr = PyObject_GetAttrString(pyObj, "__name__");
		if(name_attr && PyUnicode_Check(name_attr))
			name = boost::python::extract<std::string>(name_attr);
		Py_XDECREF(name_attr);
		return true;
	}else if(pyObj && PyUnicode_Check(pyObj)){
		name= boost::python::extract<std::string>(pyObj);
		pyObj=NULL;
		return true;
	}
	if(iP>0)
		ILogManager::glLogManager().throwError(formatMsg("Le %dme paramtre n'est pas une mthode ni un nom de mthode",iP).c_str());
	return false;
}

/*Surcharge de la classe IDistLaw pour permettre une spcialisation en Python*/
struct PyDistLaw:public IDistLaw,public boost::python::wrapper<IDistLaw>{
	PyDistLaw(PyObject*self,CNamedComp&parent):IDistLaw(self,parent){
		boost::python::detail::initialize_wrapper(self,this);
	}
	char const*name()const override{
		CPyGIL l_GIL;
		return get_override("name")();
	}
	PyTime delay()const override{
		CPyGIL l_GIL;
		TPythonOverride f=get_override("delay");
		return f?f():IDistLaw::delay();
	}
	PyTime delayBeyond(PyTime t)const override{
		CPyGIL l_GIL;
		return get_override("delayBeyond")(t);
	}
	double density(PyTime t)const override{
		CPyGIL l_GIL;
		return get_override("density")(t);
	}
	double distribution(PyTime t)const override{
		CPyGIL l_GIL;
		return get_override("distribution")(t);
	}
};

//Wrappers de la classe IDistLaw
IDistLaw* IDistLaw_newLaw(CNamedComp& parent, TLawType::ELawType type, char const*name,PyObject* pyObj) {
	return IDistLaw::newLaw(parent, type, name, pyObj);
}
IDistLaw* IDistLaw_newLaw(CNamedComp & parent, TLawType::ELawType type, PyObject * pyObj) {
	std::string name;
	if(PyFunctionCheck(pyObj,name))
		return IDistLaw_newLaw(parent,type,name.c_str(),pyObj);
	boost::python::extract<double>dbl(pyObj);
	if(dbl.check())
		return IDistLaw::newLaw(parent,type,dbl);
	return IDistLaw::newLaw(parent,type,*boost::python::extract<IVariable*>(pyObj));
}
void IDistLaw_setParameter(IDistLaw* self, char const* name, PyObject* pyObj, LAWP_ID Id) {
	self->setParameter(self->parent(), name, pyObj, Id);
}
void IDistLaw_setParameter(IDistLaw*self,char const*name,PyObject*pyObj){
	if(PyCallable_Check(pyObj))
		IDistLaw_setParameter(self,name,pyObj,0);
	else
		IDistLaw_setParameter(self,name,NULL,(LAWP_ID)PyLong_AsLong(pyObj));
}
void IDistLaw_setParameter(IDistLaw*self,PyObject*pyObj){
	std::string name;
	if(PyFunctionCheck(pyObj,name))
		IDistLaw_setParameter(self,name.c_str(),pyObj,0);
	else{
		boost::python::extract<double>dbl(pyObj);
		if(dbl.check())
			self->setParameter(dbl());
		else
			self->setParameter(*boost::python::extract<IVariable*>(pyObj));
	}
}
void IDistLaw_setParameter(IDistLaw*self,PyObject*pyObj,LAWP_ID Id){
	std::string name;
	if(PyFunctionCheck(pyObj,name))
		IDistLaw_setParameter(self,name.c_str(),pyObj,0);
	else{
		boost::python::extract<double>dbl(pyObj);
		if(dbl.check())
			self->setParameter(dbl(),Id);
		else
			self->setParameter(*boost::python::extract<IVariable*>(pyObj),Id);
	}
}
void IDistLaw_insertParameter(IDistLaw*self,double val){self->insertParameter(val);}
void IDistLaw_insertParameter(IDistLaw*self,IVariable&val){self->insertParameter(val);}

//Wrappers de la classe ITransition
IDistLaw* ITransition_setDistLaw(ITransition* self, TLawType::ELawType type, char const*name, PyObject* pyObj) {
	return self->setDistLaw(type, name, pyObj);
}
IDistLaw* ITransition_setDistLaw(ITransition* self, TLawType::ELawType type, PyObject* pyObj) {
	std::string name;
	if(PyFunctionCheck(pyObj, name))
		return ITransition_setDistLaw(self, type, name.c_str(), pyObj);
	boost::python::extract<double>dbl(pyObj);
	if(dbl.check())
		return self->setDistLaw(type,dbl());
	return self->setDistLaw(type,*boost::python::extract<IVariable*>(pyObj));
}
void ITransition_setCondition(ITransition* self, char const* name, PyObject* pyObj, bool negate) {
	self->setCondition(name, pyObj, negate);
}
void ITransition_setCondition(ITransition * self, PyObject * pyObj, bool negate) {
		std::string name;
	if(PyFunctionCheck(pyObj,name))
		ITransition_setCondition(self,name.c_str(),pyObj,negate);
	else//Ca ne peut pas tre un boolen
		self->setCondition(*boost::python::extract<IVariable*>(pyObj),negate);
}
void ITransition_setCondition(ITransition*self,PyObject*pyObj){
	boost::python::extract<bool>bl(pyObj);
	if(bl.check())
		self->setCondition(bl());
	else
		ITransition_setCondition(self,pyObj,false);
}
void ITransition_setCondition(ITransition*self,char const*val,PyObject*pyObj){//On ne sait pas si l'appel sera prioritaire par rapport  (pyObj,negate)
	if(PyCallable_Check(pyObj))
		ITransition_setCondition(self,val,pyObj,false);
	else{
		boost::python::extract<bool>bl(pyObj);
		if(bl.check())
			ITransition_setCondition(self,val,NULL,bl);
		else
			ILogManager::glLogManager().throwError("Le 2me paramtre n'est ni une mthode ni un boolen");
	}
}
void ITransition_setBEParameter(ITransition* self, char const*name, PyObject* pyObj, BEP_ID place) {
	self->setBEParameter(name, pyObj, place);
}
void ITransition_setBEParameter(ITransition* self, PyObject* pyObj, BEP_ID place) {
	if(PyCallable_Check(pyObj))
		ITransition_setBEParameter(self,NULL,pyObj,place);
	else{
		boost::python::extract<double>dbl(pyObj);
		if(dbl.check())
			self->setBEParameter(dbl,place);
		else
			self->setBEParameter(boost::python::extract<IVariable*>(pyObj),place);
	}
}

void ITransition_addCallback(ITransition* self, char const* name, PyObject* pVoid, int outState) { self->addCallback(name, NULL, pVoid, outState); }
void ITransition_addCallback(ITransition* self, PyObject*pyObj, PyObject* pyObj2) {
	std::string name;
	PyFunctionCheck(pyObj, name, 1);
	if(pyObj == NULL && PyCallable_Check(pyObj2))
		ITransition_addCallback(self, name.c_str(), pyObj2, -1);
	else
		ITransition_addCallback(self, name.c_str(), pyObj, (int)PyLong_AsLong(pyObj2));
}
void ITransition_addCallback(ITransition* self, PyObject*pyObj) {
	std::string name;
	PyFunctionCheck(pyObj, name, 1);
	ITransition_addCallback(self, name.c_str(), pyObj, -1);
}
void ITransition_addSensitiveMethod(ITransition* self, char const* name, PyObject* pVoid, int outState) { ILogManager::glLogManager().msgObsolete("ITransition::addSensitiveMethod", "addCallback"); ITransition_addCallback(self, name, pVoid, outState); }
void ITransition_addSensitiveMethod(ITransition* self, char const* name, PyObject* pVoid) { ITransition_addSensitiveMethod(self, name, pVoid, -1); }
void ITransition_addSensitiveMethod(ITransition* self, char const* name) { ITransition_addSensitiveMethod(self, name, NULL, -1); }
void ITransition_addTarget(ITransition*self,IState*st){self->addTarget(st,TTransType::trans);}

//Wrappers de la class IPDMPManager
void IPDMPManager_addBeginMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid) { self->addBeginMethod(&comp, name, NULL, pVoid); }
void IPDMPManager_addBeginMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp) { IPDMPManager_addBeginMethod(self, name, comp, NULL); }
void IPDMPManager_addEndMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid) { self->addEndMethod(&comp, name, NULL, pVoid); }
void IPDMPManager_addEndMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp) { IPDMPManager_addEndMethod(self, name, comp, NULL); }
void IPDMPManager_addPreStepMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid) { self->addPreStepMethod(&comp, name, NULL, pVoid); }
void IPDMPManager_addPreStepMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp) { IPDMPManager_addPreStepMethod(self, name, comp, NULL); }
void IPDMPManager_addPostStepMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid) { self->addPostStepMethod(&comp, name, NULL, pVoid); }
void IPDMPManager_addPostStepMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp) { IPDMPManager_addPostStepMethod(self, name, comp, NULL); }
void IPDMPManager_addEquationMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid, int order) { self->addEquationMethod(&comp, name, NULL, pVoid, order); }
void IPDMPManager_addEquationMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, int order) { IPDMPManager_addEquationMethod(self,name, comp, NULL, order); }
void IPDMPManager_addEquationMethod(IPDMPManager*self,char const*name, CSysNamedComp&comp){ IPDMPManager_addEquationMethod(self,name,comp,NULL,0);}
void IPDMPManager_addEquationMethod(IPDMPManager*self,char const*name, CSysNamedComp&comp,PyObject*pyObj){
	if(PyCallable_Check(pyObj))
		IPDMPManager_addEquationMethod(self,name,comp,pyObj,0);
	else
		IPDMPManager_addEquationMethod(self,name,comp,NULL,(int)PyLong_AsLong(pyObj));
}
void IPDMPManager_addJacobianMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid, int order) { self->addJacobianMethod(&comp, name, NULL, pVoid, order); }
void IPDMPManager_addJacobianMethod(IPDMPManager* self, char const* name, CSysNamedComp& comp, int order) { IPDMPManager_addJacobianMethod(self, name, comp, NULL, order); }
void IPDMPManager_addCondition(IPDMPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid) { self->addCondition(&comp, name, NULL, pVoid); }
void IPDMPManager_addCondition(IPDMPManager* self, char const* name, CSysNamedComp& comp) { IPDMPManager_addCondition(self, name, comp, NULL); }

void ISLEManager_addEquation(ISLEManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid) { self->addEquation(name, comp, pVoid); }
void ISLEManager_addEquation(ISLEManager* self, char const* name, CSysNamedComp& comp) { ISLEManager_addEquation(self, name, comp, NULL); }

void IMILPManager_addInequality(IMILPManager* self, char const* name, CSysNamedComp& comp, PyObject* pVoid) { self->addInequality(name, comp, pVoid); }
void IMILPManager_addInequality(IMILPManager* self, char const* name, CSysNamedComp& comp) { IMILPManager_addInequality(self, name, comp, NULL); }

void IEquation_setCoefficient(IEquation*self,IVariable*p,double c){self->setCoefficient(p,c);}
void IEquation_setConstant(IEquation*self,double c){self->setConstant(c);}

void IVarBase_addCallback(IVarBase* self, char const* name, PyObject* pyObj, int sens) { self->addCallback(name, NULL, pyObj, sens); }
void IVarBase_addCallback(IVarBase* self, PyObject* pyObj) {
	std::string name;
	PyFunctionCheck(pyObj, name, 1);
	IVarBase_addCallback(self, name.c_str(), pyObj, 0);
}
void IVarBase_addCallback(IVarBase* self, PyObject*pyObj, PyObject* pyObj2) {
	std::string name;
	PyFunctionCheck(pyObj, name, 1);
	if(pyObj==NULL && PyCallable_Check(pyObj2))
		IVarBase_addCallback(self, name.c_str(), pyObj2, 0);
	else
		IVarBase_addCallback(self, name.c_str(), pyObj, (int)PyLong_AsLong(pyObj2));
}
void IVarBase_addSensitiveMethod(IVarBase* self, char const* name, PyObject* pyObj, int sens) { ILogManager::glLogManager().msgObsolete("IVarBase::addSensitiveMethod", "addCallback"); IVarBase_addCallback(self, name, pyObj, sens); }
void IVarBase_addSensitiveMethod(IVarBase* self, char const* name) { IVarBase_addSensitiveMethod(self, name, (PyObject*)NULL, 0); }
void IVarBase_addSensitiveMethod(IVarBase*self,char const*name,PyObject*pyObj){
	if(PyCallable_Check(pyObj))
		IVarBase_addSensitiveMethod(self,name,pyObj,0);
	else
		IVarBase_addSensitiveMethod(self,name,NULL,(int)PyLong_AsLong(pyObj));
}

bool IReference_orValue(IReference*self){return self->orValue();}
bool IReference_andValue(IReference*self){return self->andValue();}
double IReference_sumValue(IReference*self){return self->sumValue();}
double IReference_productValue(IReference*self){return self->productValue();}
int IReference_valuesThat(IReference*self,PyObject*pyCond,std::vector<int>&indexes){return self->valuesThat(CPyCondFctFloat(pyCond),indexes);}

boost::python::handle<PyObject>CComponent_self(CComponent*self){
	return boost::python::handle<>(boost::python::borrowed(self->self()));
}

char const* pycatshooVersion() { return VER_FILEVERSION_STR; }

using namespace boost::python;

/**Classe de gestion de la conversion entre liste Python et tableau C*/
template<typename T>class PyArray {
	T* m_Ptr;
	size_t m_Size;
	int m_Options;
	enum {
		o_const = 1,
		o_loc = 2
	};
public:
	PyArray(T* ptr, long size) :m_Ptr(ptr), m_Size(size), m_Options(0) {}
	PyArray(T const* ptr, long size) :m_Ptr(const_cast<T*>(ptr)), m_Size(size), m_Options(o_const) {}
	PyArray(long size) :m_Ptr(size>0?new T[size]:NULL), m_Size(size>0?size:0), m_Options(o_loc) {}
	PyArray(double val, long size) :m_Ptr(size>0?new T[size]:NULL), m_Size(size>0?size:0), m_Options(o_loc) {
		for(long i = 0; i < size; i++)
			m_Ptr[i] = val;
	}
	~PyArray() {
		if(m_Options&o_loc)
			delete[]m_Ptr;
	}
	operator T* () {
		return m_Ptr;
	}
	operator T const* ()const {
		return m_Ptr;
	}
	long convert_index(long index) {
		return index < 0 ? index + (long)m_Size : index;
	}
	T item(int index) {
		index = convert_index(index);
		if(index < 0 || index >= m_Size) {
			PyErr_SetString(PyExc_IndexError, "index out of range");
			throw boost::python::error_already_set();
		}
		return m_Ptr[index];
	}
	PyArray<T>* slice(PyObject* pslice) {
		PySliceObject* slice = reinterpret_cast<PySliceObject*>(pslice);
		if(slice->step != Py_None) {
			PyErr_SetString(PyExc_IndexError, "slice step not supported");
			throw boost::python::error_already_set();
		}
		long from = convert_index(slice->start != Py_None ? PyLong_AsLong(slice->start) : 0), to = convert_index(slice->stop != Py_None ? PyLong_AsLong(slice->stop) : m_Size);
		if(from < 0 || from > to || to >= m_Size) {
			PyErr_SetString(PyExc_IndexError, "index out of range");
			throw boost::python::error_already_set();
		}
		return new PyArray(m_Ptr + from, to - from);
	}
	void setItem(long index, T val) {
		index = convert_index(index);
		if(index < 0 || index >= m_Size) {
			PyErr_SetString(PyExc_IndexError, "index out of range");
			throw boost::python::error_already_set();
		}
		if(m_Options & o_const) {
			PyErr_SetString(PyExc_IndexError, "not modifiable array");
			throw boost::python::error_already_set();
		}
		m_Ptr[index] = val;
	}
	int size()const {
		return m_Size;
	}
};

void* convertPyObject2Void(PyObject* obj) {
	return static_cast<void*>(obj);
}

double* convertPyArray2Double(PyArray<double>& pyArray) {
	return pyArray;
}

/** Type that allows for registration of conversions from python iterable types.*/
struct iterable_converter
{
	/** Registers converter from a python interable type to the provided type.*/
	template <typename Container>iterable_converter&from_python(){
		boost::python::converter::registry::push_back(
			&iterable_converter::convertible,
			&iterable_converter::construct<Container>,
			boost::python::type_id<Container>());

		// Support chaining.
		return *this;
	}

	/* Check if PyObject is iterable.*/
	static void* convertible(PyObject* object)
	{
		return PyObject_GetIter(object) ? object : NULL;
	}

	/** Convert iterable PyObject to C++ container type.

	Container Concept requirements:

	* Container::value_type is CopyConstructable.
	* Container can be constructed and populated with two iterators.
	I.e. Container(begin, end)*/
	template <typename Container>static void construct(PyObject* object, boost::python::converter::rvalue_from_python_stage1_data* data)
	{
		namespace python = boost::python;
		// Object is a borrowed reference, so create a handle indicting it is borrowed for proper reference counting.
		python::handle<> handle(python::borrowed(object));

		// Obtain a handle to the memory block that the converter has allocated for the C++ type.
		typedef python::converter::rvalue_from_python_storage<Container>storage_type;
		void* storage = reinterpret_cast<storage_type*>(data)->storage.bytes;

		typedef python::stl_input_iterator<typename Container::value_type>iterator;

		// Allocate the C++ type into the converter's memory block, and assign
		// its handle to the converter's convertible variable.  The C++
		// container is populated by passing the begin and end iterators of
		// the python object to the container's constructor.
		new (storage) Container(
			iterator(python::object(handle)), // begin
			iterator());                      // end
		data->convertible = storage;
	}
};

#ifdef _MSC_VER
#pragma warning(disable:4244 4267 4996)
#else
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#endif

/**\~frenchExportation Python des lments de base de PyCATSHOO.
* 
* Le paramtre bWithKB doit tre vrai (utilis uniquement pour viter des conflits d'exportation pour le module Pycatshoo).
* \~englishPython export of PyCATSHOO's basic elements.
*
* The bWithKB parameter must be true (used only to avoid export conflicts for the Pycatshoo module).
*/
void export_component(bool bWithKB=true){
	def("pycatshooVersion", pycatshooVersion);
	boost::python::converter::registry::insert(convertPyObject2Void, boost::python::type_id<void*>());
	boost::python::implicitly_convertible<PyArray<double>, double*>();
	boost::python::implicitly_convertible<PyArray<double>, double const*>();
	// Register interable conversions.
	iterable_converter()
		// Build-in types.
		.from_python<std::vector<bool> >()
		.from_python<std::vector<int> >()
		.from_python<std::vector<float> >()
		.from_python<std::vector<double> >()
		// Complex types.
		.from_python<std::vector<std::string> >()
		;

	/*Export de l'numr de dfinition du type de franchissement d'une transition.*/
	enum_<TTransType::ETransType>("TTransType")
		.value("trans",TTransType::trans)
		.value("fault",TTransType::fault)
		.value("rep",TTransType::rep);
		
	/*Export de l'numr de dfinition du type de loi de probabilit.*/
	enum_<TLawType::ELawType>("TLawType")
		.value("inst",TLawType::inst)
		.value("expo",TLawType::expo)
		.value("cstt",TLawType::cstt)
		.value("defer",TLawType::defer)
		.value("cstd",TLawType::cstd)
		.value("weib",TLawType::weib)
		.value("uniform",TLawType::uniform)
		.value("normal",TLawType::normal)
		;
		
	/*Export de l'numr de dfinition du type de loi de probabilit.*/
	enum_<TBEType::EBEType>("TBEType")
		.value("unknown",TBEType::unknown)
		.value("probability",TBEType::probability)
		.value("exponential",TBEType::exponential)
		.value("GLM",TBEType::GLM)
		.value("Weibull",TBEType::Weibull)
		.value("periodic4",TBEType::periodic4)
		.value("periodic5",TBEType::periodic5)
		.value("periodic11",TBEType::periodic11)
		;

	/*Export de l'numr de dfinition du type d'une variable d'tat.*/
	enum_<TVarType::EVarType>("TVarType")
		.value("t_unknown",TVarType::t_unknown)
		.value("t_bool", TVarType::t_bool)
		.value("t_int", TVarType::t_int)
		.value("t_float", TVarType::t_float)
		.value("t_double", TVarType::t_double)
		.value("t_complex", TVarType::t_complex)
		.value("t_string", TVarType::t_string)
		.value("t_abool", TVarType::t_abool)
		.value("t_aint", TVarType::t_aint)
		.value("t_afloat", TVarType::t_afloat)
		.value("t_adouble", TVarType::t_adouble)
		.value("t_acomplex", TVarType::t_acomplex)
		.value("t_astring", TVarType::t_astring)
		;

	/*Export de l'numr de dfinition de l'aspect modifiable d'une variable d'tat.*/
	enum_<TModificationMode::EModificationMode>("TModificationMode")
		.value("not_modifiable",TModificationMode::not_modif)
		.value("modifiable",TModificationMode::modif)
		.value("discrete_modification",TModificationMode::discrete_modification)
		.value("continuous_modification",TModificationMode::continuous_modification)
		.value("cont_modifiable",TModificationMode::cont_modif);

	/*Export de l'numr de dfinition du type d'une variable d'tat.*/
	enum_<TMILPAlgorithmType::EMILPAlgorithmType>("TMILPAlgorithmType")
		.value("simplex",TMILPAlgorithmType::simplex)
		.value("interior_point",TMILPAlgorithmType::interior_point)
		.value("milp",TMILPAlgorithmType::milp);

	enum_<TMILPBacktrackingType::EMILPBacktrackingType>("TMILPBacktrackingType")
		.value("depth_first",TMILPBacktrackingType::depth_first)
		.value("breadth_first",TMILPBacktrackingType::breadth_first)
		.value("best_local_bound",TMILPBacktrackingType::best_local_bound)
		.value("best_projection",TMILPBacktrackingType::best_projection);

	enum_<TMILPBranchingType::EMILPBranchingType>("TMILPBranchingType")
		.value("first_frac_variable",TMILPBranchingType::first_frac_variable)
		.value("last_frac_variable",TMILPBranchingType::last_frac_variable)
		.value("most_frac_variable",TMILPBranchingType::most_frac_variable)
		.value("DrTom_heuristic",TMILPBranchingType::DrTom_heuristic)
		.value("pseudo_cost_heuristic",TMILPBranchingType::pseudo_cost_heuristic);

	enum_<TMILPIntPointType::EMILPIntPointType>("TMILPIntPointType")
		.value("none",TMILPIntPointType::none)
		.value("quot_min_degree",TMILPIntPointType::quot_min_degree)
		.value("appr_min_degree",TMILPIntPointType::appr_min_degree)
		.value("sym_appr_min_degree",TMILPIntPointType::sym_appr_min_degree);

	enum_<TMILPMsgLevel::EMILPMsgLevel>("TMILPMsgLevel")
		.value("off",TMILPMsgLevel::off)
		.value("error",TMILPMsgLevel::error)
		.value("on",TMILPMsgLevel::on)
		.value("all",TMILPMsgLevel::all);

	enum_<TMILPOptionsType::EMILPOptionsType>("TMILPOptionsType")
		.value("none",TMILPOptionsType::none)
		.value("pr_steepest_edge",TMILPOptionsType::pr_steepest_edge)
		.value("hr_ratio_test",TMILPOptionsType::hr_ratio_test)
		.value("presolve",TMILPOptionsType::presolve)
		.value("binarize",TMILPOptionsType::binarize)
		.value("sr_heuristic",TMILPOptionsType::sr_heuristic)
		.value("fp_heuristic",TMILPOptionsType::fp_heuristic)
		.value("ps_heuristic",TMILPOptionsType::ps_heuristic)
		.value("gmi_cuts",TMILPOptionsType::gmi_cuts)
		.value("mir_cuts",TMILPOptionsType::mir_cuts)
		.value("cov_cuts",TMILPOptionsType::cov_cuts)
		.value("clq_cuts",TMILPOptionsType::clq_cuts);

	enum_<TMILPPreprocessingType::EMILPPreprocessingType>("TMILPPreprocessingType")
		.value("none",TMILPPreprocessingType::none)
		.value("root",TMILPPreprocessingType::root)
		.value("all",TMILPPreprocessingType::all);

	enum_<TMILPScalingType::EMILPScalingType>("TMILPScalingType")
		.value("geom_mean",TMILPScalingType::geom_mean)
		.value("equilib",TMILPScalingType::equilib)
		.value("near_p2",TMILPScalingType::near_p2)
		.value("none",TMILPScalingType::none)
		.value("auto",TMILPScalingType::automatic);

	enum_<TMILPSimplexType::EMILPSimplexType>("TMILPSimplexType")
		.value("primal",TMILPSimplexType::primal)
		.value("dual",TMILPSimplexType::dual)
		.value("dual_primal",TMILPSimplexType::dual_primal);

	enum_<TDistributionType::EDistributionType>("TDistributionType")
		.value("none",TDistributionType::none)
		.value("expo",TDistributionType::expo)
		.value("frechet",TDistributionType::frechet)
		.value("gamma",TDistributionType::gamma)
		.value("gumbel",TDistributionType::gumbel)
		.value("lognormal",TDistributionType::lognormal)
		.value("normal",TDistributionType::normal)
		.value("pareto",TDistributionType::pareto)
		.value("triangle",TDistributionType::triangle)
		.value("uniforme",TDistributionType::uniforme)
		.value("weibull",TDistributionType::weibull);

	enum_<TMDPSolverType::ESolverType>("TMDPSolverType")
		.value("custom",TMDPSolverType::custom)
		.value("pei",TMDPSolverType::pei)
		.value("mras",TMDPSolverType::mras);

	enum_<TCausality::ECausality>("TCausality")
		.value("unknown",TCausality::unknown)
		.value("parameter",TCausality::parameter)
		.value("input",TCausality::input)
		.value("output",TCausality::output);

	enum_<TVariability::EVariability>("TVariability")
		.value("unknown",TVariability::unknown)
		.value("parameter",TVariability::constant)
		.value("input",TVariability::discrete)
		.value("output",TVariability::continuous);

	enum_<TInitializability::EInitializability>("TInitializability")
		.value("unknown",TInitializability::unknown)
		.value("parameter",TInitializability::exact)
		.value("input",TInitializability::approx)
		.value("output",TInitializability::calculated);

	enum_<TFMUTrace::EFMUTrace>("TFMUTrace")
		.value("none",TFMUTrace::none)
		.value("management",TFMUTrace::management)
		.value("step",TFMUTrace::step)
		.value("state",TFMUTrace::state);

	class_< std::vector<IState*> >("VectorSt")
        .def(vector_indexing_suite< std::vector<IState*> >() );
	class_< std::vector<CComponent*> >("VectorComp")
        .def(vector_indexing_suite< std::vector<CComponent*> >() );
	class_< std::vector<std::string> >("VectorString")
        .def(vector_indexing_suite< std::vector<std::string> >() );
	class_< std::vector<ITransition*> >("VectorTrans")
        .def(vector_indexing_suite< std::vector<ITransition*> >() );
	class_< std::vector<IVariable*> >("VectorVar")
        .def(vector_indexing_suite< std::vector<IVariable*> >() );
	class_< std::vector<IAutomaton*> >("VectorAut")
        .def(vector_indexing_suite< std::vector<IAutomaton*> >() );
	class_< std::vector<IReference*> >("VectorRef")
        .def(vector_indexing_suite< std::vector<IReference*> >() );
	class_< std::vector<IMessageBox*> >("VectorMB")
        .def(vector_indexing_suite< std::vector<IMessageBox*> >() );
	class_< std::vector<IFunctionalMockupUnit*> >("VectorFMU")
        .def(vector_indexing_suite< std::vector<IFunctionalMockupUnit*> >() );

	class_<PyArray<double>, boost::noncopyable>("ArrayDouble", init<long>())
		.def(init<double,long>())
		.def("__getitem__", &PyArray<double>::slice, return_value_policy<manage_new_object>())
		.def("__getitem__", &PyArray<double>::item)
		.def("__setitem__", &PyArray<double>::setItem)
		.def("__len__", &PyArray<double>::size);

	class_<CSysNamedComp,boost::noncopyable>("CSysNamedComp",no_init)
		.def("basename",&CSysNamedComp::basename)
		.def("name",&CSysNamedComp::name)
	;

	class_<CNamedComp,bases<CSysNamedComp>,boost::noncopyable>("CNamedComp",no_init)
		.def("parent",&CNamedComp::parent,return_value_policy<reference_existing_object>())
		.def("description",&CNamedComp::description)
		.def("setDescription",&CNamedComp::setDescription)
	;

	/*Export de l'interface IVarBase*/
	class_<IVarBase,bases<CNamedComp>,boost::noncopyable>("IVarBase",no_init)
		.def("setCnctMax",&IVarBase::setCnctMax)
		.def("cnctMax",&IVarBase::cnctMax)
		.def("addSensitiveMethod", static_cast<void(*)(IVarBase*, char const*)>(IVarBase_addSensitiveMethod))
		.def("addSensitiveMethod", static_cast<void(*)(IVarBase*, char const*, PyObject*)>(IVarBase_addSensitiveMethod))
		.def("addSensitiveMethod", static_cast<void(*)(IVarBase*, char const*, PyObject*, int)>(IVarBase_addSensitiveMethod))
		.def("addCallback", static_cast<void(*)(IVarBase*, PyObject*)>(IVarBase_addCallback))
		.def("addCallback", static_cast<void(*)(IVarBase*, PyObject*, PyObject*)>(IVarBase_addCallback))
		.def("addCallback", static_cast<void(*)(IVarBase*, char const*, PyObject*, int)>(IVarBase_addCallback))
		.def("removeSensitiveMethod",&IVarBase::removeSensitiveMethod);

	/*Export de l'interface CMonitored utile pour l'analyseur*/
	class_<IMonitored,IMonitored*,IMonitored const*,boost::noncopyable>("IMonitored",no_init)
		.def("basename",&IMonitored::basename)
		.def("name",&IMonitored::name)
		.def("parent",&CNamedComp::parent,return_value_policy<reference_existing_object>())
		.def("setTrace",&IMonitored::setTrace)
		.def("trace",&IMonitored::trace)
		.def("isMonitored",&IMonitored::isMonitored)
		.def("isAlwaysMonitored",&IMonitored::isAlwaysMonitored)
		.def("setAlwaysMonitored",&IMonitored::setAlwaysMonitored)
		.def("setMonitor",&IMonitored::setMonitor)
		.def("type",&IMonitored::type);

	register_ptr_to_python<boost::shared_ptr<IVarBase> >();

	/*Export de l'interface IReference correspondant aux rfrences.*/
	class_<IReference,IReference*,bases<IVarBase>,boost::noncopyable>("IReference",no_init)
		.def("nbCnx",&IReference::cnctCount)
		.def("cnctCount",&IReference::cnctCount)
		.def("isValue",&IReference::isValue)
		.def("cValue", static_cast<std::complex<double>(IReference::*)(CNX_ID)const>(&IReference::cValue))
		.def("dValue", static_cast<double(IReference::*)(CNX_ID)const>(&IReference::dValue))
		.def("fValue", static_cast<float(IReference::*)(CNX_ID)const>(&IReference::fValue))
		.def("iValue", static_cast<int(IReference::*)(CNX_ID)const>(&IReference::iValue))
		.def("bValue", static_cast<bool(IReference::*)(CNX_ID)const>(&IReference::bValue))
		.def("sValue", static_cast<std::string(IReference::*)(CNX_ID)const>(&IReference::sValue))
		.def("cValue", static_cast<std::complex<double>(IReference::*)(CNX_ID,std::vector<int>const&)const>(&IReference::cValue))
		.def("dValue", static_cast<double(IReference::*)(CNX_ID, std::vector<int>const&)const>(&IReference::dValue))
		.def("fValue", static_cast<float(IReference::*)(CNX_ID, std::vector<int>const&)const>(&IReference::fValue))
		.def("iValue", static_cast<int(IReference::*)(CNX_ID, std::vector<int>const&)const>(&IReference::iValue))
		.def("bValue", static_cast<bool(IReference::*)(CNX_ID, std::vector<int>const&)const>(&IReference::bValue))
		.def("sValue", static_cast<std::string(IReference::*)(CNX_ID, std::vector<int>const&)const>(&IReference::sValue))
		.def("value", static_cast<boost::python::api::object(*)(IReference*, CNX_ID)>(IVarRef_value))
		.def("value", static_cast<boost::python::api::object(*)(IReference*, CNX_ID, std::vector<int>const&)>(IVarRef_value))
		.def("reference",&IReference::reference,return_value_policy<reference_existing_object>())
		.def("variable",&IReference::variable,return_value_policy<reference_existing_object>())
		.def("bind",&IReference::bind)
		.def("unbind",static_cast<void(IReference::*)(IVarBase&)>(&IReference::unbind))
		.def("unbind",static_cast<void(IReference::*)(CNX_ID)>(&IReference::unbind))
		.def("takeResource",&IReference::takeResource)
		.def("restoreResource",&IReference::restoreResource)
		.def("sumValue",&IReference::sumValue)
		.def("productValue",&IReference::productValue)
		.def("orValue",&IReference::orValue)
		.def("andValue",&IReference::andValue)
		.def("sumValue",IReference_sumValue)
		.def("productValue",IReference_productValue)
		.def("orValue",IReference_orValue)
		.def("andValue",IReference_andValue)
		.def("valuesThat",&IReference_valuesThat)
		;

	/*Export de l'interface IVariable correspondant aux variables.*/
	class_<IVariable, IVariable*, bases<IMonitored, IVarBase>, boost::noncopyable>("IVariable", no_init)
		.def("setCausality", &IVariable::setCausality)
		.def("causality", &IVariable::causality)
		.def("setVariability", &IVariable::setVariability)
		.def("variability", &IVariable::variability)
		.def("setInitializability", &IVariable::setInitializability)
		.def("initializability", &IVariable::initializability)
		.def("valueType", &IVariable::variableType)
		.def("variableType", &IVariable::variableType)
		.def("isArray", &IVariable::isArray)
		.def("dimensions", &IVariable::dimensions)
		.def("setDimensions", &IVariable::setDimensions)
		.def("cValue", static_cast<std::complex<double>(IVariable::*)()const>(&IVariable::cValue))
		.def("dValue", static_cast<double(IVariable::*)()const>(&IVariable::dValue))
		.def("fValue", static_cast<float(IVariable::*)()const>(&IVariable::fValue))
		.def("iValue", static_cast<int(IVariable::*)()const>(&IVariable::iValue))
		.def("bValue", static_cast<bool(IVariable::*)()const>(&IVariable::bValue))
		.def("sValue", static_cast<std::string(IVariable::*)()const>(&IVariable::sValue))
		.def("cValue", static_cast<std::complex<double>(IVariable::*)(std::vector<int>const&)const>(&IVariable::cValue))
		.def("dValue", static_cast<double(IVariable::*)(std::vector<int>const&)const>(&IVariable::dValue))
		.def("fValue", static_cast<float(IVariable::*)(std::vector<int>const&)const>(&IVariable::fValue))
		.def("iValue", static_cast<int(IVariable::*)(std::vector<int>const&)const>(&IVariable::iValue))
		.def("bValue", static_cast<bool(IVariable::*)(std::vector<int>const&)const>(&IVariable::bValue))
		.def("sValue", static_cast<std::string(IVariable::*)(std::vector<int>const&)const>(&IVariable::sValue))
		.def("cValues", &IVariable::cValues)
		.def("dValues", &IVariable::dValues)
		.def("fValues", &IVariable::fValues)
		.def("iValues", &IVariable::iValues)
		.def("bValues", &IVariable::bValues)
		.def("sValues", &IVariable::sValues)
		.def("value", static_cast<boost::python::api::object(*)(IVariable*)>(IVariable_value))
		.def("value", static_cast<boost::python::api::object(*)(IVariable*,std::vector<int>const&)>(IVariable_value))
		.def("cInitValue", static_cast<std::complex<double>(IVariable::*)(void)const>(&IVariable::cInitValue))
		.def("dInitValue", static_cast<double(IVariable::*)(void)const>(&IVariable::dInitValue))
		.def("fInitValue", static_cast<float(IVariable::*)(void)const>(&IVariable::fInitValue))
		.def("iInitValue", static_cast<int(IVariable::*)(void)const>(&IVariable::iInitValue))
		.def("bInitValue", static_cast<bool(IVariable::*)(void)const>(&IVariable::bInitValue))
		.def("sInitValue", static_cast<std::string(IVariable::*)(void)const>(&IVariable::sInitValue))
		.def("cInitValue", static_cast<std::complex<double>(IVariable::*)(std::vector<int>const&)const>(&IVariable::cInitValue))
		.def("dInitValue", static_cast<double(IVariable::*)(std::vector<int>const&)const>(&IVariable::dInitValue))
		.def("fInitValue", static_cast<float(IVariable::*)(std::vector<int>const&)const>(&IVariable::fInitValue))
		.def("iInitValue", static_cast<int(IVariable::*)(std::vector<int>const&)const>(&IVariable::iInitValue))
		.def("bInitValue", static_cast<bool(IVariable::*)(std::vector<int>const&)const>(&IVariable::bInitValue))
		.def("sInitValue", static_cast<std::string(IVariable::*)(std::vector<int>const&)const>(&IVariable::sInitValue))
		.def("cInitValue",static_cast<std::complex<double>(IVariable::*)(PyTime)const>(&IVariable::cInitValue))
		.def("dInitValue",static_cast<double(IVariable::*)(PyTime)const>(&IVariable::dInitValue))
		.def("fInitValue",static_cast<float(IVariable::*)(PyTime)const>(&IVariable::fInitValue))
		.def("iInitValue",static_cast<int(IVariable::*)(PyTime)const>(&IVariable::iInitValue))
		.def("bInitValue",static_cast<bool(IVariable::*)(PyTime)const>(&IVariable::bInitValue))
		.def("initValue", static_cast<boost::python::api::object(*)(IVariable*)>(IVariable_initValue))
		.def("initValue", static_cast<boost::python::api::object(*)(IVariable*, std::vector<int>const&)>(IVariable_initValue))
		.def("dvdtODE",&IVariable::dvdtODE)
		.def("setValue", static_cast<void(IVariable::*)(std::complex<double>const&)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(double)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(float)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(bool)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(int)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(char const*)>(&IVariable::setValue))
		.def("setCValue", static_cast<void(IVariable::*)(std::complex<double>const&)>(&IVariable::setValue))
		.def("setDValue", static_cast<void(IVariable::*)(double)>(&IVariable::setValue))
		.def("setFValue", static_cast<void(IVariable::*)(float)>(&IVariable::setValue))
		.def("setIValue", static_cast<void(IVariable::*)(int)>(&IVariable::setValue))
		.def("setBValue", static_cast<void(IVariable::*)(bool)>(&IVariable::setValue))
		.def("setSValue", static_cast<void(IVariable::*)(char const*)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(std::vector<int>const&, std::complex<double>const&)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(std::vector<int>const&, double)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(std::vector<int>const&, float)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(std::vector<int>const&, bool)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(std::vector<int>const&, int)>(&IVariable::setValue))
		.def("setValue", static_cast<void(IVariable::*)(std::vector<int>const&, char const*)>(&IVariable::setValue))
		.def("setCValue", static_cast<void(IVariable::*)(std::vector<int>const&, std::complex<double>const&)>(&IVariable::setValue))
		.def("setDValue", static_cast<void(IVariable::*)(std::vector<int>const&, double)>(&IVariable::setValue))
		.def("setFValue", static_cast<void(IVariable::*)(std::vector<int>const&, float)>(&IVariable::setValue))
		.def("setIValue", static_cast<void(IVariable::*)(std::vector<int>const&, int)>(&IVariable::setValue))
		.def("setBValue", static_cast<void(IVariable::*)(std::vector<int>const&, bool)>(&IVariable::setValue))
		.def("setSValue", static_cast<void(IVariable::*)(std::vector<int>const&, char const*)>(&IVariable::setValue))
		.def("setCValues", &IVariable::setCValues)
		.def("setDValues", &IVariable::setDValues)
		.def("setFValues", &IVariable::setFValues)
		.def("setIValues", &IVariable::setIValues)
		.def("setBValues", &IVariable::setBValues)
		.def("setSValues", &IVariable::setSValues)
		.def("setValue",static_cast<void(IVariable::*)(PyTime,std::complex<double>const&)>(&IVariable::setValue))
		.def("setValue",static_cast<void(IVariable::*)(PyTime,double)>(&IVariable::setValue))
		.def("setValue",static_cast<void(IVariable::*)(PyTime,float)>(&IVariable::setValue))
		.def("setValue",static_cast<void(IVariable::*)(PyTime,bool)>(&IVariable::setValue))
		.def("setValue",static_cast<void(IVariable::*)(PyTime,int)>(&IVariable::setValue))
		.def("setCValue",static_cast<void(IVariable::*)(PyTime,std::complex<double>const&)>(&IVariable::setValue))
		.def("setDValue",static_cast<void(IVariable::*)(PyTime,double)>(&IVariable::setValue))
		.def("setFValue",static_cast<void(IVariable::*)(PyTime,float)>(&IVariable::setValue))
		.def("setIValue",static_cast<void(IVariable::*)(PyTime,int)>(&IVariable::setValue))
		.def("setBValue",static_cast<void(IVariable::*)(PyTime,bool)>(&IVariable::setValue))
		.def("clearInitValues",&IVariable::clearInitValues)
		.def("setDvdtODE",&IVariable::setDvdtODE)
		.def("setResidualValue",&IVariable::setResidualValue)
		.def("setPartial",&IVariable::setPartial)
		.def("setPartialP",&IVariable::setPartialP)
		.def("setModifiable",&IVariable::setModifiable)
		.def("setInterpolated",&IVariable::setInterpolated)
		.def("isInterpolated",&IVariable::isInterpolated)
		.def("integrate",static_cast<void(IVariable::*)(IVariable*)>(&IVariable::integrate))
		.def("setReinitialized",&IVariable::setReinitialized)
		.def("derivative",&IVariable::derivative)
		.def("dtDerivative",&IVariable::dtDerivative)
		.def("setUncertainty",static_cast<void(IVariable::*)(TDistributionType::EDistributionType,std::vector<double>)>(&IVariable::setUncertainty))
		.def("setUncertaintyLimits",&IVariable::setUncertaintyLimits)
		.def("uncertaintyDistribution",&IVariable::uncertaintyDistribution)
		.def("uncertaintyParameter",&IVariable::uncertaintyParameter)
		.def("uncertaintyParametersCount",&IVariable::uncertaintyParametersCount)
		.def("uncertaintyMin",&IVariable::uncertaintyMin)
		.def("uncertaintyMax",&IVariable::uncertaintyMax)
		.def("setDelay",&IVariable::setDelay)
		.def("delay",&IVariable::delay)
		;

	/*Export de l'interface ICnctInfo.*/
	class_<ICnctInfo,ICnctInfo*,boost::noncopyable>("ICnctInfo",no_init)
		.def("bValue",&ICnctInfo::bValue)
		.def("iValue",&ICnctInfo::iValue)
		.def("fValue",&ICnctInfo::fValue)
		.def("dValue",&ICnctInfo::dValue)
		.def("cValue",&ICnctInfo::cValue)
		.def("sValue",&ICnctInfo::sValue)
		.def("setValue",static_cast<void(ICnctInfo::*)(char const*,std::complex<double>const&)>(&ICnctInfo::setValue))
		.def("setValue",static_cast<void(ICnctInfo::*)(char const*,double)>(&ICnctInfo::setValue))
		.def("setValue",static_cast<void(ICnctInfo::*)(char const*,float)>(&ICnctInfo::setValue))
		.def("setValue",static_cast<void(ICnctInfo::*)(char const*,bool)>(&ICnctInfo::setValue))
		.def("setValue",static_cast<void(ICnctInfo::*)(char const*,int)>(&ICnctInfo::setValue))
		.def("setValue",static_cast<void(ICnctInfo::*)(char const*,char const*)>(&ICnctInfo::setValue))
		.def("setCValue",static_cast<void(ICnctInfo::*)(char const*,std::complex<double>const&)>(&ICnctInfo::setValue))
		.def("setDValue",static_cast<void(ICnctInfo::*)(char const*,double)>(&ICnctInfo::setValue))
		.def("setFValue",static_cast<void(ICnctInfo::*)(char const*,float)>(&ICnctInfo::setValue))
		.def("setIValue",static_cast<void(ICnctInfo::*)(char const*,int)>(&ICnctInfo::setValue))
		.def("setBValue",static_cast<void(ICnctInfo::*)(char const*,bool)>(&ICnctInfo::setValue))
		.def("setSValue",static_cast<void(ICnctInfo::*)(char const*,char const*)>(&ICnctInfo::setValue))
		.def("delValue",&ICnctInfo::delValue)
		.def("valueType", &ICnctInfo::variableType)
		.def("variableType", &ICnctInfo::variableType)
		;

	/*Export de l'interface IMessageBox correspondant aux botes de messages.*/
	class_<IMessageBox,IMessageBox*,bases<CNamedComp>,boost::noncopyable>("IMessageBox",no_init)
		.def("addExport",static_cast<void(IMessageBox::*)(IVarBase&,char const*)>(&IMessageBox::addExport))
		.def("exportedVariable",&IMessageBox::exportedVariable,return_value_policy<reference_existing_object>())
		.def("addImport",static_cast<void(IMessageBox::*)(IReference*,char const*)>(&IMessageBox::addImport))
		.def("addOptionalImport",static_cast<void(IMessageBox::*)(IReference&,int,double,char const*)>(&IMessageBox::addOptionalImport))
		.def("connectTo",static_cast<ICnctInfo&(IMessageBox::*)(IMessageBox&,double)>(&IMessageBox::connectTo),return_value_policy<reference_existing_object>())
		.def("connectTo",static_cast<ICnctInfo&(IMessageBox::*)(IMessageBox&)>(&IMessageBox::connectTo),return_value_policy<reference_existing_object>())
		.def("disconnectFrom",&IMessageBox::disconnectFrom)
		.def("cnctInfo",static_cast<ICnctInfo&(IMessageBox::*)(CNX_ID)const>(&IMessageBox::cnctInfo),return_value_policy<reference_existing_object>())
		.def("cnctInfo",static_cast<ICnctInfo&(IMessageBox::*)(IMessageBox&)const>(&IMessageBox::cnctInfo),return_value_policy<reference_existing_object>())
		.def("cnctCount",&IMessageBox::cnctCount)
		.def("component",&IMessageBox::component,return_value_policy<reference_existing_object>())
		.def("components",&IMessageBox::components)
		.def("cnct",&IMessageBox::cnct,return_value_policy<reference_existing_object>())
		.def("weight",&IMessageBox::weight)
		.def("cValue",&IMessageBox::cValue)
		.def("dValue",&IMessageBox::dValue)
		.def("fValue",&IMessageBox::fValue)
		.def("iValue",&IMessageBox::iValue)
		.def("bValue",&IMessageBox::bValue)
		.def("sValue",&IMessageBox::sValue)
		;

	/*Export de l'interface IDistLaw*/
	class_<IDistLaw,PyDistLaw,boost::noncopyable>("IDistLaw",init<CNamedComp&>())
		.def("name",&IDistLaw::name)
		.def("upToDate",&IDistLaw::upToDate)
		.def("delay",static_cast<PyTime(IDistLaw::*)()const>(&IDistLaw::delay))
		.def("delay",static_cast<PyTime(IDistLaw::*)(unsigned int)const>(&IDistLaw::delay))
		.def("delayBeyond",&IDistLaw::delayBeyond)
		.def("density",&IDistLaw::density)
		.def("distribution", &IDistLaw::distribution)
		.def("invDistribution", &IDistLaw::invDistribution)
		.def("index",&IDistLaw::index)
		.def("nbIndex",&IDistLaw::nbIndex)
		.def("nbParam",&IDistLaw::nbParam)
		.def("setParameter",static_cast<void(*)(IDistLaw*,char const*,PyObject*,LAWP_ID)>(IDistLaw_setParameter))
		.def("setParameter",static_cast<void(*)(IDistLaw*,PyObject*)>(IDistLaw_setParameter))
		.def("setParameter",static_cast<void(*)(IDistLaw*,PyObject*,LAWP_ID)>(IDistLaw_setParameter))
		.def("setParameter",static_cast<void(*)(IDistLaw*,char const*,PyObject*)>(IDistLaw_setParameter))
		.def("insertParameter",static_cast<void(IDistLaw::*)(IVariable&,LAWP_ID)>(&IDistLaw::insertParameter))
		.def("insertParameter",static_cast<void(IDistLaw::*)(double,LAWP_ID)>(&IDistLaw::insertParameter))
		.def("insertParameter",static_cast<void(*)(IDistLaw*,IVariable&)>(IDistLaw_insertParameter))
		.def("insertParameter",static_cast<void(*)(IDistLaw*,double)>(IDistLaw_insertParameter))
		.def("parameter",&IDistLaw::parameter)
		.def("setNbSamples",&IDistLaw::setNbSamples)
		.def("nbSamples",&IDistLaw::nbSamples)
		.def("type",&IDistLaw::type)
		.def("newLaw",static_cast<IDistLaw*(*)(CNamedComp&,TLawType::ELawType,PyObject*)>(IDistLaw_newLaw),return_value_policy<reference_existing_object>())
		.def("newLaw",static_cast<IDistLaw*(*)(CNamedComp&,TLawType::ELawType,char const*,PyObject*)>(IDistLaw_newLaw),return_value_policy<reference_existing_object>()).staticmethod("newLaw")
		;

	/*Export de l'interface ITransition.*/
	class_<ITransition,ITransition*,bases<IMonitored, CNamedComp>,boost::noncopyable>("ITransition",no_init)
		.def("setInterruptible",&ITransition::setInterruptible)
		.def("interruptible",&ITransition::interruptible)
		.def("setModifiable",&ITransition::setModifiable)
		.def("modifiable",&ITransition::modifiable)
		.def("startState",&ITransition::startState,return_value_policy<reference_existing_object>())
		.def("addTarget",static_cast<void(ITransition::*)(IState*,int)>(&ITransition::addTarget))
		.def("addTarget",&ITransition_addTarget)
		.def("insertTarget",static_cast<void(ITransition::*)(IState&,int,int)>(&ITransition::insertTarget))
		.def("targetCount",&ITransition::targetCount)
		.def("getTarget",&ITransition::getTarget,return_value_policy<reference_existing_object>())
		.def("target",&ITransition::target,return_value_policy<reference_existing_object>())
		.def("getTargetType",&ITransition::getTargetType)
		.def("targetType",&ITransition::targetType)
		.def("setCondition",static_cast<void(*)(ITransition*,char const*,PyObject*,bool)>(ITransition_setCondition))
		.def("setCondition",static_cast<void(*)(ITransition*,char const*,PyObject*)>(ITransition_setCondition))
		.def("setCondition",static_cast<void(*)(ITransition*,PyObject*,bool)>(ITransition_setCondition))
		.def("setCondition",static_cast<void(*)(ITransition*,PyObject*)>(ITransition_setCondition))
		.def("getCondition",&ITransition::getCondition)
		.def("conditionValue",&ITransition::conditionValue)
		.def("setDistLaw",static_cast<IDistLaw*(*)(ITransition*,TLawType::ELawType,PyObject*)>(ITransition_setDistLaw),return_value_policy<reference_existing_object>())
		.def("setDistLaw",static_cast<void(ITransition::*)(IDistLaw*)>(&ITransition::setDistLaw),return_internal_reference<1,with_custodian_and_ward<1,2> >())
		.def("setDistLaw",static_cast<IDistLaw*(*)(ITransition*,TLawType::ELawType,char const*name,PyObject*)>(ITransition_setDistLaw),return_value_policy<reference_existing_object>())
		.def("distLaw",&ITransition::distLaw,return_value_policy<reference_existing_object>())
		.def("addSensitiveMethod", static_cast<void(*)(ITransition* self, char const*)>(ITransition_addSensitiveMethod))
		.def("addSensitiveMethod", static_cast<void(*)(ITransition* self, char const*, PyObject*)>(ITransition_addSensitiveMethod))
		.def("addSensitiveMethod", static_cast<void(*)(ITransition* self, char const*, PyObject*, int)>(ITransition_addSensitiveMethod))
		.def("addCallback", static_cast<void(*)(ITransition* self, PyObject*)>(ITransition_addCallback))
		.def("addCallback", static_cast<void(*)(ITransition* self, PyObject*, PyObject*)>(ITransition_addCallback))
		.def("addCallback", static_cast<void(*)(ITransition* self, char const*, PyObject*, int)>(ITransition_addCallback))
		.def("removeSensitiveMethod", &ITransition::removeSensitiveMethod)
		.def("removeCallback", &ITransition::removeCallback)
		.def("inhibateTarget",static_cast<void(ITransition::*)(IState*,bool)>(&ITransition::inhibateTarget))
		.def("inhibateTarget",static_cast<void(ITransition::*)(int,bool)>(&ITransition::inhibateTarget))
		.def("firedState",&ITransition::firedState)
		.def("startTime",&ITransition::startTime)
		.def("endTime",&ITransition::endTime)
		.def("setDelay",&ITransition::setDelay)
		.def("indOutState",&ITransition::indOutState)
		.def("setIndOutState",&ITransition::setIndOutState)
		.def("invalidate",&ITransition::invalidate)
		.def("setMonitoredOutStateMask",&ITransition::setMonitoredOutStateMask)
		.def("monitoredOutStateMask",&ITransition::monitoredOutStateMask)
		.def("setConditionFT",&ITransition::setConditionFT)
		.def("conditionFT",&ITransition::conditionFT)
		.def("setBEType",&ITransition::setBEType)
		.def("BEType",&ITransition::BEType)
		.def("setBEParameter",static_cast<void(ITransition::*)(double,BEP_ID)>(&ITransition::setBEParameter))
		.def("setBEParameter",static_cast<void(ITransition::*)(IVariable*,BEP_ID)>(&ITransition::setBEParameter))
		.def("setBEParameter",static_cast<void(*)(ITransition*,char const*,PyObject*,BEP_ID)>(ITransition_setBEParameter))
		.def("setBEParameter",static_cast<void(*)(ITransition*,PyObject*,BEP_ID)>(ITransition_setBEParameter))
		.def("BEParameterValue",&ITransition::BEParameterValue)
		.def("setDefaultStrategy",&ITransition::setDefaultStrategy)
		.def("defaultStrategy",&ITransition::defaultStrategy)
		.def("setStrategy",&ITransition::setStrategy)
		.def("strategy",&ITransition::strategy)
		;

	/*Export de l'interface IState.*/
	class_<IState,IState*,bases<IVariable>,boost::noncopyable>("IState",no_init)
		.def("addTransition",&IState::addTransition,return_value_policy<reference_existing_object>())
		.def("isActive",&IState::isActive)
		.def("index",&IState::index)
		.def("exitTime",&IState::exitTime)
		.def("automaton",&IState::automaton,return_value_policy<reference_existing_object>());

	/*Export de l'interface IAutomaton.*/
	class_<IAutomaton,IAutomaton*,bases<IVariable>,boost::noncopyable>("IAutomaton",no_init)
		.def("addState",&IAutomaton::addState,return_value_policy<reference_existing_object>())
		.def("initState",&IAutomaton::initState,return_value_policy<reference_existing_object>())
		.def("states",&IAutomaton::states)
		.def("setInitState",static_cast<void(IAutomaton::*)(IState&)>(&IAutomaton::setInitState))
		.def("currentIndex",&IAutomaton::currentIndex)
		.def("currentState",&IAutomaton::currentState,return_value_policy<reference_existing_object>())
		.def("state",&IAutomaton::state,return_value_policy<reference_existing_object>())
		.def("writeGraph",&IAutomaton::writeGraph)
		;

	/*Export de l'interface IPDMPManager.*/
	class_<IPDMPManager,bases<CNamedComp>,boost::noncopyable>("IPDMPManager",no_init)
		.def("setDt",&IPDMPManager::setDt)
		.def("getDt",&IPDMPManager::getDt)
		.def("dt",&IPDMPManager::dt)
		.def("setDtMin",&IPDMPManager::setDtMin)
		.def("dtMin",&IPDMPManager::dtMin)
		.def("setDtMax",&IPDMPManager::setDtMax)
		.def("dtMax",&IPDMPManager::dtMax)
		.def("setDtCond",&IPDMPManager::setDtCond)
		.def("dtCond",&IPDMPManager::dtCond)
		.def("setDtMem",&IPDMPManager::setDtMem)
		.def("dtMem",&IPDMPManager::dtMem)
		.def("setAlwaysMem",&IPDMPManager::setAlwaysMem)
		.def("alwaysMem",&IPDMPManager::alwaysMem)
		.def("setUseJacobian",&IPDMPManager::setUseJacobian)
		.def("useJacobian",&IPDMPManager::useJacobian)
		.def("parameter",&IPDMPManager::parameter)
		.def("setParameter",&IPDMPManager::setParameter)
		.def("addODEVariable",static_cast<void(IPDMPManager::*)(IVariable&)>(&IPDMPManager::addODEVariable))
		.def("addDifferentialVariable",static_cast<void(IPDMPManager::*)(IVariable&)>(&IPDMPManager::addDifferentialVariable))
		.def("addAlgebraicVariable",static_cast<void(IPDMPManager::*)(IVariable&)>(&IPDMPManager::addAlgebraicVariable))
		.def("addExplicitVariable",static_cast<void(IPDMPManager::*)(IVariable&)>(&IPDMPManager::addExplicitVariable))
		.def("addBeginMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&)>(IPDMPManager_addBeginMethod))
		.def("addFMU",static_cast<void(IPDMPManager::*)(IFunctionalMockupUnit&)>(&IPDMPManager::addFMU))
		.def("addBeginMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*)>(IPDMPManager_addBeginMethod))
		.def("addEndMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&)>(IPDMPManager_addEndMethod))
		.def("addEndMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*)>(IPDMPManager_addEndMethod))
		.def("addPreStepMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&)>(IPDMPManager_addPreStepMethod))
		.def("addPreStepMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*)>(IPDMPManager_addPreStepMethod))
		.def("addPostStepMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&)>(IPDMPManager_addPostStepMethod))
		.def("addPostStepMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*)>(IPDMPManager_addPostStepMethod))
		.def("addEquationMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,int)>(IPDMPManager_addEquationMethod))
		.def("addEquationMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*,int)>(IPDMPManager_addEquationMethod))
		.def("addEquationMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&)>(IPDMPManager_addEquationMethod))
		.def("addEquationMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*)>(IPDMPManager_addEquationMethod))
		.def("addWatchedTransition",static_cast<void(IPDMPManager::*)(ITransition&)>(&IPDMPManager::addWatchedTransition))
		.def("addCondition",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&)>(IPDMPManager_addCondition))
		.def("addCondition",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*)>(IPDMPManager_addCondition))
		.def("addBoundaryCheckerMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&)>(IPDMPManager_addCondition))
		.def("addBoundaryCheckerMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*)>(IPDMPManager_addCondition))
		.def("addJacobianMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,int)>(IPDMPManager_addJacobianMethod))
		.def("addJacobianMethod",static_cast<void(*)(IPDMPManager*,char const*,CSysNamedComp&,PyObject*,int)>(IPDMPManager_addJacobianMethod))
		.def("setVariableConstraint",static_cast<void(IPDMPManager::*)(IVariable&,TVarConstraintType::EVarConstraintType)>(&IPDMPManager::setVariableConstraint))
		.def("setSchema",static_cast<void(IPDMPManager::*)(TSchemaType::ESchemaType)>(&IPDMPManager::setSchema))
		.def("schema",&IPDMPManager::schema)
		.def("setTrace",&IPDMPManager::setTrace);

	/*Export de l'interface IEquation*/
	class_<IEquation,boost::noncopyable>("IEquation",no_init)
		.def("setCoefficient",static_cast<void(IEquation::*)(IVariable&,double,double)>(&IEquation::setCoefficient))
		.def("setCoefficient",IEquation_setCoefficient)
		.def("setConstant",&IEquation::setConstant)
		.def("setConstant",IEquation_setConstant);

	/*Export de l'interface ISLEManager.*/
	class_<ISLEManager,bases<CNamedComp>,boost::noncopyable>("ISLEManager",no_init)
		.def("addEquation",static_cast<void(*)(ISLEManager*,char const*,CSysNamedComp&,PyObject*)>(ISLEManager_addEquation))
		.def("addEquation",static_cast<void(*)(ISLEManager*,char const*,CSysNamedComp&)>(ISLEManager_addEquation))
		.def("addVariable",static_cast<void(ISLEManager::*)(IVariable&)>(&ISLEManager::addVariable))
		.def("print",&ISLEManager::print)
		.def("solve",&ISLEManager::solve);

	/*Export de l'interface IInequality*/
	class_<IInequality,boost::noncopyable>("IInequality",no_init)
		.def("setCoefficient",static_cast<void(IInequality::*)(IVariable&,double)>(&IInequality::setCoefficient))
		.def("setLimits",&IInequality::setLimits)
		.def("setMax",&IInequality::setMax)
		.def("setMin",&IInequality::setMin);

	/*Export de l'interface IMILPManager.*/
	class_<IMILPManager,bases<CNamedComp>,boost::noncopyable>("IMILPManager",no_init)
		.def("addInequality",static_cast<void(*)(IMILPManager*,char const*,CSysNamedComp&,PyObject*)>(IMILPManager_addInequality))
		.def("addEquation",static_cast<void(*)(IMILPManager*,char const*,CSysNamedComp&)>(IMILPManager_addInequality))
		.def("addVariable",static_cast<void(IMILPManager::*)(IVariable&,double)>(&IMILPManager::addVariable))
		.def("addVariable",static_cast<void(IMILPManager::*)(IVariable&)>(&IMILPManager::addVariable))
		.def("setVariableLimits",static_cast<void(IMILPManager::*)(IVariable&,double,double)>(&IMILPManager::setVariableLimits))
		.def("setVariableMin",static_cast<void(IMILPManager::*)(IVariable&,double)>(&IMILPManager::setVariableMin))
		.def("setVariableMax",static_cast<void(IMILPManager::*)(IVariable&,double)>(&IMILPManager::setVariableMax))
		.def("setVariableObjCoef",static_cast<void(IMILPManager::*)(IVariable&,double)>(&IMILPManager::setVariableObjCoef))
		.def("setObjectiveVar",static_cast<void(IMILPManager::*)(IVariable*)>(&IMILPManager::setObjectiveVar))
		.def("setMaximize",&IMILPManager::setMaximize)
		.def("setAlgorithm",&IMILPManager::setAlgorithm)
		.def("setMsgOptions",&IMILPManager::setMsgOptions)
		.def("setScaling",&IMILPManager::setScaling)
		.def("setOptions",&IMILPManager::setOptions)
		.def("setOption",&IMILPManager::setOption)
		.def("setTimeLimit",&IMILPManager::setTimeLimit)
		.def("setIterLimit",&IMILPManager::setIterLimit)
		.def("addOptions",&IMILPManager::addOptions)
		.def("delOptions",&IMILPManager::delOptions)
		.def("setSimplexOpt",&IMILPManager::setSimplexOpt)
		.def("setIntPointOpt",&IMILPManager::setIntPointOpt)
		.def("setBacktracking",&IMILPManager::setBacktracking)
		.def("setBranching",&IMILPManager::setBranching)
		.def("setPreprocessing",&IMILPManager::setPreprocessing)
		.def("objectiveValue",&IMILPManager::objectiveValue)
		.def("print",&IMILPManager::print)
		.def("solve",&IMILPManager::solve);

	class_<INetCDFManager,boost::noncopyable>("INetCDFManager",no_init)
		.def("newManager",INetCDFManager::newManager,return_value_policy<manage_new_object>()).staticmethod("newManager")
		.def("dimensions",&INetCDFManager::dimensions)
		.def("dRead1D",&INetCDFManager::dRead1D)
		.def("fRead1D",&INetCDFManager::fRead1D);

	/*Export de la classe IFunctionalMockupUnit.*/
	class_<IFunctionalMockupUnit,IFunctionalMockupUnit*,bases<CNamedComp>,boost::noncopyable>("IFunctionalMockupUnit",no_init)
		.def("modelName",&IFunctionalMockupUnit::modelName)
		.def("hasVariable",&IFunctionalMockupUnit::hasVariable)
		.def("variable",&IFunctionalMockupUnit::variable,return_value_policy<reference_existing_object>())
		.def("renamedVariable",&IFunctionalMockupUnit::renamedVariable,return_value_policy<reference_existing_object>())
		.def("variables",static_cast<std::vector<IVariable*>(IFunctionalMockupUnit::*)()const>(&IFunctionalMockupUnit::variables))
		.def("variables",static_cast<std::vector<IVariable*>(IFunctionalMockupUnit::*)(char const*,TVarType::EVarType*)const>(&IFunctionalMockupUnit::variables))
		.def("setTimeRatio",&IFunctionalMockupUnit::setTimeRatio)
		.def("setTrace",&IFunctionalMockupUnit::setTrace)
		.def("trace",&IFunctionalMockupUnit::trace)
		;

	/*Export de la classe CComponent sans possibilit de modification.*/
	class_<CComponent,CComponent*,bases<CNamedComp>,boost::noncopyable>("IComponent", no_init)
		.def("getMessageBox",&CComponent::getMessageBox,return_value_policy<reference_existing_object>())
		.def("getPDMPManager",&CComponent::getPDMPManager,return_value_policy<reference_existing_object>())
		.def("getMILPManager",&CComponent::getMILPManager,return_value_policy<reference_existing_object>())
		.def("getSLEManager",&CComponent::getSLEManager,return_value_policy<reference_existing_object>())
		.def("getAutomaton",&CComponent::getAutomaton,return_value_policy<reference_existing_object>())
		.def("getAutomata",&CComponent::getAutomata)
		.def("getComponent",&CComponent::getComponent,return_value_policy<reference_existing_object>())
		.def("getComponents",&CComponent::getComponents)
		.def("getReference",&CComponent::getReference,return_value_policy<reference_existing_object>())
		.def("getReferences",&CComponent::getReferences)
		.def("getVariable",&CComponent::getVariable,return_value_policy<reference_existing_object>())
		.def("getVariables",&CComponent::getVariables)
		.def("getState",&CComponent::getState,return_value_policy<reference_existing_object>())
		.def("getStates",&CComponent::getStates)
		.def("getTransition",&CComponent::getTransition,return_value_policy<reference_existing_object>())
		.def("messageBox",&CComponent::messageBox,return_value_policy<reference_existing_object>())
		.def("messageBoxes",&CComponent::messageBoxes)
		.def("PDMPManager",&CComponent::PDMPManager,return_value_policy<reference_existing_object>())
		.def("MILPManager", &CComponent::MILPManager, return_value_policy<reference_existing_object>())
		.def("SLEManager",&CComponent::SLEManager,return_value_policy<reference_existing_object>())
		.def("element",&CComponent::element,return_value_policy<reference_existing_object>())
		.def("automaton",&CComponent::automaton,return_value_policy<reference_existing_object>())
		.def("automata",&CComponent::automata)
		.def("component",&CComponent::component,return_value_policy<reference_existing_object>())
		.def("components",&CComponent::components)
		.def("reference",&CComponent::reference,return_value_policy<reference_existing_object>())
		.def("references",&CComponent::references)
		.def("variable",&CComponent::variable,return_value_policy<reference_existing_object>())
		.def("variables",static_cast<std::vector<IVariable*>(CComponent::*)()const>(&CComponent::variables))
		.def("variables",static_cast<std::vector<IVariable*>(CComponent::*)(char const*,TVarType::EVarType*)const>(&CComponent::variables))
		.def("state",&CComponent::state,return_value_policy<reference_existing_object>())
		.def("states",&CComponent::states)
		.def("transition",&CComponent::transition,return_value_policy<reference_existing_object>())
		.def("transitions",&CComponent::transitions)
		.def("isInitState",&CComponent::isInitState)
		.def("setInitState",static_cast<void(CComponent::*)(char const*)>(&CComponent::setInitState))
		.def("setInitState",static_cast<void(CComponent::*)(IState*)>(&CComponent::setInitState))
		.def("currentTime",&CComponent::currentTime)
		.def("deleteCmp",&CComponent::deleteCmp)
		.def("className",&CComponent::className)
		.def("rename",&CComponent::rename)
#ifdef SYSTEME_H
		.def("system",&CComponent::system, return_value_policy<reference_existing_object>())
#endif
		.def("pySelf",&CComponent_self)
        ;

	if(bWithKB)/*Export de la classe IBdC.*/
	class_<IBdC, IBdC, boost::noncopyable>("IBdC", init<char const*>())
		.def("name", &IBdC::name)
		.def("newComponent", static_cast<CComponent* (IBdC::*)(char const*, char const*, CSysNamedComp&)const>(&IBdC::newComponent), return_value_policy<reference_existing_object>())
		.def("classes", &IBdC::classes)
		.def("parentClasses", &IBdC::parentClasses)
		.def("version", &IBdC::version)
		;
}

#ifdef _MSC_VER
#pragma warning(default:4244 4267 4996)
#else
#pragma GCC diagnostic warning "-Wdeprecated-declarations"
#endif
