# -*- coding: iso-8859-15 -*-
"""
Copyright EDF 2017
@author: Hassane CHRAIBI - hassane.chraibi@edf.fr
"""

from PyCaUtils import *                                                            #S1
import time

######################################################################################
class Heater(Pyc.CComponent):                                                      #S1

    def __init__(self, name):                                                      #S1
        Pyc.CComponent.__init__(self, name)                                        #S1

        # The intrinsic state variables ----------------------------------------------
        # The nominal power value supplied by the heater when on state ON
        # This value is held by a variable which type is double, with 5 as
        # initial value and known as "nominalPower"
        # Note : The initial values given here may be overridden by values
        # given by a parameter file loaded before
        # simulation


        self.p_nominalPower  = self.addVariable("nominalPower",
                                                 Pyc.TVarType.t_double, 5)         #S1

        # Holds the power supplied : 0 when OFF and po_nominalPower when ON
        self.v_power         = self.addVariable("power"       ,
                                                 Pyc.TVarType.t_double, 0)         #S1

        # The failure rate of the heater
        self.p_lambda = self.addVariable("lambda",
                                          Pyc.TVarType.t_double, 0.01)             #S1

        # The repair rate of the heater
        self.p_mu     = self.addVariable("mu"    ,
                                          Pyc.TVarType.t_double, 0.1)              #S1


        # The Automata declaration----------------------------------------------------
        # heater dysfunctional automaton
        # Creation of an automaton
        self.a_dysfunctional = self.addAutomaton("DysfunctionalAutomaton")          #S1
        # Creates a state named OK and adds it to the automaton
        self.s_ok    = self.addState("DysfunctionalAutomaton", "OK", 1)             #S1
        # Creates a state named KO and adds it to the automaton
        self.s_ko     = self.addState("DysfunctionalAutomaton", "KO", 0)            #S1

        # Sets OK state as the initial state
        self.setInitState("OK")                                                     #S1

        # Creation of the transition starting from the OK state
        trans = self.s_ok.addTransition("OK_to_KO")                                 #S1
        # Setting of an exponential law as the probability law of the
        # transition law to the failure state
        # po_lambda is the parameter of this law
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self,
                                             Pyc.TLawType.expo, self.p_lambda))    #S1
        # Definition of the target of the transition
        trans.addTarget(self.s_ko, Pyc.TTransType.fault)                           #S1

        # The definition of the repair transition
        trans = self.s_ko.addTransition("KO_to_OK")                                #S1
        # Setting of an exponential law as the probability law of
        # the transition law to the fixed state
        # po_mu is the parameter of this law
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self,
                                             Pyc.TLawType.expo, self.p_mu))         #S1
        # Definition of the target of the transition
        trans.addTarget(self.s_ok, Pyc.TTransType.rep)                              #S1



        # heater functional automaton
        # Creation of an automaton
        self.a_functional = self.addAutomaton("FunctionalAutomaton")                #S1
        # Creates a state named OK and adds it to the automaton
        self.s_on  = self.addState("FunctionalAutomaton", "ON" , 1)                 #S1
        # Creates a state named KO and adds it to the automaton
        self.s_off = self.addState("FunctionalAutomaton", "OFF", 0)                 #S1
        # Sets OK state as the initial state
        self.setInitState("OFF")                                                    #S1

        # Sets the method to be called when the functional automaton changes
        # in order to update the value of the supplied heating power according to
        # the states ON and OFF
        self.a_functional.addCallback("updateSuppliedPower")                        #S1

        # Creation of the transition starting from the OFF state
        trans = self.s_off.addTransition("OFF_to_ON")                               #S1
        # Sets the condition of the transition as a boolean function
        trans.setCondition("OFF2ONCondition")
        # Definition of the target of the transition
        trans.addTarget(self.s_on, Pyc.TTransType.trans)                            #S1

        # Creation of the transition starting from the ON state
        trans = self.s_on.addTransition("ON_to_OFF")                                #S1
        # Sets the condition of the transition as a boolean function
        trans.setCondition("ON2OFFCondition")
        # Definition of the target of the transition
        trans.addTarget(self.s_off, Pyc.TTransType.trans)                           #S1



        # Makes updateSuppliedPower method to be called at the beginning of every
        # simulation in order to update the value of the supplied heating power
        # according to the initial states (ON or OFF)
        self.addStartMethod("updateSuppliedPower")                                 #S1


    # This method returns True if the heater has to switch to ON
    def OFF2ONCondition(self):                                                     #S1
        # We switch to ON only if the heater is OK
        return self.s_ok.isActive()                                                #S1


    # This method returns True if the heater has to switch to OFF
    def ON2OFFCondition(self):                                                     #S1
        # We switch to OFF when the heater is KO
        return self.s_ko.isActive()                                                #S1


    # This method is called every time the functional automaton changes
    # and also at the beginning of every simulation
    def updateSuppliedPower(self):                                                 #S1
        if self.s_on.isActive():                                                   #S1
            # If ON, the heater delivers its nominal heating power
            self.v_power.setDValue(self.p_nominalPower.value())                    #S1
        else :                                                                     #S1
            # If OFF, the heater does not deliver any  heating power
            self.v_power.setDValue(0)                                              #S1


######################################################################################
class MySystem(Pyc.CSystem):                                                       #S1
    def __init__(self, name):                                                      #S1
        Pyc.CSystem.__init__(self, name)                                           #S1

        # Instantiation of a Heater
        self.aHeater = Heater("aHeater")                                           #S1


######################################################################################
if __name__ == '__main__':                                                         #S1

    print("PyCATSHOO Version : ", Pyc.ILogManager.glLogManager().version())

    # An instance of the system is constructed (Only one system can be
    # constructed in a session)
    system = MySystem("S1")                                                        #S1

    # The simulation parameters and initial values are loaded from an XML file
    system.loadParameters("HeatedRoom_S1.xml")                                     #S1

    # Launches the simulation

    tb = time.time()
    system.simulate()                                                             #S1
    te = time.time()
    print(te-tb)

    # We create an anlyzer which holds the values of all the monitored elements
    analyser = Pyc.CAnalyser(system)                                              #S1

    # This vector will hold the instants where the values of the indicators
    # will be calculated
    vectorOfInstants = stepsVector(system.tMax(), 500)                            #S1

    # Calculates the mean values of the supplied power
    meanValues = analyser.meanValues("aHeater.power", vectorOfInstants)           #S1

    # This is acall to a convenient function which draws the curve of the
    # indicatore evolution over time
    plot(vectorOfInstants, True, True,                                            #S1
          (
             (("Output of system S1"),
              "Time",
              "Expected supplied power",
              ((meanValues, 'r', "aHeater.power"),)),
          )
         )


