# -*- coding: iso-8859-15 -*-
"""
Copyright EDF 2017
@author: Hassane CHRAIBI - hassane.chraibi@edf.fr
"""

from PyCaUtils import *                                                             #S1


######################################################################################
class Heater(Pyc.CComponent):                                                      #S1

    def __init__(self, name):                                                      #S1
        Pyc.CComponent.__init__(self, name)                                        #S1

        # The intrinsic state variables ----------------------------------------------
        # The nominal power value supplied by the heater when on state ON
        # This value is held by a variable which type is double, with 5 as
        # initial value and known as "nominalPower"
        # Note : The initial values given here may be overridden by values
        # given by a parameter file loaded before
        # simulation
        self.p_nominalPower  = self.addVariable("nominalPower",
                                                 Pyc.TVarType.t_double, 5)         #S1

        # Holds the power supplied : 0 when OFF and po_nominalPower when ON
        self.v_power         = self.addVariable("power"       ,
                                                 Pyc.TVarType.t_double, 0)         #S1

        # The failure rate of the heater
        self.p_lambda = self.addVariable("lambda",
                                          Pyc.TVarType.t_double, 0.01)             #S1

        # The repair rate of the heater
        self.p_mu     = self.addVariable("mu"    ,
                                          Pyc.TVarType.t_double, 0.1)              #S1

        # When the temperature of the heated room goes below minTemperature
        # the heater must start if requested
        self.p_minTemperature = self.addVariable("minTemperature",
                                                  Pyc.TVarType.t_double, 15)       #S2

        # When the temperature of the heated room exceeds maxTemperature
        # the heater must stop
        self.p_maxTemperature = self.addVariable("maxTemperature",
                                                  Pyc.TVarType.t_double, 20)       #S2

        # The references to external information -------------------------------------
        # This reference will be  bound to the current temperature of the room
        self.r_roomTemperature = self.addReference("roomTemperature")              #S2


        # The Automata declaration----------------------------------------------------
        # heater dysfunctional automaton
        # Creation of an automaton
        self.a_dysfunctional = self.addAutomaton("DysfunctionalAutomaton")          #S1
        # Creates a state named OK and adds it to the automaton
        self.s_ok     = self.addState("DysfunctionalAutomaton", "OK", 1)            #S1
        # Creates a state named KO and adds it to the automaton
        self.s_ko     = self.addState("DysfunctionalAutomaton", "KO", 0)            #S1

        # Sets OK state as the initial state
        self.setInitState("OK")                                                     #S1

        # Creation of the transition starting from the OK state
        trans = self.s_ok.addTransition("OK_to_KO")                                 #S1
        # Setting of an exponential law as the probability law of the
        # transition law to the failure state
        # po_lambda is the parameter of this law
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self,
                                             Pyc.TLawType.expo, self.p_lambda))   #S1
        # The definition of the target of the transition
        trans.addTarget(self.s_ko, Pyc.TTransType.fault)                          #S1

        # The definition of the repair transition
        trans = self.s_ko.addTransition("KO_to_OK")                               #S1
        # Setting of an exponential law as the probability law of
        # the transition law to the fixed state
        # po_mu is the parameter of this law
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self,
                                             Pyc.TLawType.expo, self.p_mu))       #S1
        # The definition of the target of the transition
        trans.addTarget(self.s_ok, Pyc.TTransType.rep)                            #S1



        # heater functional automaton
        # Creation of an automaton
        self.a_functional = self.addAutomaton("FunctionalAutomaton")                #S1
        # Creates a state named OK and adds it to the automaton
        self.s_on  = self.addState("FunctionalAutomaton", "ON" , 1)                 #S1
        # Creates a state named KO and adds it to the automaton
        self.s_off = self.addState("FunctionalAutomaton", "OFF", 0)                 #S1
        # Sets OK state as the initial state
        self.setInitState("OFF")                                                    #S1

        # Sets the method to be called when the functional automaton changes
        # in order to update the value of the supplied heating power according to
        # the states ON and OFF
        self.a_functional.addCallback("updateSuppliedPower")                        #S1

        # Creation of the transition starting from the OFF state
        transOff2On = self.s_off.addTransition("OFF_to_ON")                          #S1
        # Set the condition of the transition as a boolean function
        transOff2On.setCondition("OFF2ONCondition")
        # The definition of the target of the transition
        transOff2On.addTarget(self.s_on, Pyc.TTransType.trans)                       #S1

        # Creation of the transition starting from the ON state
        transOn2Off = self.s_on.addTransition("ON_to_OFF")                            #S1
        # Set condition of the transition
        # Set the condition of the transition as a boolean function
        transOn2Off.setCondition("ON2OFFCondition")
        # The definition of the target of the transition
        transOn2Off.addTarget(self.s_off, Pyc.TTransType.trans)                       #S1



        # Message boxes --------------------------------------------------------------
        # This message box will link to the  room
        messageBox = self.addMessageBox("MB-toRoom")                                 #S2
        # Transmits the value of the power supplied
        messageBox.addExport(self.v_power, "power")                                  #S2
        # Receives the value of the room current temperature
        messageBox.addImport(self.r_roomTemperature, "temperature")                  #S2


        # The PDMP Manager -----------------------------------------------------------
        pdmpManager = self.addPDMPManager("PDMP-Manager")                           #S2
        # Watch These transitions to check if the PDMP goes outside the valid region
        # according to current states (ON and OFF)
        pdmpManager.addWatchedTransition(transOn2Off)                               #S3
        pdmpManager.addWatchedTransition(transOff2On)                               #S3



        # Makes updateSuppliedPower method to be called at the beginning of every
        # simulation in order to update the value of the supplied heating power
        # according the initial states (ON or OFF)
        self.addStartMethod("updateSuppliedPower")                                 #S1


    # This method returns True if the heater has to switch to ON
    def OFF2ONCondition(self):                                                     #S1
        # We don't switch to ON if the heater is not OK
        if not self.s_ok.isActive() :                                              #S1
            return False                                                           #S1

        # We switch to ON if the temperature is too low
        return (self.r_roomTemperature.value(0) <                                  #S2
                    self.p_minTemperature.value())                                 #S2


    # This method returns True if the heater has to switch to OFF
    def ON2OFFCondition(self):                                                     #S1
        # We switch to OFF if the heater is not OK
        if not self.s_ok.isActive() :                                              #S1
            return True                                                            #S1

        # We switch to OFF if the temperature is too high
        return (self.r_roomTemperature.value(0) >                                 #S2
                self.p_maxTemperature.value())                                    #S2



    # This method is called every time the functional automaton changes
    # and also at the beginning of every simulation
    def updateSuppliedPower(self):                                                 #S1
        if self.s_on.isActive():                                                   #S1
            # If ON, the heater delivers its nominal heating power
            self.v_power.setDValue(self.p_nominalPower.value())                    #S1
        else :                                                                     #S1
            # If OFF, the heater does not deliver any  heating power
            self.v_power.setDValue(0)                                              #S1


######################################################################################
class Room(Pyc.CComponent):
    def __init__(self, name):
        Pyc.CComponent.__init__(self, name)

        # The intrinsic state variables ----------------------------------------------
        # The heat leakage rate in the room
        self.p_leakageRate = self.addVariable("leakageRate",
                                               Pyc.TVarType.t_double, 3.0)         #S2

        # The temperature of the room
        self.v_temperature = self.addVariable("temperature",
                                               Pyc.TVarType.t_double, 17)          #S2
        # The temperature of the outside
        self.p_outsideTemperature = self.addVariable("outsideTemperature",
                                                      Pyc.TVarType.t_double, 13)   #S2
        # The initial temperature of the room
        self.p_initialTemperature = self.addVariable("initialTemperature",
                                                      Pyc.TVarType.t_double, 17)   #S2

        # The references to external information -------------------------------------
        # This value will hold the value(s) of power supplied by the heater(s)
        self.r_power = self.addReference("power")                                 #S2

        # Message boxes --------------------------------------------------------------
        # This message box will link to the heater(s)
        messageBox = self.addMessageBox("MB-toHeaters")                           #S2
        # From the heaters, the values of the power they supply is received
        messageBox.addImport(self.r_power, "power")                               #S2
        # To the heater, the value of the current temperature will be sent
        messageBox.addExport(self.v_temperature, "temperature")                   #S2

        # The PDMP Manager -----------------------------------------------------------
        pdmpManager = self.addPDMPManager("PDMP-Manager")                          #S2
        pdmpManager.addEquationMethod("pdmpMethod", self)                          #S2
        pdmpManager.addODEVariable   (self.v_temperature)                          #S2

        # Makes start method to be called at the beginning of every simulation
        self.addStartMethod("start", self.start)                                   #S2

    # Start method makes the currentTemperature to be equal to the
    # initialTemperature at the beginning of the simulation
    def start(self):                                                               #S2
        self.v_temperature.setDValue(self.p_initialTemperature.dValue())           #S2

    # The pdmpMethode gives the derivative of the room temperature
    # The derivative expression is deduced from the energy
    # balance in the room
    def pdmpMethod(self):                                                          #S2
        self.v_temperature.setDvdtODE \
            (                                                                      #S2
              self.r_power.sumValue(0) -                                           #S2
              self.p_leakageRate.value() *                                         #S2
              (self.v_temperature.value() - self.p_outsideTemperature.value())     #S2
            )                                                                      #S2


######################################################################################
class MySystem(Pyc.CSystem):                                                       #S1
    def __init__(self, name):                                                      #S1
        Pyc.CSystem.__init__(self, name)                                           #S1

        # Instantiation of a Master and Slave Heater
        self.aHeater = Heater("aHeater")                                           #S1
        self.room          = Room("Room")                                          #S2

        # connecting the heater to the room :
        self.connect("aHeater", "MB-toRoom", "Room", "MB-toHeaters")               #S2



######################################################################################
if __name__ == '__main__':                                                         #S1

    print("PyCATSHOO Version : ", Pyc.ILogManager.glLogManager().version())

    # An instance of the system is constructed (Only one system can be
    # constructed in a session)
    system = MySystem("S4")                                                        #S2

    # The simulation et initial values are load from an XML file
    system.loadParameters("HeatedRoom_S4.xml")                                     #S2

    # Lanches the simulation
    system.simulate()                                                              #S1

    # We create an anlyzer which holds the values of all the monitored elements
    analyser = Pyc.CAnalyser(system)                                               #S1

    # This vector will holds the instants where the values of the indicators
    # will be calculated
    vectorOfInstants = stepsVector(system.tMax(), 500)                             #S1

    # Calculates the evolution over time of the mean of the room temperature :
    temperatureMeans = analyser.meanValues("Room.temperature",                     #S2
                                            vectorOfInstants)                      #S2


    #This is call to a convenient function which draws the curves of the
    # indicators' evolution over time
    plot(vectorOfInstants, True, False,                                            #S2
             (                                                                     #S2
                 (("Output of system S4"),                                         #S2
                  "Time",                                                          #S2
                  "Temperature",                                                   #S2
                  ((temperatureMeans, 'g', "Room mean temperature"),)),            #S2
             )                                                                     #S2
            )                                                                      #S2


    # This vector will hold the cumulative probability distribution
    # of the Indicator Temperature < 14
    hotTemperatureOccurrences = analyser.realized("Room.temperature",              #S4
                                                  vectorOfInstants,                #S4
                                                  lambda t: t < 14)                #S4

    # This is call to a convenient function which draws the curve of the
    # cumulative probability distribution calculated above
    plot(vectorOfInstants, True, True,                                             #S4
         (                                                                         #S4
             (("Output of system S4"),                                             #S4
              "Time",                                                              #S4
              "Probability",                                                       #S4
              ((hotTemperatureOccurrences, 'r', "CPDF Ind : Temperature < 14."),)  #S4
              ),                                                                   #S4
         )                                                                         #S4
         )                                                                         #S4
