# -*- coding: iso-8859-15 -*-
"""
Copyright EDF 2022
@author: Hassane CHRAIBI - hassane.chraibi@edf.fr
"""

from PyCaUtils import *


class Tank(Pyc.CComponent):
    def __init__(self, name):
        Pyc.CComponent.__init__(self, name)

        self.p_FMUPath = self.addVariable("FMUPath", Pyc.TVarType.t_string, "")
        self.p_hotTemperature = self.addVariable("hotTemperature", Pyc.TVarType.t_double, 100.)
        self.p_overFlowLevel  = self.addVariable("overFlowLevel" , Pyc.TVarType.t_double,  10.)
        self.p_dryOutLevel    = self.addVariable("dryOutLevel"   , Pyc.TVarType.t_double,   4.)

        if len(self.p_FMUPath.sValue()) == 0:
            self.fmu = None
            self.p_area        = self.addVariable("area", Pyc.TVarType.t_double, 1.)
            self.v_temperature = self.addVariable("temperature", Pyc.TVarType.t_double, 7.)
            self.v_level       = self.addVariable("level", Pyc.TVarType.t_double, 7.)
        else :
            try:
                self.fmu = self.system().addFMU(self.p_FMUPath.sValue(), "FMU")
            except e :
                raise (e)

            if self.fmu == None :
                raise

            self.p_area        = self.fmu.variable('FMUTank.Area')
            self.v_fmuPower    = self.fmu.variable('FMUTank.Power')
            self.v_fmuOutFlow  = self.fmu.variable('FMUTank.OutputFlow')
            self.v_temperature = self.fmu.variable("FMUTank.Temperature")
            self.v_level       = self.fmu.variable("FMUTank.Level")

            self.vs_fmuInFlow        = []
            self.vs_fmuInTemperature = []
            for i in range(10) :
                if self.fmu.hasVariable(f'FMUTank.InputFlow_{i+1}'):
                    self.vs_fmuInFlow.append       (self.fmu.variable(f'FMUTank.InputFlow_{i+1}'))
                    self.vs_fmuInFlow[-1].setDValue( 0.)
                    self.vs_fmuInTemperature.append(self.fmu.variable(f'FMUTank.InputTemperature_{i+1}'))
                    self.vs_fmuInTemperature[-1].setDValue( 0.)
                else :
                    self.vs_fmuInFlow.append       (None)
                    self.vs_fmuInTemperature.append(None)


        self.r_temperature    = self.addReference("temperature")
        self.r_inFlow         = self.addReference("inFlow")  # S1
        self.r_outFlow        = self.addReference("outFlow")  # S2a
        self.r_power          = self.addReference("power")


        self.addMessageBox("Fuel")
        self.addMessageBoxImport("Fuel", self.r_power, "power")

        self.addMessageBox("Pump")
        self.addMessageBoxImport("Pump", self.r_inFlow 	, "flow")
        self.addMessageBoxImport("Pump", self.r_temperature, "temperature")
        self.addMessageBoxExport("Pump", self.v_level      , "level")
        self.addMessageBoxExport("Pump", self.v_temperature, "temperature")

        self.addMessageBox("Valve")
        self.addMessageBoxImport("Valve", self.r_outFlow 	 , "flow")
        self.addMessageBoxExport("Valve", self.v_level      , "level")
        self.addMessageBoxExport("Valve", self.v_temperature, "temperature")



        self.levelAutomaton = self.addAutomaton("LevelAutomaton")
        self.levelOK  = self.addState("LevelAutomaton", "LevelOK" , 0)
        self.overFlow = self.addState("LevelAutomaton", "OverFlow", 1)
        self.dryOut   = self.addState("LevelAutomaton", "DryOut"  , 2)
        self.levelAutomaton.setInitState(self.levelOK)

        trans_lok_ovf = self.levelOK.addTransition("LOK_to_OVF")
        trans_lok_ovf.setCondition("overFlowCondition", self.overFlowCondition, False)
        trans_lok_ovf.addTarget(self.overFlow, Pyc.TTransType.trans)

        trans_ovf_lok = self.overFlow.addTransition("OVF_to_LOK")
        trans_ovf_lok.setCondition("overFlowCondition", self.overFlowCondition, True)
        trans_ovf_lok.addTarget(self.levelOK, Pyc.TTransType.trans)

        trans_lok_do = self.levelOK.addTransition("LOK_to_DO")
        trans_lok_do.setCondition("dryOutCondition", self.dryOutCondition, False)
        trans_lok_do.addTarget(self.dryOut, Pyc.TTransType.trans)

        trans_do_lok = self.dryOut.addTransition("DO_to_LOK")
        trans_do_lok.setCondition("dryOutCondition", self.dryOutCondition, True)
        trans_do_lok.addTarget(self.levelOK, Pyc.TTransType.trans)

        self.temperaturAutomaton = self.addAutomaton("TemperatureAutomaton")
        self.temperatureOK  = self.addState("TemperatureAutomaton", "TemperatureOK" , 0)
        self.hotTemperature = self.addState("TemperatureAutomaton", "HotTemperature", 1)
        self.setInitState("TemperatureOK")

        trans_tok_th = self.temperatureOK.addTransition("TOK_to_THeigh")
        trans_tok_th.setCondition("hotTemperatureCondition", self.hotTemperatureCondition, False)
        trans_tok_th.addTarget(self.hotTemperature, Pyc.TTransType.trans)

        trans_th_tok = self.hotTemperature.addTransition("THeigh_to_TOK")
        trans_th_tok.setCondition("hotTemperatureCondition", self.hotTemperatureCondition, True)
        trans_th_tok.addTarget(self.temperatureOK, Pyc.TTransType.trans)

        pdmp = self.addPDMPManager  ("pdmpManager")

        self.addPDMPWatchedTransition("pdmpManager", trans_lok_ovf)
        self.addPDMPWatchedTransition("pdmpManager", trans_ovf_lok)
        self.addPDMPWatchedTransition("pdmpManager", trans_lok_do)
        self.addPDMPWatchedTransition("pdmpManager", trans_do_lok)
        self.addPDMPWatchedTransition("pdmpManager", trans_tok_th)
        self.addPDMPWatchedTransition("pdmpManager", trans_th_tok)


        if self.fmu == None:
            self.addPDMPEquationMethod       ("pdmpManager", "odeMethod"      , self.odeMethod)
            self.addPDMPODEVariable		     ("pdmpManager", self.v_level)
            self.addPDMPODEVariable		     ("pdmpManager", self.v_temperature)
        else :
            pdmp.addFMU( self.system().FMU("FMU"))
            self.addPDMPBeginMethod     ("pdmpManager", "beginMethod",  self.beginMethod)
            self.addPDMPExplicitVariable("pdmpManager", self.v_level)
            self.addPDMPExplicitVariable("pdmpManager", self.v_temperature)


    def odeMethod(self):
        iFlow    = self.r_inFlow.sumValue()
        oFlow    = self.r_outFlow.sumValue()

        self.v_level.setDvdtODE((iFlow - oFlow) / self.p_area.dValue())

        sumiFiT = 0
        for i in range(self.r_inFlow.nbCnx()):
            sumiFiT = sumiFiT + self.r_inFlow.dValue(i) * \
                                (self.r_temperature.dValue(i) - self.v_temperature.value())

        self.v_temperature.setDvdtODE(
            (sumiFiT + self.r_power.dValue(0)) /
            (self.v_level.dValue() * self.p_area.dValue())
                                      )


    def beginMethod(self):
        for i in range(self.r_inFlow.cnctCount()):
            self.vs_fmuInFlow[i].setDValue       (self.r_inFlow.dValue(i))
            self.vs_fmuInTemperature[i].setDValue(self.r_temperature.dValue(i))

        self.v_fmuPower.setDValue(self.r_power.sumValue());
        self.v_fmuOutFlow.setDValue(self.r_outFlow.sumValue())


    def overFlowCondition(self):
        return (self.v_level.dValue() > self.p_overFlowLevel.dValue())


    def dryOutCondition(self):
        return (self.v_level.dValue() < self.p_dryOutLevel.dValue())


    def hotTemperatureCondition(self):
        return (self.v_temperature.dValue() > self.p_hotTemperature.dValue())

########################################################################################
class HydroDevice(Pyc.CComponent):                                                              
    def __init__(self, name):                                                                   
        Pyc.CComponent.__init__(self, name)

        self.p_nominalFlow       = self.addVariable("nominalFlow", Pyc.TVarType.t_double, 1.5)
        self.v_flow              = self.addVariable("flow"       , Pyc.TVarType.t_double, 1.5)
        self.p_lambda0           = self.addVariable("lambda0"    , Pyc.TVarType.t_double, 0.001)  
        self.v_lambda            = self.addVariable("lambda"     , Pyc.TVarType.t_double, 0.001)  

        self.p_levelMax          = self.addVariable("levelMax"   , Pyc.TVarType.t_double, 8.)
        self.p_levelMin          = self.addVariable("levelMin"   , Pyc.TVarType.t_double, 6.)

        self.r_tankLevel         = self.addReference("tankLevel")                              
        self.r_tankTemperature   = self.addReference("tankTemperature")                       



        self.addMessageBox("Tank")                                                            
        self.addMessageBoxExport("Tank", self.v_flow 	  , "flow")                         
        self.addMessageBoxImport("Tank", self.r_tankLevel, "level")                            
        self.addMessageBoxImport("Tank", self.r_tankTemperature, "temperature")               

        self.a_automaton  = self.addAutomaton("FuncAutomation")
        self.s_openState  = self.addState("FuncAutomation"     , "OPEN" , 0)
        self.s_closeState = self.addState("FuncAutomation"     , "CLOSE", 1)
        self.s_stuckOpenState  = self.addState("FuncAutomation", "STUCKOPEN" , 2)
        self.stuckCloseState = self.addState("FuncAutomation"  , "STUCKCLOSE", 3)
        self.a_automaton.setInitState(self.s_openState)
        self.a_automaton.addSensitiveMethod("updateFlow", self.updateFlow)

        trans_o2c = self.s_openState.addTransition("OPEN_to_CLOSE")
        trans_o2c.setCondition("closeCondition", self.closeCondition, False)
        trans_o2c.addTarget(self.s_closeState, Pyc.TTransType.trans)

        trans_c2o = self.s_closeState.addTransition("CLOSE_to_OPEN")
        trans_c2o.setCondition("openCondition", self.openCondition, False)
        trans_c2o.addTarget(self.s_openState, Pyc.TTransType.trans)

        trans = self.s_openState.addTransition("OPEN_to_STUCKOPEN")
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))
        trans.addTarget(self.s_stuckOpenState, Pyc.TTransType.fault)
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)

        trans = self.s_openState.addTransition("OPEN_to_STUCKCLOSE")
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))
        trans.addTarget(self.stuckCloseState, Pyc.TTransType.fault)
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)

        trans = self.s_closeState.addTransition("CLOSE_to_STUCKOPEN")                             
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))          
        trans.addTarget(self.s_stuckOpenState, Pyc.TTransType.fault)                              
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)                      

        trans = self.s_closeState.addTransition("CLOSE_to_STUCKCLOSE")                            
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))          
        trans.addTarget(self.stuckCloseState, Pyc.TTransType.fault)                            
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)                      

        self.addPDMPManager		     ("pdmpManager")
        self.addPDMPWatchedTransition("pdmpManager", trans_o2c)
        self.addPDMPWatchedTransition("pdmpManager", trans_c2o)

        self.addPDMPEquationMethod    ("pdmpManager", "odeMethod"      , self.odeMethod, 0)
        self.addPDMPExplicitVariable  ("pdmpManager", self.v_lambda)

        self.addStartMethod("updateFlow", self.updateFlow)                                      #S3

    def updateFlow(self):                                                                       #S3
        if self.s_openState.isActive() or self.s_stuckOpenState.isActive():                         #S3 ==> S4
            self.v_flow.setValue(self.p_nominalFlow.dValue())                                 #S3
        else :                                                                                  #S3
            self.v_flow.setValue(0)                                                            #S3

    def odeMethod(self):                                                                        #S8b
        b1 = 3.0295                                                                             #S8b
        b2 = 0.7578                                                                             #S8b
        bc = 0.05756                                                                            #S8b
        bd = 0.2301                                                                             #S8b
        lmbda = self.p_lambda0.value()*\
                 ( b1 * np.exp(+ bc * (self.r_tankTemperature.value(0) - 20)) +               #S8b
                   b2 * np.exp(- bd * (self.r_tankTemperature.value(0) - 20))                 #S8b
                 ) / (b1 + b2)

        self.v_lambda.setDValue(lmbda)



########################################################################################
class Pump(HydroDevice):                                                                        #S2a ==> S2b
    def __init__(self, name):                                                                   #S1
        HydroDevice.__init__(self, name)                                                        #S2a ==> S2b


        self.v_temperature = self.addVariable("temperature", Pyc.TVarType.t_double, 15.)          #S5

        self.addMessageBoxExport("Tank", self.v_temperature, "temperature")                    #S5


    def openCondition(self):                                                                    #S6a
        return (self.r_tankLevel.dValue(0) < self.p_levelMin.dValue())                        #S6a

    def closeCondition(self):                                                                   #S6a
        return (self.r_tankLevel.dValue(0) > self.p_levelMax.dValue())                        #S6a



########################################################################################
class Valve(HydroDevice):
    def __init__(self, name)    :
        HydroDevice.__init__(self, name)


    def openCondition(self):
        return self.r_tankLevel.dValue(0) > self.p_levelMax.dValue()

    def closeCondition(self):
            return (self.r_tankLevel.dValue(0) < self.p_levelMin.dValue())



########################################################################################
class Fuel(Pyc.CComponent):
    def __init__(self, name):
        Pyc.CComponent.__init__(self, name)

        self.p_power = self.addVariable("power", Pyc.TVarType.t_double, 5.)
        self.addMessageBox("Tank")
        self.addMessageBoxExport("Tank", self.p_power, "power")


class MySystem(Pyc.CSystem):
    def __init__(self, name):
        Pyc.CSystem.__init__(self, name)

    def construct(self):
        self.P1   = Pump("P1")
        self.P2   = Pump("P2")
        self.V3   = Valve("V3")
        self.tank = Tank("Tank")
        self.fuel = Fuel("Fuel")

        self.connect("P1"  , "Tank", "Tank", "Pump" )
        self.connect("P2"  , "Tank", "Tank", "Pump" )
        self.connect("V3"  , "Tank", "Tank", "Valve")
        self.connect("Fuel", "Tank", "Tank", "Fuel" )




def mySequenceFilter(sequence):
    ofElement = Pyc.CSystem.glSystem().monitoredElt("Tank.OverFlow"      , "ST")
    doElement = Pyc.CSystem.glSystem().monitoredElt("Tank.DryOut"        , "ST")
    htElement = Pyc.CSystem.glSystem().monitoredElt("Tank.HotTemperature", "ST")

    return( sequence.value(ofElement, system.tMax()) == 1 or
            sequence.value(doElement, system.tMax()) == 1 or
            sequence.value(htElement, system.tMax()) == 1
          )




if __name__ == '__main__':

    print("PyCATSHOO Version : ", Pyc.ILogManager.glLogManager().version())

    FMU = True
    try :
        Pyc.glLogManager().setDisplayLevel(Pyc.TErrorLevel.error)

        system = MySystem("HeatedTankSystem")

        if FMU:
            system.loadParameters("HTBaseP.xml")
            system.construct()
            system.loadParameters("HTBaseFMU.xml")
        else :
            system.construct()
            system.loadParameters("HTBase.xml")



        system.setRNGSeed(4536)

        system.setResultFileName("HTBaseRes.xml", True)

        system.addTarget("HotTemperature", "Tank.HotTemperature", "ST")
        system.addTarget("OverFlow"      , "Tank.OverFlow"      , "ST")
        system.addTarget("DryOut"        , "Tank.DryOut"        , "ST")


        system.addInstants(0, system.tMax(), 100)


        ofIndicator = system.addIndicator("Overflow", "Tank.OverFlow", "ST")
        ofIndicator.setRestitutions(Pyc.TIndicatorType.mean_values)

        doIndicator = system.addIndicator("DryOut", "Tank.DryOut", "ST")
        doIndicator.setRestitutions(Pyc.TIndicatorType.mean_values)

        htIndicator = system.addIndicator("HotTemp", "Tank.HotTemperature", "ST")
        htIndicator.setRestitutions(Pyc.TIndicatorType.mean_values)

        tb = time.time()
        system.simulate()

        if system.MPIRank() > 0 :
            exit(0)

        te=time .time()
        sTime = te-tb
        print ("Simulation time ", sTime )

        meanOVF = ofIndicator.means()
        meanDO  = doIndicator.means()
        meanHT  = htIndicator.means()

        tps = system.instants()

        plot(tps, True, True,
                (
                    (("Tank"), "Time", "OverFlow", ((meanOVF, 'r-', 'Tank OverFlow'),)),
                    (("Tank"), "Time", "DryOut"  , ((meanDO , 'g-', 'Tank DryOut'  ),)),
                    (("Tank"), "Time", "Burnt"   , ((meanHT , 'b-', 'Tank Burnt'   ),)),
                )
             )


    except Exception as e:
        print(e)
