# -*- coding: iso-8859-15 -*-
"""
Copyright EDF 2022
@author: Hassane CHRAIBI - hassane.chraibi@edf.fr
"""

from PyCaUtils import *                                                                              #S1


class Tank(Pyc.CComponent):                                                                     #S1
    def __init__(self, name):                                                                   #S1
        Pyc.CComponent.__init__(self, name)                                                     #S1

        self.p_area            = self.addVariable("area"          , Pyc.TVarType.t_double, 1.)     #S1
        self.v_level           = self.addVariable("level"         , Pyc.TVarType.t_double, 7.)     #S1
        self.v_temperature     = self.addVariable("temperature"   , Pyc.TVarType.t_double, 31.)    #S5

        self.r_inFlow         = self.addReference("inFlow")                                    #S1
        self.r_outFlow        = self.addReference("outFlow")                                   #S2a
        self.r_temperature    = self.addReference("temperature")                               #S5
        self.r_power          = self.addReference("power")                                     #S7

        self.p_hotTemperature = self.addVariable("hotTemperature", Pyc.TVarType.t_double, 100.)   #S8c
        self.p_overFlowLevel  = self.addVariable("overFlowLevel" , Pyc.TVarType.t_double, 10.)    #S8c
        self.p_dryOutLevel    = self.addVariable("dryOutLevel"   , Pyc.TVarType.t_double, 4.)     #S8c


        self.addMessageBox("Pump")                                                              #S1
        self.addMessageBoxImport("Pump", self.r_inFlow 	, "flow")                           #S1
        self.addMessageBoxImport("Pump", self.r_temperature, "temperature")                    #S5
        self.addMessageBoxExport("Pump", self.v_level      , "level")                          #S6a
        self.addMessageBoxExport("Pump", self.v_temperature, "temperature")                    #S8b

        self.addMessageBox("Valve")                                                              #S2a
        self.addMessageBoxImport("Valve", self.r_outFlow 	 , "flow")                           #S2a
        self.addMessageBoxExport("Valve", self.v_level      , "level")                           #S6a
        self.addMessageBoxExport("Valve", self.v_temperature, "temperature")                     #S8b

        self.addMessageBox("Fuel")                                                                #S7
        self.addMessageBoxImport("Fuel", self.r_power, "power")                                   #S7

        self.a_levelAutomaton = self.addAutomaton("LevelAutomaton")                               #S8c
        self.s_LevelOk  = self.addState("LevelAutomaton", "LevelOK" , 0)                          #S8c
        self.s_overFlow = self.addState("LevelAutomaton", "OverFlow", 1)                          #S8c
        self.s_dryOut   = self.addState("LevelAutomaton", "DryOut"  , 2)                          #S8c
        self.a_levelAutomaton.setInitState(self.s_LevelOk)                                        #S8c

        transOk2Ovf = self.s_LevelOk.addTransition("LOK_to_OVF")                                  #S8c
        transOk2Ovf.setCondition("overFlowCondition", self.overFlowCondition, False)              #S8c
        transOk2Ovf.addTarget(self.s_overFlow, Pyc.TTransType.trans)                              #S8c

        transOvf2Ok = self.s_overFlow.addTransition("OVF_to_LOK")                                 #S8c
        transOvf2Ok.setCondition("overFlowCondition", self.overFlowCondition, True)               #S8c
        transOvf2Ok.addTarget(self.s_LevelOk, Pyc.TTransType.trans)                               #S8c

        transOk2Do = self.s_LevelOk.addTransition("LOK_to_DO")                                    #S8c
        transOk2Do.setCondition("dryOutCondition", self.dryOutCondition, False)                 #S8c
        transOk2Do.addTarget(self.s_dryOut, Pyc.TTransType.trans)                                 #S8c

        transDo2Ok = self.s_dryOut.addTransition("DO_to_LOK")                                     #S8c
        transDo2Ok.setCondition("dryOutCondition", self.dryOutCondition, True)                  #S8c
        transDo2Ok.addTarget(self.s_LevelOk, Pyc.TTransType.trans)                                #S8c

        self.a_temperatureAutomaton = self.addAutomaton("TemperatureAutomaton")                    #S8c
        self.s_temperatureOK  = self.addState("TemperatureAutomaton", "TemperatureOK" , 0)        #S8c
        self.s_hotTemperature = self.addState("TemperatureAutomaton", "HotTemperature", 1)        #S8c
        self.setInitState("TemperatureOK")

        transOk2HT = self.s_temperatureOK.addTransition("TOK_to_THeigh")                               #S8c
        transOk2HT.setCondition("hotTemperatureCondition", self.hotTemperatureCondition, False)      #S8c
        transOk2HT.addTarget(self.s_hotTemperature, Pyc.TTransType.trans)                              #S8c

        transHT2Ok = self.s_hotTemperature.addTransition("THeigh_to_TOK")                             #S8c
        transHT2Ok.setCondition("hotTemperatureCondition", self.hotTemperatureCondition, True)        #S8c
        transHT2Ok.addTarget(self.s_temperatureOK, Pyc.TTransType.trans)                              #S8c

        self.addPDMPManager			 ("pdmpManager")                                            #S1
        self.addPDMPEquationMethod   ("pdmpManager", "odeMethod"      , self.odeMethod)         #S1
        self.addPDMPWatchedTransition("pdmpManager",transOk2Ovf)                                #S8c
        self.addPDMPWatchedTransition("pdmpManager",transOk2Do)                                 #S8c
        self.addPDMPWatchedTransition("pdmpManager",transOk2HT)                                 #S8c
        self.addPDMPWatchedTransition("pdmpManager",transOvf2Ok)                                #S8c
        self.addPDMPWatchedTransition("pdmpManager",transDo2Ok)                                 #S8c
        self.addPDMPWatchedTransition("pdmpManager",transHT2Ok)                                 #S8c

        self.addPDMPODEVariable		 ("pdmpManager", self.v_level)                          #S1
        self.addPDMPODEVariable		 ("pdmpManager", self.v_temperature)                    #S5

    def odeMethod(self):                                                                        #S1
        iFlow    = self.r_inFlow.sumValue()                                                     #S1
        oFlow    = self.r_outFlow.sumValue()                                                    #S2a

        self.v_level.setDvdtODE((iFlow - oFlow) / self.p_area.dValue())                         #S1 ==> S2a

        sumiFiT = 0                                                                             #S5
        for i in range(self.r_inFlow.nbCnx()):                                                  #S5
            sumiFiT = sumiFiT + self.r_inFlow.dValue(i) * \
                                (self.r_temperature.dValue(i) - self.v_temperature.value())     #S5

        self.v_temperature.setDvdtODE(                                                          #S5
            (sumiFiT + self.r_power.dValue(0)) /                                                #S5 ==> S7
            (self.v_level.dValue() * self.p_area.dValue())                                      #S5
                                      )                                                         #S5                                                                     #S8c


    def overFlowCondition(self):                                                                #S8c
        return (self.v_level.dValue() > self.p_overFlowLevel.dValue())                          #S8c


    def dryOutCondition(self):                                                                  #S8c
        return (self.v_level.dValue() < self.p_dryOutLevel.dValue())                            #S8c


    def hotTemperatureCondition(self):                                                          #S8c
        return (self.v_temperature.dValue() > self.p_hotTemperature.dValue())                   #S8c

########################################################################################
class HydroDevice(Pyc.CComponent):                                                              #S2b
    def __init__(self, name):                                                                   #S2b
        Pyc.CComponent.__init__(self, name)                                                     #S2b

        self.p_nominalFlow       = self.addVariable("nominalFlow", Pyc.TVarType.t_double, 1.5)  #S3
        self.v_flow              = self.addVariable("flow"       , Pyc.TVarType.t_double, 1.5)  #S2b

        self.p_lambda0           = self.addVariable("lambda0"    , Pyc.TVarType.t_double, 0.001) #S8b
        self.v_lambda            = self.addVariable("lambda"     , Pyc.TVarType.t_double, 0.001) #S4

        self.p_levelMax          = self.addVariable("levelMax"   , Pyc.TVarType.t_double, 8.)    #S6a
        self.p_levelMin          = self.addVariable("levelMin"   , Pyc.TVarType.t_double, 6.)    #S6a

        self.r_tankLevel         = self.addReference("tankLevel")                                #S6a
        self.r_tankTemperature   = self.addReference("tankTemperature")                          #S8b



        self.addMessageBox("Tank")                                                                #S2b
        self.addMessageBoxExport("Tank", self.v_flow 	  , "flow")               #S2b
        self.addMessageBoxImport("Tank", self.r_tankLevel, "level")              #S6a
        self.addMessageBoxImport("Tank", self.r_tankTemperature, "temperature")  #S8b

        self.a_automaton  = self.addAutomaton("FuncAutomation")                                  #S3
        self.s_open  = self.addState("FuncAutomation", "OPEN" , 0)              #S3
        self.s_close = self.addState("FuncAutomation", "CLOSE", 1)              #S3
        self.s_stuckOpen  = self.addState("FuncAutomation", "STUCKOPEN" , 2)    #S4
        self.s_stuckClose = self.addState("FuncAutomation", "STUCKCLOSE", 3)    #S4
        self.a_automaton.setInitState(self.s_open)                                                #S3
        self.a_automaton.addCallback("updateFlow", self.updateFlow)                               #S3

        transO2C = self.s_open.addTransition("OPEN_to_CLOSE")                                     #S3
        transO2C.setCondition("closeCondition", self.closeCondition, False)                       #S3 ==> S6a
        transO2C.addTarget(self.s_close, Pyc.TTransType.trans)                                    #S3

        transC2O = self.s_close.addTransition("CLOSE_to_OPEN")                                    #S3
        transC2O.setCondition("openCondition", self.openCondition, False)                         #S3 ==> S6a
        transC2O.addTarget(self.s_open, Pyc.TTransType.trans)                                     #S3

        trans = self.s_open.addTransition("OPEN_to_STUCKOPEN")                                    #S4
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))             #S4
        trans.addTarget(self.s_stuckOpen, Pyc.TTransType.fault)                                   #S4
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)                        #S8b

        trans = self.s_open.addTransition("OPEN_to_STUCKCLOSE")                                   #S4
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))             #S4
        trans.addTarget(self.s_stuckClose, Pyc.TTransType.fault)                                  #S4
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)                        #S8b

        trans = self.s_close.addTransition("CLOSE_to_STUCKOPEN")                                  #S4
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))             #S4
        trans.addTarget(self.s_stuckOpen, Pyc.TTransType.fault)                                   #S4
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)                        #S8b

        trans = self.s_close.addTransition("CLOSE_to_STUCKCLOSE")                                 #S4
        trans.setDistLaw(Pyc.IDistLaw.newLaw(self, Pyc.TLawType.expo, self.v_lambda))             #S4
        trans.addTarget(self.s_stuckClose, Pyc.TTransType.fault)                                  #S4
        trans.setModifiable(Pyc.TModificationMode.continuous_modification)                        #S8b

        self.addPDMPManager		         ("pdmpManager")                                            #S6b
        self.addPDMPWatchedTransition("pdmpManager", transO2C)                                      #S6b
        self.addPDMPWatchedTransition("pdmpManager", transC2O)                                      #S6b
        self.addPDMPEquationMethod       ("pdmpManager", "odeMethod"      , self.odeMethod, 0)      #S8b
        self.addPDMPExplicitVariable     ("pdmpManager", self.v_lambda)                             #S8b

        self.addStartMethod("updateFlow", self.updateFlow)                                          #S3

    def updateFlow(self):                                                                           #S3
        if self.s_open.isActive() or self.s_stuckOpen.isActive():                             #S3 ==> S4
            self.v_flow.setValue(self.p_nominalFlow.dValue())                                       #S3
        else :                                                                                      #S3
            self.v_flow.setValue(0)                                                                 #S3

    def odeMethod(self):                                                                        #S8b
        b1 = 3.0295                                                                             #S8b
        b2 = 0.7578                                                                             #S8b
        bc = 0.05756                                                                            #S8b
        bd = 0.2301                                                                             #S8b
        lmbda = self.p_lambda0.value()*\
                 ( b1 * np.exp(+ bc * (self.r_tankTemperature.value(0) - 20)) +               #S8b
                   b2 * np.exp(- bd * (self.r_tankTemperature.value(0) - 20))                 #S8b
                 ) / (b1 + b2)

        self.v_lambda.setDValue(lmbda)


########################################################################################
class Pump(HydroDevice):                                                                        #S2a ==> S2b
    def __init__(self, name):                                                                   #S1
        HydroDevice.__init__(self, name)                                                        #S2a ==> S2b


        self.v_temperature = self.addVariable("temperature", Pyc.TVarType.t_double, 15.)          #S5

        self.addMessageBoxExport("Tank", self.v_temperature, "temperature")                    #S5


    def openCondition(self):                                                                    #S6a
        return (self.r_tankLevel.dValue(0) < self.p_levelMin.dValue())                        #S6a

    def closeCondition(self):                                                                   #S6a
        return (self.r_tankLevel.dValue(0) > self.p_levelMax.dValue())                        #S6a



########################################################################################
class Valve(HydroDevice):                                                                       #S2a ==> S2b
    def __init__(self, name)    :                                                               #S2a
        HydroDevice.__init__(self, name)                                                        #S2a ==> S2b


    def openCondition(self):                                                                    #S6a
        return self.r_tankLevel.dValue(0) > self.p_levelMax.dValue()                          #S6a

    def closeCondition(self):                                                                   #S6a
            return (self.r_tankLevel.dValue(0) < self.p_levelMin.dValue())                    #S6a



########################################################################################
class Fuel(Pyc.CComponent):                                                                         #S7
    def __init__(self, name):                                                                       #S7
        Pyc.CComponent.__init__(self, name)                                                         #S7

        self.p_power = self.addVariable("power", Pyc.TVarType.t_double, 5.)                           #S7
        self.addMessageBox("Tank")                                                                  #S7
        self.addMessageBoxExport("Tank", self.p_power, "power")                                    #S7


class MySystem(Pyc.CSystem):                                                                    #S1
    def __init__(self, name):                                                                   #S1
        Pyc.CSystem.__init__(self, name)                                                        #S1

        self.P1   = Pump("P1")                                                                  #S1
        self.P2   = Pump("P2")                                                                  #S1
        self.V3   = Valve("V3")                                                                 #S2a
        self.tank = Tank("Tank")                                                                #S1
        self.fuel = Fuel("Fuel")                                                                #S7

        self.connect("P1"  , "Tank", "Tank", "Pump" )                                           #S1
        self.connect("P2"  , "Tank", "Tank", "Pump" )                                           #S1
        self.connect("V3"  , "Tank", "Tank", "Valve")                                           #S2a
        self.connect("Fuel", "Tank", "Tank", "Fuel" )                                           #S7




def mySequenceFilter(sequence):                                                                 #S8d
    ofElement = Pyc.CSystem.glSystem().monitoredElt("Tank.OverFlow"      , "ST")             #S8d
    doElement = Pyc.CSystem.glSystem().monitoredElt("Tank.DryOut"        , "ST")             #S8d
    htElement = Pyc.CSystem.glSystem().monitoredElt("Tank.HotTemperature", "ST")             #S8d

    return( sequence.value(ofElement, system.tMax()) == 1 or                                    #S8d
            sequence.value(doElement, system.tMax()) == 1 or                                    #S8d
            sequence.value(htElement, system.tMax()) == 1                                       #S8d
          )




if __name__ == '__main__':                                                                      #S1

    print("PyCATSHOO Version : ", Pyc.ILogManager.glLogManager().version())

    system = MySystem("HeatedTankSystem")                                                       #S1
    system.loadParameters("S8dHTBase.xml")                                                           #S1
    system.setRNGSeed(4536)                                                                     #S1

    system.setResultFileName("S8dHTBaseRes.xml", True)

    system.addTarget("HotTemperature", "Tank.HotTemperature", "ST")                             #S8c
    system.addTarget("OverFlow"      , "Tank.OverFlow"      , "ST")                             #S8c
    system.addTarget("DryOut"        , "Tank.DryOut"        , "ST")                             #S8c


    system.addInstants(0, system.tMax(), 100)                                                   #S8c


    ofIndicator = system.addIndicator("Overflow", "Tank.OverFlow", "ST")                        #S8c
    ofIndicator.setRestitutions(Pyc.TIndicatorType.mean_values)                                 #S8c

    doIndicator = system.addIndicator("DryOut", "Tank.DryOut", "ST")                            #S8c
    doIndicator.setRestitutions(Pyc.TIndicatorType.mean_values)                                 #S8c


    htIndicator = system.addIndicator("HotTemp", "Tank.HotTemperature", "ST")                   #S8c
    htIndicator.setRestitutions(Pyc.TIndicatorType.mean_values)                                 #S8c



    system.setSeqFilter(mySequenceFilter)                                                       #S8d
    system.setKeepFilteredSeqForInd(True)                                                       #S8d

    tb = time.time()                                                                            #S1
    system.simulate()                                                                           #S1

    if system.MPIRank() > 0 :
        exit(0)

    te=time .time()                                                                             #S1
    sTime = te-tb                                                                               #S1
    print ("Simulation time ", sTime )                                                          #S1

    meanOVF = ofIndicator.means()                                                               #S8c
    meanDO  = doIndicator.means()                                                               #S8c
    meanHT  = htIndicator.means()                                                               #S8c

    tps = system.instants()                                                                     #S8c


    plot(tps, True, True,                                                                      #S5 ==> S8a
            (                                                                                  #S1
                (("Tank"), "Time", "OverFlow", ((meanOVF, 'r-', 'Tank OverFlow'),)),           # S8a
                (("Tank"), "Time", "DryOut"  , ((meanDO , 'g-', 'Tank DryOut'  ),)),           # S8a
                (("Tank"), "Time", "Burnt"   , ((meanHT , 'b-', 'Tank Burnt'   ),)),           # S8a
            )                                                                                  #S1
         )                                                                                     #S1


    # analyzer = Pyc.CAnalyser(system)                                                           #S8d
    # if not analyzer.printFilteredSeq(10., "S8eHTBaseSeq.xml", "PycSeq.xsl"):                   #S8d
    #     Pyc.ILogManager.glLogManager().printMessages()                                         #S8d


