# -*- coding: iso-8859-15 -*-
"""
Copyright EDF 2018
@author: Hassane CHRAIBI - hassane.chraibi@edf.fr
"""

from PyCaUtils import *


######################################################################################
class Source(Pyc.CComponent):

    def __init__(self, name):
        Pyc.CComponent.__init__(self, name)

        # Parameters, Variables and References ----------------------------------------------
        self.p_nominalFlow     = self.addVariable("nominalFlow", Pyc.TVarType.t_double, 1)
        self.p_lambda          = self.addVariable("lambda"     , Pyc.TVarType.t_double, 1.e-3)
        self.p_mu              = self.addVariable("mu"         , Pyc.TVarType.t_double, 1.e-2)

        self.v_flow            = self.addVariable("flow", Pyc.TVarType.t_double, 0)
        self.r_reservoirVolume = self.addReference("reservoirVolume")

        self.r_maxVolume       = self.addReference("maxVolume")


        # Automata  ----------------------------------------------------------------------
        self.a_automaton  = self.addAutomaton("Automaton")
        self.s_open       = self.addState("Automaton", "OPEN" , 1)
        self.s_close      = self.addState("Automaton", "CLOSE", 2)
        self.s_stuck      = self.addState("Automaton", "STUCK", 3)

        self.setInitState("CLOSE")

        self.a_automaton.addCallback("updateFlow", self.updateFlow)


        close2Open = self.s_close.addTransition("CLOSE_to_OPEN")
        close2Open.setCondition(lambda : self.r_reservoirVolume.dValue(0) < self.r_maxVolume.dValue(0))
        close2Open.addTarget(self.s_open, Pyc.TTransType.trans)

        open2Close = self.s_open.addTransition("OPEN_to_CLOSE")
        open2Close.setCondition(lambda : self.r_reservoirVolume.dValue(0) >= self.r_maxVolume.dValue(0))
        open2Close.addTarget(self.s_close, Pyc.TTransType.trans)

        open2Stuck = self.s_open.addTransition("OPEN_to_STUCK")
        open2Stuck.setDistLaw(Pyc.TLawType.expo, self.p_lambda)
        open2Stuck.addTarget(self.s_stuck, Pyc.TTransType.fault)

        stuck2Close = self.s_stuck.addTransition("STUCK_to_CLOSE")
        stuck2Close.setDistLaw(Pyc.TLawType.expo, self.p_mu)
        stuck2Close.addTarget(self.s_close, Pyc.TTransType.fault)



        # Message boxes --------------------------------------------------------------
        self.addMessageBox = self.addMessageBox("MB-toReservoir")
        self.addMessageBoxExport("MB-toReservoir", self.v_flow           , "flow"     )
        self.addMessageBoxImport("MB-toReservoir", self.r_reservoirVolume, "volume"   )
        self.addMessageBoxImport("MB-toReservoir", self.r_maxVolume      , "maxVolume")


        # The PDMP Manager -----------------------------------------------------------
        self.addPDMPManager("PDMPManager")
        self.addPDMPWatchedTransition("PDMPManager", close2Open)
        self.addPDMPWatchedTransition("PDMPManager", open2Close)

    def updateFlow(self):
        if self.s_open.isActive():
            self.v_flow.setDValue(self.p_nominalFlow.dValue())
        elif self.s_close.isActive():
            self.v_flow.setDValue(0)

######################################################################################
class Reservoir(Pyc.CComponent):
    def __init__(self, name):
        Pyc.CComponent.__init__(self, name)

        # Parameters, Variables and References ----------------------------------------------
        self.p_maxVolume = self.addVariable("maxVolume", Pyc.TVarType.t_double, 200)

        self.v_volume    = self.addVariable("volume"   , Pyc.TVarType.t_double, 0)

        self.r_inFlow    = self.addReference("inFlow")

        # Message boxes --------------------------------------------------------------
        self.addMessageBox("MB-toSource")
        self.addMessageBoxImport("MB-toSource", self.r_inFlow   , "flow"     )
        self.addMessageBoxExport("MB-toSource", self.v_volume   , "volume"   )
        self.addMessageBoxExport("MB-toSource", self.p_maxVolume, "maxVolume")


        # The PDMP Manager -----------------------------------------------------------
        self.addPDMPManager("PDMPManager")
        self.addPDMPODEVariable   ("PDMPManager",self.v_volume)
        self.addPDMPEquationMethod("PDMPManager","pdmpEquation", self.pdmpEquation)



    def pdmpEquation(self):
        self.v_volume.setDvdtODE(self.r_inFlow.sumValue(0))


######################################################################################
class MySystem(Pyc.CSystem):
    def __init__(self, name):
        Pyc.CSystem.__init__(self, name)

        self.aSource    = Source   ("aSource"   )
        self.bSource    = Source   ("bSource"   )

        self.aReservoir = Reservoir("aReservoir")

        self.connect("aSource", "MB-toReservoir", "aReservoir", "MB-toSource")
        self.connect("bSource", "MB-toReservoir", "aReservoir", "MB-toSource")



######################################################################################
if __name__ == '__main__':

    print ("PyCATSHOO Version : ", Pyc.ILogManager.glLogManager().version())


    system = MySystem("ModelExample")

    system.loadParameters("ModelExample.xml")


    system.simulate()

    analyser = Pyc.CAnalyser(system)


    vectorOfInstants = stepsVector(system.tMax(), 500)

    # Calculates the evolution over time of the mean of the room temperature :
    meanVolume = analyser.meanValues("aReservoir.volume",  vectorOfInstants)

    # This vector will hold the cumulative probability distribution
    # of the Indicator volume > 500
    overFlowOccurrences = analyser.realized("aReservoir.volume",
                                            vectorOfInstants,
                                            lambda v: v > 500)


    # indicators:  evolution over time
    plot(vectorOfInstants, True, False,
           (
             (("Simulation of %d Sequences") % system.nbSequences(),
              "Time",
              "Volume",
              ((meanVolume, 'r', "Reservoir volume"),)),
           )
         )
    # indicators:  cumulative probability of occurrence
    plot(vectorOfInstants, True, True,
           (
             (("Simulation of %d Sequences") % system.nbSequences(),
              "Time",
              "Probability",
              ((overFlowOccurrences, 'r', "Over Flow CPDF"),)),
           )
         )

    print (meanVolume[-1])
